#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
replace_json_strings.py
複数の JSON ファイル内の文字列を一括置換するスクリプトです。

主な機能:
- 指定フォルダ配下の .json ファイルを走査（デフォルトは再帰）
- JSON を安全に読み込み、あらゆるネスト（辞書・配列・文字列）に含まれる文字列を置換
- （オプション）キー名に対する置換にも対応（衝突検出付き）
- 置換件数の集計、ドライラン、バックアップ作成、JSON 解析エラー時のテキスト置換フォールバック

使い方の例:
    python replace_json_strings.py /path/to/folder --old OnevADMIN --new OnevASSISTANT
    python replace_json_strings.py . --dry-run
    python replace_json_strings.py ./data --replace-keys --raw-text-fallback

注意:
- --replace-keys を付けるとキー名にも置換を行いますが、置換後にキーが重複しうるため、
  デフォルトでは衝突検出時に該当ファイルの処理をスキップします。
  強制的に上書きしたい場合は --allow-key-collisions を付けてください。
"""

from __future__ import annotations

import argparse
import json
import shutil
import sys
from datetime import datetime
from pathlib import Path
from typing import Any, Dict, List, Tuple


def iter_json_files(base_dir: Path, recursive: bool) -> List[Path]:
    """指定ディレクトリ配下の .json ファイル一覧を取得する。"""
    if recursive:
        return sorted([p for p in base_dir.rglob("*.json") if p.is_file()])
    return sorted([p for p in base_dir.glob("*.json") if p.is_file()])


def replace_in_data(
    obj: Any,
    old: str,
    new: str,
    replace_keys: bool,
    allow_key_collisions: bool,
) -> Tuple[Any, int]:
    """
    JSON データ（Python オブジェクト）内の文字列を置換し、置換回数を返す。

    Parameters
    ----------
    obj : Any
        対象オブジェクト（dict, list, str, int など）
    old : str
        置換前の文字列
    new : str
        置換後の文字列
    replace_keys : bool
        True の場合、辞書のキー名にも置換を適用する
    allow_key_collisions : bool
        True の場合、キー名置換後の衝突を許容する（後勝ち上書き）。
        False の場合、衝突が起きたら例外を送出する。

    Returns
    -------
    Tuple[Any, int]
        (置換後オブジェクト, 置換件数の合計)
    """
    total_replacements = 0

    if isinstance(obj, str):
        count = obj.count(old)
        if count > 0:
            obj = obj.replace(old, new)
            total_replacements += count
        return obj, total_replacements

    if isinstance(obj, list):
        new_list: List[Any] = []
        for item in obj:
            replaced_item, cnt = replace_in_data(
                item, old, new, replace_keys, allow_key_collisions
            )
            new_list.append(replaced_item)
            total_replacements += cnt
        return new_list, total_replacements

    if isinstance(obj, dict):
        new_dict: Dict[Any, Any] = {}
        for key, value in obj.items():
            # キー名の置換（必要なら）
            if replace_keys and isinstance(key, str):
                key_repl_count = key.count(old)
                new_key = key.replace(old, new) if key_repl_count > 0 else key
                total_replacements += key_repl_count
            else:
                new_key = key

            # 値の置換（再帰）
            replaced_value, value_repl_count = replace_in_data(
                value, old, new, replace_keys, allow_key_collisions
            )
            total_replacements += value_repl_count

            # 衝突検出
            if not allow_key_collisions and new_key in new_dict and new_key != key:
                raise KeyError(
                    f"キー衝突を検出しました: '{key}' -> '{new_key}' （既に存在するキーと重複）"
                )
            new_dict[new_key] = replaced_value
        return new_dict, total_replacements

    # 数値・bool・None などはそのまま
    return obj, total_replacements


def backup_file(path: Path) -> Path:
    """対象ファイルのバックアップを .bak.TIMESTAMP 付きで作成し、そのパスを返す。"""
    timestamp = datetime.now().strftime("%Y%m%d-%H%M%S")
    backup_path = path.with_suffix(path.suffix + f".bak.{timestamp}")
    shutil.copy2(path, backup_path)
    return backup_path


def process_json_file(
    path: Path,
    old: str,
    new: str,
    make_backup: bool,
    dry_run: bool,
    replace_keys: bool,
    allow_key_collisions: bool,
    raw_text_fallback: bool,
    indent: int,
) -> Tuple[bool, int, str]:
    """
    1 ファイルを処理するヘルパー。

    Returns
    -------
    Tuple[bool, int, str]
        (変更があったか, 置換件数, 情報/エラーメッセージ)
    """
    try:
        text = path.read_text(encoding="utf-8")
    except Exception as e:
        return False, 0, f"[ERROR] 読み込み失敗: {path} -> {e}"

    # まず JSON としてパースを試みる
    try:
        data = json.loads(text)
    except json.JSONDecodeError as e:
        if not raw_text_fallback:
            return (
                False,
                0,
                f"[ERROR] JSON パース失敗（--raw-text-fallback 未指定のためスキップ）: {path} -> {e}",
            )
        # フォールバック: 純テキスト置換
        count = text.count(old)
        if count == 0:
            return False, 0, f"[SKIP] 置換対象なし（テキスト）: {path}"
        if dry_run:
            return True, count, f"[DRY-RUN] テキスト置換のみ {count} 件: {path}"
        if make_backup:
            backup_path = backup_file(path)
            msg_backup = f"（バックアップ作成: {backup_path.name}）"
        else:
            msg_backup = "（バックアップなし）"
        new_text = text.replace(old, new)
        path.write_text(new_text, encoding="utf-8", newline="\n")
        return True, count, f"[OK] テキスト置換 {count} 件: {path} {msg_backup}"

    # JSON オブジェクトに対して安全に置換
    try:
        replaced, count = replace_in_data(
            data, old, new, replace_keys, allow_key_collisions
        )
    except KeyError as e:
        return False, 0, f"[ERROR] キー衝突により中止: {path} -> {e}"

    if count == 0:
        return False, 0, f"[SKIP] 置換対象なし（JSON）: {path}"

    if dry_run:
        return True, count, f"[DRY-RUN] JSON 置換 {count} 件: {path}"

    if make_backup:
        backup_path = backup_file(path)
        msg_backup = f"（バックアップ作成: {backup_path.name}）"
    else:
        msg_backup = "（バックアップなし）"

    try:
        # 上書き保存（整形して出力）
        with path.open("w", encoding="utf-8", newline="\n") as f:
            json.dump(replaced, f, ensure_ascii=False, indent=indent)
            f.write("\n")
    except Exception as e:
        return False, 0, f"[ERROR] 書き込み失敗: {path} -> {e}"

    return True, count, f"[OK] JSON 置換 {count} 件: {path} {msg_backup}"


def parse_args(argv: List[str]) -> argparse.Namespace:
    parser = argparse.ArgumentParser(
        description="指定フォルダ配下の JSON ファイル内の文字列を一括置換します。"
    )
    parser.add_argument(
        "folder",
        type=str,
        help="探索するフォルダのパス（例: ./data や C:\\\\work\\\\json など）",
    )
    parser.add_argument(
        "--old",
        type=str,
        default="OnevADMIN",
        help="置換前の文字列（デフォルト: OnevADMIN）",
    )
    parser.add_argument(
        "--new",
        type=str,
        default="OnevASSISTANT",
        help="置換後の文字列（デフォルト: OnevASSISTANT）",
    )
    parser.add_argument(
        "--non-recursive",
        action="store_true",
        help="再帰せず、直下の .json のみ処理します（デフォルトは再帰有効）。",
    )
    parser.add_argument(
        "--no-backup",
        action="store_true",
        help="バックアップ .bak.TIMESTAMP を作成しません（デフォルトは作成）。",
    )
    parser.add_argument(
        "--dry-run",
        action="store_true",
        help="書き込みを行わず、置換件数と対象を表示します。",
    )
    parser.add_argument(
        "--replace-keys",
        action="store_true",
        help="辞書のキー名にも置換を適用します（キー衝突に注意）。",
    )
    parser.add_argument(
        "--allow-key-collisions",
        action="store_true",
        help="キー名置換後の衝突を許容します（後勝ちで上書き）。",
    )
    parser.add_argument(
        "--raw-text-fallback",
        action="store_true",
        help="JSON パースに失敗したファイルは純テキスト置換で処理します。",
    )
    parser.add_argument(
        "--indent",
        type=int,
        default=2,
        help="JSON 出力時のインデント幅（デフォルト: 2）。",
    )
    return parser.parse_args(argv)


def main(argv: List[str]) -> int:
    args = parse_args(argv)

    base_dir = Path(args.folder).expanduser().resolve()
    if not base_dir.exists() or not base_dir.is_dir():
        print(f"[FATAL] フォルダが見つかりません: {base_dir}", file=sys.stderr)
        return 2

    recursive = not args.non_recursive
    make_backup = not args.no_backup

    json_files = iter_json_files(base_dir, recursive)
    if not json_files:
        print(f"[INFO] 対象となる .json ファイルが見つかりませんでした: {base_dir}")
        return 0

    print(
        f"[INFO] 探索開始: {base_dir} / 対象ファイル数: {len(json_files)} / "
        f"再帰: {'ON' if recursive else 'OFF'} / バックアップ: {'ON' if make_backup else 'OFF'} / "
        f"ドライラン: {'ON' if args.dry_run else 'OFF'} / キー置換: {'ON' if args.replace_keys else 'OFF'} / "
        f"衝突許容: {'ON' if args.allow_key_collisions else 'OFF'} / テキストフォールバック: {'ON' if args.raw_text_fallback else 'OFF'}"
    )

    total_changed_files = 0
    total_replacements = 0
    errors: List[str] = []

    for path in json_files:
        changed, count, message = process_json_file(
            path=path,
            old=args.old,
            new=args.new,
            make_backup=make_backup,
            dry_run=args.dry_run,
            replace_keys=args.replace_keys,
            allow_key_collisions=args.allow_key_collisions,
            raw_text_fallback=args.raw_text_fallback,
            indent=args.indent,
        )
        print(message)
        if "[ERROR]" in message:
            errors.append(message)
        if changed:
            total_changed_files += 1
            total_replacements += count

    print(
        f"[SUMMARY] 変更ファイル数: {total_changed_files} / 置換合計件数: {total_replacements} / エラー数: {len(errors)}"
    )

    # エラーがあれば終了コード 1 にする
    return 1 if errors else 0


if __name__ == "__main__":
    exit_code = main(sys.argv[1:])
    sys.exit(exit_code)
