/*:
 * @target MZ
 * @plugindesc ピクチャ表示プリセットを名前で管理して呼び出すプラグイン v1.1
 * @author 
 *
 * @param Presets
 * @text ピクチャプリセット一覧
 * @type struct<Preset>[]
 * @default []
 * @desc 登録しておくピクチャのプリセット
 *
 * @command ShowPreset
 * @text プリセット呼び出し
 * @desc プリセット名を指定してピクチャを表示します
 *
 * @arg pictureId
 * @text ピクチャID
 * @type number
 * @default 1
 * @desc 表示に使用するピクチャID
 *
 * @arg presetName
 * @text プリセット名
 * @type string
 * @default
 * @desc 呼び出すプリセットの名前
 */

/*~struct~Preset:
 * @param presetName
 * @text プリセット名
 * @desc このプリセットの識別名（プラグインコマンドから呼び出すときに使用）
 *
 * @param name
 * @text 画像ファイル名
 * @type file
 * @dir img/pictures
 *
 * @param origin
 * @text 原点
 * @type select
 * @option 左上
 * @value 0
 * @option 中心
 * @value 1
 * @default 0
 *
 * @param x
 * @text X座標
 * @type number
 * @default 0
 *
 * @param y
 * @text Y座標
 * @type number
 * @default 0
 *
 * @param scaleX
 * @text 拡大率X(%)
 * @type number
 * @default 100
 *
 * @param scaleY
 * @text 拡大率Y(%)
 * @type number
 * @default 100
 *
 * @param opacity
 * @text 不透明度
 * @type number
 * @max 255
 * @default 255
 */

(() => {
    const pluginName = "PicturePreset";

    const parameters = PluginManager.parameters(pluginName);
    const presets = {};
    try {
        const rawPresets = JSON.parse(parameters["Presets"] || "[]");
        for (const str of rawPresets) {
            const p = JSON.parse(str);
            if (p.presetName) {
                                presets[p.presetName] = {
                    name: p.name || "",
                    origin: Number(p.origin || 0),
                    x: Number(p.x || 0),
                    y: Number(p.y || 0),
                    scaleX: Number(p.scaleX || 100),
                    scaleY: Number(p.scaleY || 100),
                    opacity: Number(p.opacity || 255),
                };
            }
        }
    } catch (e) {
        console.error(`[${pluginName}] パラメータ解析エラー:`, e);
    }

    PluginManager.registerCommand(pluginName, "ShowPreset", args => {
        const pictureId = Number(args.pictureId || 1);
        const presetName = String(args.presetName || "");
        const preset = presets[presetName];
        if (!preset) {
            console.warn(`[${pluginName}] プリセット "${presetName}" が見つかりません`);
            return;
        }

        console.log(`[${pluginName}] プリセット "${presetName}" を表示:`, preset);

        $gameScreen.showPicture(
            pictureId,
            preset.name,
            preset.origin,
            preset.x,
            preset.y,
            preset.scaleX,
            preset.scaleY,
            preset.opacity,
            0
        );
    });
})();
