//=============================================================================
// WindowTitleCustom.js
// ----------------------------------------------------------------------------
// This software is released under the MIT License.
// http://opensource.org/licenses/mit-license.php
// ----------------------------------------------------------------------------
// Version
// 1.0.0 2024/12/22 初版
//=============================================================================

/*:
 * @plugindesc ウィンドウタイトルカスタマイズプラグイン
 * @target MZ
 * @author User
 *
 * @param CustomText
 * @text カスタム文章
 * @desc ゲームタイトルの右に追加表示する文章
 * @default 
 * @type string
 *
 * @param Separator
 * @text 区切り文字
 * @desc ゲームタイトルとカスタム文章の間に入れる区切り文字
 * @default  - 
 * @type string
 *
 * @param UseVariableText
 * @text 変数テキストを使用
 * @desc ONにすると、指定した変数の値をカスタム文章として使用します
 * @default false
 * @type boolean
 *
 * @param VariableId
 * @text 変数ID
 * @desc カスタム文章として使用する変数のID（変数テキストを使用がONの場合）
 * @default 1
 * @type variable
 *
 * @command setCustomText
 * @text カスタム文章を設定
 * @desc ウィンドウタイトルに表示するカスタム文章を設定します
 *
 * @arg text
 * @text 文章
 * @desc 表示する文章
 * @type string
 *
 * @command clearCustomText
 * @text カスタム文章をクリア
 * @desc ウィンドウタイトルのカスタム文章をクリアします
 *
 * @command updateTitle
 * @text タイトルを更新
 * @desc ウィンドウタイトルを現在の設定で更新します
 *
 * @help
 * ゲームウィンドウのタイトルバー（左上に表示されるタイトル）の右側に
 * 任意の文章を追加表示できるプラグインです。
 *
 * ============================================================================
 * 使用方法
 * ============================================================================
 * 
 * 1. プラグインパラメータで表示したい文章を設定
 * 2. ゲーム起動時に自動的にタイトルに反映されます
 *
 * ============================================================================
 * プラグインコマンド
 * ============================================================================
 * 
 * ◆ カスタム文章を設定
 *   ゲーム中にカスタム文章を変更できます。
 *   例: 現在のチャプター名を表示するなど
 *
 * ◆ カスタム文章をクリア
 *   カスタム文章をクリアして、ゲームタイトルのみの表示に戻します。
 *
 * ◆ タイトルを更新
 *   変数を使用している場合など、タイトルを最新の状態に更新します。
 *
 * ============================================================================
 * スクリプトから使用する場合
 * ============================================================================
 * 
 * // カスタム文章を設定
 * WindowTitleCustom.setCustomText("任意の文章");
 * 
 * // カスタム文章をクリア
 * WindowTitleCustom.clearCustomText();
 * 
 * // タイトルを更新
 * WindowTitleCustom.updateTitle();
 *
 */

(() => {
    'use strict';

    const pluginName = 'WindowTitleCustom';
    const parameters = PluginManager.parameters(pluginName);
    
    let paramCustomText = String(parameters['CustomText'] || '');
    const paramSeparator = String(parameters['Separator'] || ' - ');
    const paramUseVariableText = parameters['UseVariableText'] === 'true';
    const paramVariableId = Number(parameters['VariableId'] || 1);

    // 動的に変更可能なカスタム文章
    let dynamicCustomText = null;

    //=============================================================================
    // WindowTitleCustom - グローバルアクセス用
    //=============================================================================
    window.WindowTitleCustom = {
        setCustomText: function(text) {
            dynamicCustomText = text;
            this.updateTitle();
        },

        clearCustomText: function() {
            dynamicCustomText = null;
            this.updateTitle();
        },

        updateTitle: function() {
            if (!$dataSystem) return;
            
            let title = $dataSystem.gameTitle;
            let customText = this.getCustomText();
            
            if (customText && customText.length > 0) {
                title = title + paramSeparator + customText;
            }
            
            document.title = title;
        },

        getCustomText: function() {
            // 動的に設定されたテキストが優先
            if (dynamicCustomText !== null) {
                return dynamicCustomText;
            }
            
            // 変数テキストを使用する場合
            if (paramUseVariableText && $gameVariables) {
                const value = $gameVariables.value(paramVariableId);
                if (value) {
                    return String(value);
                }
            }
            
            // パラメータで設定されたテキスト
            return paramCustomText;
        }
    };

    //=============================================================================
    // プラグインコマンド
    //=============================================================================
    PluginManager.registerCommand(pluginName, 'setCustomText', args => {
        WindowTitleCustom.setCustomText(args.text);
    });

    PluginManager.registerCommand(pluginName, 'clearCustomText', () => {
        WindowTitleCustom.clearCustomText();
    });

    PluginManager.registerCommand(pluginName, 'updateTitle', () => {
        WindowTitleCustom.updateTitle();
    });

    //=============================================================================
    // Scene_Boot - タイトル更新を上書き
    //=============================================================================
    const _Scene_Boot_updateDocumentTitle = Scene_Boot.prototype.updateDocumentTitle;
    Scene_Boot.prototype.updateDocumentTitle = function() {
        _Scene_Boot_updateDocumentTitle.call(this);
        WindowTitleCustom.updateTitle();
    };

    //=============================================================================
    // Scene_Title - タイトル画面でも更新
    //=============================================================================
    const _Scene_Title_start = Scene_Title.prototype.start;
    Scene_Title.prototype.start = function() {
        _Scene_Title_start.call(this);
        WindowTitleCustom.updateTitle();
    };

    //=============================================================================
    // Scene_Map - マップ切り替え時にも更新（変数使用時用）
    //=============================================================================
    const _Scene_Map_start = Scene_Map.prototype.start;
    Scene_Map.prototype.start = function() {
        _Scene_Map_start.call(this);
        if (paramUseVariableText) {
            WindowTitleCustom.updateTitle();
        }
    };

})();
