/*:
 * @target MZ
 * @plugindesc ピクチャにモーション（うなずく/揺れる/走る等）を付与するプラグイン v2.1 (MZ対応版) 
 * @author hykw
 *
 * @command PlayMotion
 * @text モーション実行
 * @desc 指定したピクチャにモーションを実行する
 *
 * @arg motionName
 * @text モーション名
 * @type select
 * @option うなずく
 * @option 二回うなずく
 * @option 否定
 * @option ジャンプ
 * @option 揺れる
 * @option ゆれる
 * @option 笑う
 * @option 震える
 * @option 怒る
 * @option 驚く
 * @option ゆっくりうなずく
 * @option 左から出現
 * @option 右から出現
 * @option 走る左
 * @option 走る右
 *
 * @arg pictureId
 * @text ピクチャID
 * @type number
 * @default 1
 *
 * @arg duration
 * @text 継続フレーム数
 * @type number
 * @default 24
 *
 * @arg targetX
 * @text targetX
 * @type number
 * @default 0
 * @desc 「出現」などで移動先のX座標を指定したいときに使用
 *
 * @arg loop
 * @text ループ
 * @type boolean
 * @on ON
 * @off OFF
 * @default false
 *
 * @command StopMotion
 * @text モーション停止
 * @desc 指定したピクチャのモーションを停止する
 *
 * @arg pictureId
 * @text ピクチャID
 * @type number
 * @default 1
 *
 * @help
 * 指定したピクチャに「うなずく」「揺れる」「走る」などのモーションを適用できます。
 *
 * 使用例：
 *   プラグインコマンド → モーション実行 → 「うなずく」「ピクチャID=3」
 *   プラグインコマンド → モーション停止 → 「ピクチャID=3」
 *
 * 旧MV形式の「スクリプト」コマンドによる
 *   モーション実行 うなずく 3
 *   モーション停止 3
 * も引き続き使用可能です。
 */

(() => {
    const pluginName = "pictureLLmotion";

    const motionStateMap = {};

    function easeOutQuad(t) { return t * (2 - t); }
    function easeOutBack(t) {
        const c1 = 1.70158, c3 = c1 + 1;
        return 1 + c3 * Math.pow(t - 1, 3) + c1 * Math.pow(t - 1, 2);
    }
    function SPRITE_CONTAINER() {
        return SceneManager._scene._spriteset._pictureContainer;
    }

    function registerMotion(pictureId, motionType, duration = 24, params = {}) {
        const sprite = SPRITE_CONTAINER().children.find(sp => sp._pictureId === pictureId);
        if (!sprite) return;
        let startX = sprite.x, startY = sprite.y;
        const targetX = params.targetX ?? sprite.x;
        const targetY = params.targetY ?? sprite.y;
        motionStateMap[pictureId] = {
            type: motionType, frame: 0, duration,
            startX, startY, targetX, targetY,
            loop: params.loop || false, ...params
        };
    }

    function updateMotion(sprite, id) {
        const state = motionStateMap[id];
        if (!state) return;
        state.frame++;
        const t = state.frame / state.duration;
        const eased = easeOutQuad(t);

        switch (state.type) {
            case "yes": sprite.y = state.startY + Math.sin(t * Math.PI) * 16; break;
            case "yesyes": sprite.y = state.startY + Math.sin(t * Math.PI * 4) * 10; break;
            case "no": sprite.x = state.startX + Math.sin(t * Math.PI * 4) * 8; break;
            case "shake": sprite.x = state.startX + Math.sin(state.frame * 0.5) * 4; break;
            case "jump": sprite.y = state.startY + Math.sin(t * Math.PI) * -20; break;
            case "runleft":
                sprite.x = state.startX - eased * (Graphics.width + 300);
                if (t >= 1) $gameScreen.erasePicture(state.pictureId);
                break;
            case "runright":
                sprite.x = state.startX + eased * (Graphics.width + 300);
                if (t >= 1) $gameScreen.erasePicture(state.pictureId);
                break;
            case "hover": sprite.y = state.startY + Math.sin(t * Math.PI * 2) * 8; break;
            case "laugh":
                sprite.y = state.startY + Math.sin(t * Math.PI * 3) * 4;
                sprite.x = state.startX + Math.sin(t * Math.PI * 6) * 1.5;
                break;
            case "tremble": sprite.x = state.startX + Math.sin(state.frame * 0.7) * 3; break;
            case "stomp": sprite.y = state.startY + Math.abs(Math.sin(state.frame * 0.5)) * 10; break;
            case "surprise": sprite.y = state.startY - Math.sin(t * Math.PI) * 16; break;
            case "nodslow": sprite.y = state.startY + Math.sin(t * Math.PI) * 16; break;
            case "slideinright":
            case "slideinleft":
                sprite.x = state.startX + (state.targetX - state.startX) * easeOutBack(t);
                break;
        }

        if (state.frame >= state.duration) {
            if (state.loop) { state.frame = 0; return; }
            const pid = id;
    const picture = $gameScreen.picture(pid);
    if (picture) {
        picture._x = sprite.x;
        picture._y = sprite.y;
    }
            delete motionStateMap[id];
        }
    }

    const _Sprite_Picture_update = Sprite_Picture.prototype.update;
    Sprite_Picture.prototype.update = function() {
        _Sprite_Picture_update.call(this);
        if (this._pictureId != null) updateMotion(this, this._pictureId);
    };

    // --- MZ形式のプラグインコマンド登録 ---
    PluginManager.registerCommand(pluginName, "PlayMotion", args => {
        const motionName = args.motionName;
        const pictureId = Number(args.pictureId);
        const duration = Number(args.duration || 24);
        const targetX = args.targetX ? Number(args.targetX) : undefined;
        const loop = args.loop === "ON";

        const motionMap = {
            "うなずく": "yes", "二回うなずく": "yesyes", "否定": "no",
            "ジャンプ": "jump", "揺れる": "shake", "走る左": "runleft", "走る右": "runright",
            "ゆれる": "hover", "笑う": "laugh", "震える": "tremble", "怒る": "stomp",
            "驚く": "surprise", "ゆっくりうなずく": "nodslow",
            "右から出現": "slideinright", "左から出現": "slideinleft"
        };
        const type = motionMap[motionName] || motionName;

        registerMotion(pictureId, type, duration, { pictureId, targetX, loop });
    });

    PluginManager.registerCommand(pluginName, "StopMotion", args => {
        const pictureId = Number(args.pictureId);
        delete motionStateMap[pictureId];
    });

    // --- MV互換: pluginCommandも残す ---
    const _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
    Game_Interpreter.prototype.pluginCommand = function(command, args) {
        _Game_Interpreter_pluginCommand.call(this, command, args);
        if (command === "モーション実行") {
            const motionName = args[0];
            const pictureId = Number(args[1]);
            const duration = Number(args[2] || 24);
            const targetX = args[3] ? Number(args[3]) : undefined;
            const loop = args.includes("ループ");
            const motionMap = {
                "うなずく": "yes", "二回うなずく": "yesyes", "否定": "no",
                "ジャンプ": "jump", "揺れる": "shake", "走る左": "runleft", "走る右": "runright",
                "ゆれる": "hover", "笑う": "laugh", "震える": "tremble", "怒る": "stomp",
                "驚く": "surprise", "ゆっくりうなずく": "nodslow",
                "右から出現": "slideinright", "左から出現": "slideinleft"
            };
            const type = motionMap[motionName] || motionName;
            registerMotion(pictureId, type, duration, { pictureId, targetX, loop });
        }
        if (command === "モーション停止") {
            const pictureId = Number(args[0]);
            delete motionStateMap[pictureId];
        }
    };
})();
