//=============================================================================
// AtsumaruPaymentMV.js
//
// Copyright (c) 2018-2020 RPGアツマール開発チーム(https://game.nicovideo.jp/atsumaru)
// Released under the MIT license
// http://opensource.org/licenses/mit-license.php
//=============================================================================

(function () {
    'use strict';

    function isNumber(value) {
        return value !== "" && !isNaN(value);
    }
    function isInteger(value) {
        return typeof value === "number" && isFinite(value) && Math.floor(value) === value;
    }
    function isNatural(value) {
        return isInteger(value) && value > 0;
    }
    function isValidVariableId(variableId) {
        return isNatural(variableId) && variableId < $dataSystem.variables.length;
    }

    // 既存のクラスとメソッド名を取り、そのメソッドに処理を追加する
    function hook(baseClass, target, f) {
        baseClass.prototype[target] = f(baseClass.prototype[target]);
    }
    function hookStatic(baseClass, target, f) {
        baseClass[target] = f(baseClass[target]);
    }
    // プラグインコマンドを追加する
    function addPluginCommand(commands) {
        hook(Game_Interpreter, "pluginCommand", function (origin) { return function (command, args) {
            origin.apply(this, arguments);
            if (commands[command]) {
                commands[command].apply(this, [command].concat(args));
            }
        }; });
    }
    // Promiseが終了するまでイベントコマンドをウェイトするための処理を追加する
    function prepareBindPromise() {
        if (Game_Interpreter.prototype.bindPromiseForRPGAtsumaruPlugin) {
            return;
        }
        // Promiseを実行しつつ、それをツクールのインタプリタと結びつけて解決されるまで進行を止める
        Game_Interpreter.prototype.bindPromiseForRPGAtsumaruPlugin = function (promise, resolve, reject) {
            var _this = this;
            this._index--;
            this._promiseResolverForRPGAtsumaruPlugin = function () { return false; };
            promise.then(function (value) { return _this._promiseResolverForRPGAtsumaruPlugin = function () {
                _this._index++;
                delete _this._promiseResolverForRPGAtsumaruPlugin;
                if (resolve) {
                    resolve(value);
                }
                return true;
            }; }, function (error) { return _this._promiseResolverForRPGAtsumaruPlugin = function () {
                for (var key in _this._eventInfo) {
                    error[key] = _this._eventInfo[key];
                }
                error.line = _this._index + 1;
                error.eventCommand = "plugin_command";
                error.content = _this._params[0];
                switch (error.code) {
                    case "BAD_REQUEST":
                        throw error;
                    case "UNAUTHORIZED":
                    case "FORBIDDEN":
                    case "INTERNAL_SERVER_ERROR":
                    case "API_CALL_LIMIT_EXCEEDED":
                    default:
                        console.error(error.code + ": " + error.message);
                        console.error(error.stack);
                        if (Graphics._showErrorDetail && Graphics._formatEventInfo && Graphics._formatEventCommandInfo) {
                            var eventInfo = Graphics._formatEventInfo(error);
                            var eventCommandInfo = Graphics._formatEventCommandInfo(error);
                            console.error(eventCommandInfo ? eventInfo + ", " + eventCommandInfo : eventInfo);
                        }
                        _this._index++;
                        delete _this._promiseResolverForRPGAtsumaruPlugin;
                        if (reject) {
                            reject(error);
                        }
                        return true;
                }
            }; });
        };
        // 通信待機中はこのコマンドで足踏みし、通信に成功または失敗した時にPromiseの続きを解決する
        // このタイミングまで遅延することで、以下のようなメリットが生まれる
        // １．解決が次のコマンドの直前なので、他の並列処理に結果を上書きされない
        // ２．ゲームループ内でエラーが発生するので、エラー発生箇所とスタックトレースが自然に詳細化される
        // ３．ソフトリセット後、リセット前のexecuteCommandは叩かれなくなるので、
        //     リセット前のPromiseのresolverがリセット後のグローバルオブジェクトを荒らす事故がなくなる
        hook(Game_Interpreter, "executeCommand", function (origin) { return function () {
            if (this._promiseResolverForRPGAtsumaruPlugin) {
                var resolved = this._promiseResolverForRPGAtsumaruPlugin();
                if (!resolved) {
                    return false;
                }
            }
            return origin.apply(this, arguments);
        }; });
    }

    function toNaturalOrUndefined(value, command, name) {
        if (value === undefined) {
            return value;
        }
        var number = +value;
        if (isNumber(value) && isNatural(number)) {
            return number;
        }
        else {
            throw new Error("「" + command + "」コマンドでは、" + name + "を指定する場合は自然数を指定してください。" + name + ": " + value);
        }
    }
    function toTypedParameters(parameters, isArray) {
        if (isArray === void 0) { isArray = false; }
        var result = isArray ? [] : {};
        for (var key in parameters) {
            try {
                var value = JSON.parse(parameters[key]);
                result[key] = value instanceof Array ? toTypedParameters(value, true)
                    : value instanceof Object ? toTypedParameters(value)
                        : value;
            }
            catch (error) {
                result[key] = parameters[key];
            }
        }
        return result;
    }
    function ensureValidVariableIds(parameters) {
        hookStatic(DataManager, "isDatabaseLoaded", function (origin) { return function () {
            if (!origin.apply(this, arguments)) {
                return false;
            }
            for (var key in parameters) {
                var variableId = parameters[key];
                if (variableId !== 0 && !isValidVariableId(variableId)) {
                    throw new Error("プラグインパラメータ「" + key + "」には、0～" + ($dataSystem.variables.length - 1) + "までの整数を指定してください。" + key + ": " + variableId);
                }
            }
            return true;
        }; });
    }

    /*:
     * @plugindesc RPGアツマールの販売アイテムのためのプラグインです
     * @author RPGアツマール開発チーム
     *
     * @param purchaseResult
     * @type variable
     * @text 販売アイテム購入結果
     * @desc 「販売アイテム購入画面表示」コマンドの結果を代入する変数の番号を指定します。(購入=1,キャンセル=2)
     * @default 0
     *
     * @param purchaseResultItemCode
     * @type variable
     * @text 販売アイテム購入結果（販売アイテムコード）
     * @desc 「販売アイテム購入画面表示」コマンドで販売アイテムが購入された場合に販売アイテムコードを代入する変数の番号を指定します。
     * @default 0
     *
     * @param itemCountList
     * @type struct<itemCount>[]
     * @text 販売アイテムリスト
     * @desc 「販売アイテム所持数取得」コマンドで取得したいアイテムのコードと、所持数の代入先変数の組を指定します。
     * @default []
     *
     * @param errorMessage
     * @type variable
     * @text エラーメッセージ
     * @desc エラーが発生した場合に、エラーメッセージを代入する変数の番号を指定します。
     * @default 0
     *
     * @help
     * このプラグインは、アツマールAPIの「販売アイテム」を利用するためのプラグインです。
     * 詳しくはアツマールAPIリファレンス(https://beta.game.nicovideo.jp/atsumaru-api-reference/payment)を参照してください。
     *
     * プラグインコマンド（英語版と日本語版のコマンドがありますが、どちらも同じ動作です）:
     *   DisplayPurchaseModal <itemCode>
     *   販売アイテム購入画面表示 <itemCode>
     *      # 指定した<itemCode>のアイテムの購入を促す画面を表示します。
     *      # ユーザーが画面を閉じると「販売アイテム購入結果」で指定した変数に
     *        その販売アイテムを購入した場合は1、キャンセルした場合は2が代入されます。
     *      # さらに「販売アイテム購入結果（販売アイテムコード）」には、その時購入した販売アイテムの
     *        販売アイテムコードが代入されます。ランダム販売の結果を知りたいときにはこちらを参照してください。
     *        （パック販売の場合は、いずれかの販売アイテムコードが代表で代入されます）
     *
     *   DisplayCatalogModal <catalogCode>
     *   販売カタログ画面表示 <catalogCode>
     *      # 指定した<catalogCode>のカタログ（複数の販売アイテムのリスト）を表示します。
     *      # ユーザーが画面を閉じると「販売アイテム購入結果」で指定した変数に
     *        何か販売アイテムを購入した場合は1、キャンセルした場合は2が代入されます。
     *      # さらに「販売アイテム購入結果（販売アイテムコード）」には、その時購入した販売アイテムの
     *        販売アイテムコードが代入されます。どのアイテムを購入したか知りたいときにはこちらを参照してください。
     *        （パック販売の場合は、いずれかの販売アイテムコードが代表で代入されます）
     *
     *   GetPaymentItemCounts
     *   販売アイテム所持数取得
     *      # プラグインパラメータ「販売アイテムリスト」で指定したアイテムを、
     *        このプレイヤーがいくつ持っているかがリスト上の対応する変数に代入されます。
     *      # 「販売アイテムリスト」で指定した販売アイテムコードがそのゲームに存在しない場合は、対応する変数には何も代入されません。
     *
     *   GetPaymentItemCounts <gameId>
     *   販売アイテム所持数取得 <gameId>
     *      # 指定した<gameId>のゲームにおいて、
     *        プラグインパラメータ「販売アイテムリスト」で指定したアイテムを、
     *        このプレイヤーがいくつ持っているかがリスト上の対応する変数に代入されます。
     *      # 「販売アイテムリスト」で指定した販売アイテムコードがそのゲームに存在しない場合は、対応する変数には何も代入されません。
     *
     * アツマール外（テストプレイや他のサイト、ダウンロード版）での挙動:
     *      DisplayPurchaseModal（販売アイテム購入画面表示）
     *          無視される（エラーメッセージにも何も代入されない）
     *      DisplayCatalogModal（販売カタログ画面表示）
     *          無視される（エラーメッセージにも何も代入されない）
     *      GetPaymentItemCounts（販売アイテム所持数取得）
     *          無視される（エラーメッセージにも何も代入されない）
     */
    var parameters = toTypedParameters(PluginManager.parameters("AtsumaruPaymentMV"));
    var variableMap = {};
    parameters.itemCountList.forEach(function (item) { return variableMap[item.itemCode] = { itemCount: item.itemCount || 0, expiredAt: item.expiredAt || 0, remainingSeconds: item.remainingSeconds || 0 }; });
    var itemCodes = Object.keys(variableMap);
    var payment = window.RPGAtsumaru && window.RPGAtsumaru.payment;
    var getPaymentItems = payment && payment.getPaymentItems;
    var displayPurchaseModal = payment && payment.displayPurchaseModal;
    var displayCatalogModal = payment && payment.displayCatalogModal;
    {
        var purchaseResult = parameters.purchaseResult, errorMessage = parameters.errorMessage;
        ensureValidVariableIds({ purchaseResult: purchaseResult, errorMessage: errorMessage });
    }
    for (var itemCode in variableMap) {
        var _a = variableMap[itemCode], itemCount = _a.itemCount, expiredAt = _a.expiredAt, remainingSeconds = _a.remainingSeconds;
        ensureValidVariableIds({ itemCount: itemCount, expiredAt: expiredAt, remainingSeconds: remainingSeconds });
    }
    prepareBindPromise();
    addPluginCommand({
        DisplayPurchaseModal: DisplayPurchaseModal,
        "販売アイテム購入画面表示": DisplayPurchaseModal,
        "課金アイテム購入画面表示": DisplayPurchaseModal,
        DisplayCatalogModal: DisplayCatalogModal,
        "販売カタログ画面表示": DisplayCatalogModal,
        GetPaymentItemCounts: GetPaymentItemCounts,
        "販売アイテム所持数取得": GetPaymentItemCounts,
        "課金アイテム所持数取得": GetPaymentItemCounts // 互換性のために残す
    });
    function DisplayPurchaseModal(command, itemCode) {
        if (displayPurchaseModal) {
            this.bindPromiseForRPGAtsumaruPlugin(displayPurchaseModal(itemCode), function (result) {
                $gameVariables.setValue(parameters.purchaseResultItemCode, 0);
                if (result.state === "purchase_success") {
                    $gameVariables.setValue(parameters.purchaseResult, 1);
                    $gameVariables.setValue(parameters.purchaseResultItemCode, result.purchaseResultItems[0].itemCode);
                }
                else if (result.state === "purchase_canceled") {
                    $gameVariables.setValue(parameters.purchaseResult, 2);
                }
                $gameVariables.setValue(parameters.errorMessage, 0);
            }, function (error) { return $gameVariables.setValue(parameters.errorMessage, error.message); });
        }
    }
    function DisplayCatalogModal(command, catalogCode) {
        if (displayCatalogModal) {
            this.bindPromiseForRPGAtsumaruPlugin(displayCatalogModal(catalogCode), function (result) {
                $gameVariables.setValue(parameters.purchaseResultItemCode, 0);
                if (result.state === "purchase_success") {
                    $gameVariables.setValue(parameters.purchaseResult, 1);
                    $gameVariables.setValue(parameters.purchaseResultItemCode, result.purchaseResultItems[0].itemCode);
                }
                else if (result.state === "purchase_canceled") {
                    $gameVariables.setValue(parameters.purchaseResult, 2);
                }
                $gameVariables.setValue(parameters.errorMessage, 0);
            }, function (error) { return $gameVariables.setValue(parameters.errorMessage, error.message); });
        }
    }
    function GetPaymentItemCounts(command, gameIdStr) {
        var gameId = toNaturalOrUndefined(gameIdStr, command, "gameId");
        if (getPaymentItems) {
            this.bindPromiseForRPGAtsumaruPlugin(getPaymentItems(itemCodes, gameId), function (items) {
                $gameVariables.setValue(parameters.errorMessage, 0);
                for (var itemCode in variableMap) {
                    if (itemCode in items) {
                        $gameVariables.setValue(variableMap[itemCode].itemCount, items[itemCode].itemCount);
                        var expiredAt = items[itemCode].expiredAt;
                        var remainingSeconds = items[itemCode].remainingSeconds;
                        if (expiredAt !== null)
                            $gameVariables.setValue(variableMap[itemCode].expiredAt, new Date(expiredAt * 1000).toLocaleString("ja-JP"));
                        if (remainingSeconds !== null)
                            $gameVariables.setValue(variableMap[itemCode].remainingSeconds, items[itemCode].remainingSeconds);
                    }
                }
            }, function (error) { return $gameVariables.setValue(parameters.errorMessage, error.message); });
        }
    }
    /*~struct~itemCount:
     * @param itemCode
     * @text アイテムコード
     * @desc アイテムコードを指定します。
     *
     * @param itemCount
     * @type variable
     * @text アイテム所持数
     * @desc アイテム所持数を代入する変数の番号を指定します。
     *
     * @param expiredAt
     * @type variable
     * @text アイテム期限
     * @desc アイテムの期限を代入する変数の番号を指定します。値は日付の文字列で代入されます。期限付きアイテムの場合のみ代入されます。
     *
     * @param remainingSeconds
     * @type variable
     * @text アイテム残り時間
     * @desc アイテムの期限までの残り時間（秒）を代入する変数の番号を指定します。期限付きアイテムの場合のみ代入されます。
     */

}());
