# SFX抽象化ガイド

## 概要

SFX（擬音）定義を抽象化し、全モンスター共通のデフォルトSFXを使用するようにしました。
これにより、新しいモンスター追加時のコピペ作業が不要になり、保守性が大幅に向上しました。

## 変更内容

### Before（抽象化前）

```javascript
window.OverlaySets = {
  slime: {
    mapping: { /* ... */ },
    sfx: {
      insert_entry:  "default_pose_overlay_ズプッ1-6",
      insert_piston: "default_pose_overlay_ぬちょっ1-6",
      insert_deep:   "default_pose_overlay_ズポ1-6",
      rub_pussy:     "default_pose_overlay_ぬちゃっ1-6",
      rub_breast:    "default_pose_overlay_ぬちゃっ1-6",
      orgasm:        "default_pose_overlay_ぐちゃ1-6",
      liquid:        "default_pose_overlay_ぐちゃ1-6"
    }
  },
  goblin: {
    mapping: { /* ... */ },
    sfx: { /* 同じ7行をコピペ */ }
  },
  tentacle: {
    mapping: { /* ... */ },
    sfx: { /* 同じ7行をコピペ */ }
  }
};
```

**問題点**:
- 完全に同じ定義が3回重複（21行）
- 新モンスター追加時に毎回7行コピペが必要
- SFXを1つ変更する場合、全モンスター分修正が必要

---

### After（抽象化後）

```javascript
// デフォルトSFX定義（全モンスター共通）
const DEFAULT_SFX = {
  insert_entry:  "default_pose_overlay_ズプッ1-6",
  insert_piston: "default_pose_overlay_ぬちょっ1-6",
  insert_deep:   "default_pose_overlay_ズポ1-6",
  rub_pussy:     "default_pose_overlay_ぬちゃっ1-6",
  rub_breast:    "default_pose_overlay_ぬちゃっ1-6",
  orgasm:        "default_pose_overlay_ぐちゃ1-6",
  liquid:        "default_pose_overlay_ぐちゃ1-6"
};

window.OverlaySets = {
  slime: {
    mapping: { /* ... */ }
    // sfx: DEFAULT_SFXを使用（上書きする場合はここに記述）
  },
  goblin: {
    mapping: { /* ... */ }
    // sfx: DEFAULT_SFXを使用
  },
  tentacle: {
    mapping: { /* ... */ }
    // sfx: DEFAULT_SFXを使用
  }
};

// グローバルに公開
window.DEFAULT_SFX = DEFAULT_SFX;
```

**改善点**:
- ✅ 重複コードを削減（21行 → 7行、67%削減）
- ✅ 新モンスター追加時にSFX定義不要（0行）
- ✅ SFX変更時は1箇所だけ修正すればOK
- ✅ 特殊な音が必要な場合のみ個別定義

---

## 使い方

### 1. 通常のモンスター（デフォルトSFX使用）

```javascript
window.OverlaySets = {
  newMonster: {
    mapping: {
      restrain: "restrain",
      attack: { /* ... */ },
      insert: { /* ... */ }
    }
    // sfxは省略 = 自動的にDEFAULT_SFXを使用
  }
};
```

**必要な作業**:
1. 画像フォルダ `img/pictures/busts/newMonster_restrained/` を作成
2. デフォルトSFX画像をコピー:
   - `default_pose_overlay_ズプッ1-6_left.png`
   - `default_pose_overlay_ズプッ1-6_right.png`
   - `default_pose_overlay_ぬちょっ1-6_left.png`
   - `default_pose_overlay_ぬちょっ1-6_right.png`
   - `default_pose_overlay_ズポ1-6_left.png`
   - `default_pose_overlay_ズポ1-6_right.png`
   - `default_pose_overlay_ぬちゃっ1-6_left.png`
   - `default_pose_overlay_ぬちゃっ1-6_right.png`
   - `default_pose_overlay_ぐちゃ1-6_left.png`
   - `default_pose_overlay_ぐちゃ1-6_right.png`
3. コード変更不要！

---

### 2. 特殊な音が必要なモンスター（一部上書き）

```javascript
window.OverlaySets = {
  demon: {
    mapping: { /* ... */ },
    sfx: {
      // デフォルトを継承しつつ、一部だけ上書き
      insert_entry: "default_pose_overlay_ドスッ1-6",  // 悪魔専用
      orgasm: "default_pose_overlay_ドロッ1-6"         // 悪魔専用
      // 他のSFX（insert_piston, rub_pussy等）はDEFAULT_SFXを使用
    }
  }
};
```

**動作**:
- `insert_entry` → `"default_pose_overlay_ドスッ1-6"` （上書き）
- `insert_piston` → `"default_pose_overlay_ぬちょっ1-6"` （デフォルト）
- `insert_deep` → `"default_pose_overlay_ズポ1-6"` （デフォルト）
- `rub_pussy` → `"default_pose_overlay_ぬちゃっ1-6"` （デフォルト）
- `rub_breast` → `"default_pose_overlay_ぬちゃっ1-6"` （デフォルト）
- `orgasm` → `"default_pose_overlay_ドロッ1-6"` （上書き）
- `liquid` → `"default_pose_overlay_ぐちゃ1-6"` （デフォルト）

---

### 3. 完全に独自のSFXを使うモンスター

```javascript
window.OverlaySets = {
  alien: {
    mapping: { /* ... */ },
    sfx: {
      // 全て独自定義
      insert_entry:  "default_pose_overlay_ビュルッ1-6",
      insert_piston: "default_pose_overlay_グチュ1-6",
      insert_deep:   "default_pose_overlay_ズブッ1-6",
      rub_pussy:     "default_pose_overlay_ヌルッ1-6",
      rub_breast:    "default_pose_overlay_ヌルッ1-6",
      orgasm:        "default_pose_overlay_ドピュ1-6",
      liquid:        "default_pose_overlay_ドピュ1-6"
    }
  }
};
```

---

## 内部動作

### OverlayBuilder.js の処理

```javascript
function getBubbleOverlaysFromContext(context, intensity) {
  const { enemyKey, eventType, targetPart, isInserted, phase } = context;
  
  // 1. 敵固有のSFXを取得
  const enemySfx = window.OverlaySets?.[enemyKey]?.sfx;
  
  // 2. デフォルトSFXを取得
  const defaultSfx = window.DEFAULT_SFX;
  
  // 3. マージ（敵固有 > デフォルト の優先順位）
  const sfxMap = enemySfx ? { ...defaultSfx, ...enemySfx } : defaultSfx;
  
  // 4. 状況に応じてSFXを選択
  // ...
}
```

**マージの仕組み**:
```javascript
// 例: demonの場合
defaultSfx = {
  insert_entry: "ズプッ",
  insert_piston: "ぬちょっ",
  orgasm: "ぐちゃ"
};

enemySfx = {
  insert_entry: "ドスッ",  // 上書き
  orgasm: "ドロッ"         // 上書き
};

// マージ結果
sfxMap = {
  insert_entry: "ドスッ",    // 敵固有が優先
  insert_piston: "ぬちょっ", // デフォルト
  orgasm: "ドロッ"          // 敵固有が優先
};
```

---

## デバッグ

### コンソールで確認

```javascript
// デフォルトSFXを確認
console.log(window.DEFAULT_SFX);
// → { insert_entry: "...", insert_piston: "...", ... }

// 特定の敵のSFXを確認
console.log(window.OverlaySets.slime.sfx);
// → undefined（デフォルトを使用）

console.log(window.OverlaySets.demon.sfx);
// → { insert_entry: "ドスッ", orgasm: "ドロッ" }（一部上書き）
```

### ログ出力

```
[VisualOverlayData] DEFAULT_SFX defined: { insert_entry: "...", ... }
[OverlayBuilder-SFX] Selected: type="insert_entry", key="default_pose_overlay_ズプッ1-6", enemy="slime", ...
```

---

## メリット

### 1. コード削減
- **Before**: 21行（7行 × 3モンスター）
- **After**: 7行（デフォルト定義のみ）
- **削減率**: 67%

### 2. 新モンスター追加コスト
- **Before**: 7行コピペ
- **After**: 0行（画像をコピーするだけ）

### 3. SFX変更コスト
- **Before**: 全モンスター分修正（最悪10箇所）
- **After**: 1箇所（DEFAULT_SFX）

### 4. 拡張性
- デフォルトで動作
- 特殊な音が必要な場合のみ個別定義
- 一部だけ上書きも可能

---

## 画像ファイル配置

### 標準的な配置（デフォルトSFX使用）

```
img/pictures/busts/
  ├── slime_restrained/
  │   ├── default_pose_overlay_ズプッ1-6_left.png
  │   ├── default_pose_overlay_ズプッ1-6_right.png
  │   ├── default_pose_overlay_ぬちょっ1-6_left.png
  │   ├── default_pose_overlay_ぬちょっ1-6_right.png
  │   ├── default_pose_overlay_ズポ1-6_left.png
  │   ├── default_pose_overlay_ズポ1-6_right.png
  │   ├── default_pose_overlay_ぬちゃっ1-6_left.png
  │   ├── default_pose_overlay_ぬちゃっ1-6_right.png
  │   ├── default_pose_overlay_ぐちゃ1-6_left.png
  │   └── default_pose_overlay_ぐちゃ1-6_right.png
  ├── goblin_restrained/
  │   └── （同じファイル名）
  └── tentacle_restrained/
      └── （同じファイル名）
```

**重要**: ファイル名は全モンスター共通ですが、各フォルダに配置することで、モンスターごとに異なる画像を使用できます。

---

## まとめ

- ✅ **コード削減**: 67%削減（21行 → 7行）
- ✅ **保守性向上**: 修正箇所が1箇所に集約
- ✅ **拡張性向上**: 新モンスター追加が0行
- ✅ **柔軟性**: 特殊な音が必要な場合のみ個別定義
- ✅ **後方互換性**: 既存のコードは変更なしで動作

抽象化により、システムがよりシンプルで保守しやすくなりました！

