# VisualAutoFill システムアーキテクチャ

## 概要

VisualAutoFillシステムは、表情とオーバーレイを自動的に決定するシステムです。
バージョン2.0で、保守性と拡張性を向上させるため、3つのモジュールに分割されました。

## アーキテクチャ

```
┌─────────────────────────────────────────────────────┐
│         VisualAutoFill.js (統合層)                   │
│  - autoFillDialogueSequence()                       │
│  - autoFillSingleVisual()                           │
│  - 後方互換性API                                      │
└──────────────┬──────────────────┬───────────────────┘
               │                  │
       ┌───────┴────────┐  ┌─────┴──────────┐
       │                │  │                 │
┌──────▼──────────┐  ┌──▼──────────────┐  │
│ ExpressionSelector│  │ OverlayBuilder  │  │
│ (表情選択)        │  │ (オーバーレイ)   │  │
└──────────────────┘  └─────────────────┘  │
                                            │
                                  ┌─────────▼──────────┐
                                  │ VisualOverlayData  │
                                  │ (データ定義)        │
                                  └────────────────────┘
```

## モジュール構成

### 1. ExpressionSelector.js (表情選択専門)

**責務**: 状況に応じた表情の自動選択

**主要機能**:
- 重み付きランダム選択
- 台詞解析による表情決定
- 表情遷移ルール
- 堕落度・イベントタイプによる補正

**公開API**:
```javascript
window.ExpressionSelector = {
  // データ
  EXPRESSION_WEIGHTS,
  EXPRESSION_TRANSITIONS,
  
  // 関数
  weightedRandomExpression(stage, eventType, dialogueText, isRestrained),
  getNextExpression(currentExpression, stage, eventType),
  
  // 内部関数（デバッグ用）
  _internal: {
    weightedRandom,
    analyzeDialogueEmotion,
    getExpressionWeightsAdvanced
  }
};
```

**ファイルサイズ**: 約300行

---

### 2. OverlayBuilder.js (オーバーレイ構築専門)

**責務**: オーバーレイとSFXの自動構築

**主要機能**:
- 攻撃エフェクト選択
- 頬染め選択
- SFX（擬音）選択
- レイヤー配置決定
- シーケンスパターン対応

**公開API**:
```javascript
window.OverlayBuilder = {
  // データ
  BLUSH_BY_STAGE,
  
  // 関数
  buildOverlaysForLine(context, progress, frameIndex, totalFrames),
  getAttackOverlays(enemyKey, targetPart, isInserted, intensity),
  getBlushOverlays(stage, progress),
  getBubbleOverlaysFromContext(context, intensity),
  
  // 内部関数（デバッグ用）
  _internal: {
    resolveAttackOverlay,
    getAttackOverlayForFrame
  }
};
```

**ファイルサイズ**: 約350行

---

### 3. VisualAutoFill.js (統合層)

**責務**: ExpressionSelector と OverlayBuilder を統合し、使いやすいAPIを提供

**主要機能**:
- dialogueSequence 全体の自動補完
- 単一ビジュアルノードの自動補完
- 後方互換性の維持

**公開API**:
```javascript
window.VisualAutoFill = {
  // 統合API
  autoFillDialogueSequence(sequence, context),
  autoFillSingleVisual(context, progress, dialogueText),
  
  // 再エクスポート（後方互換性）
  weightedRandomExpression,
  getNextExpression,
  buildOverlaysForLine,
  getAttackOverlays,
  getBubbleOverlaysFromContext,
  
  // データ
  BLUSH_BY_STAGE,
  EXPRESSION_WEIGHTS,
  EXPRESSION_TRANSITIONS
};
```

**ファイルサイズ**: 約150行

---

### 4. VisualOverlayData.js (データ定義)

**責務**: 敵ごとのオーバーレイマッピングとSFX定義

**データ構造**:
```javascript
window.OverlaySets = {
  slime: {
    mapping: { /* オーバーレイマッピング */ },
    sfx: { /* SFXマッピング */ }
  },
  goblin: { /* ... */ },
  tentacle: { /* ... */ }
};
```

---

## プラグイン読み込み順序

**重要**: 以下の順序で読み込む必要があります：

```javascript
// js/plugins.js
[
  // ...
  {"name":"VisualOverlayData","status":true},        // 1. データ定義
  {"name":"ExpressionSelector","status":true},       // 2. 表情選択
  {"name":"OverlayBuilder","status":true},           // 3. オーバーレイ構築
  {"name":"VisualAutoFill","status":true},           // 4. 統合層
  // ...
]
```

---

## 使用例

### 例1: dialogueSequence の自動補完

```javascript
const sequence = [
  { text: "やめて...！" },
  { text: "んっ...！" },
  { text: "あっ...♡" }
];

const context = {
  stage: "stage2",
  eventType: "danger",
  targetPart: "breast",
  enemyKey: "slime",
  isInserted: false,
  isRestrained: true
};

const filled = window.VisualAutoFill.autoFillDialogueSequence(sequence, context);
// → 各行に expression, overlays, underOverlays が自動補完される
```

### 例2: 単一ビジュアルの自動補完

```javascript
const context = {
  stage: "stage3",
  eventType: "climax",
  targetPart: "pussy",
  enemyKey: "goblin",
  isInserted: true,
  isRestrained: true
};

const visual = window.VisualAutoFill.autoFillSingleVisual(context, 0.8);
// → { expression: "ahe", overlays: [...], underOverlays: [...] }
```

### 例3: 表情のみ選択

```javascript
const expression = window.ExpressionSelector.weightedRandomExpression(
  "stage2",      // 堕落度
  "danger",      // イベントタイプ
  "やめて...！",  // 台詞
  true           // 拘束状態
);
// → "endure" または "confused" など
```

### 例4: オーバーレイのみ構築

```javascript
const context = {
  stage: "stage2",
  targetPart: "breast",
  enemyKey: "slime",
  isInserted: false,
  isRestrained: true
};

const overlays = window.OverlayBuilder.buildOverlaysForLine(context, 0.5);
// → { overlays: ["restrain", "blush_mid", "slime_breast_low"], underOverlays: [] }
```

---

## 後方互換性

既存のコードは変更なしで動作します：

```javascript
// これらは全て動作する
window.VisualAutoFill.weightedRandomExpression(stage, eventType);
window.VisualAutoFill.buildOverlaysForLine(context, progress);
window.VisualAutoFill.getBubbleOverlaysFromContext(context, intensity);
window.VisualAutoFill.EXPRESSION_WEIGHTS;
```

---

## 拡張方法

### 新しい表情を追加

1. `ExpressionSelector.js` の `EXPRESSION_WEIGHTS` を編集
2. 必要に応じて `EXPRESSION_TRANSITIONS` を編集

### 新しい敵を追加

1. `VisualOverlayData.js` に新しい敵のマッピングを追加
2. 画像ファイルを配置
3. コード変更不要（自動的に認識される）

### 新しいSFXを追加

1. `VisualOverlayData.js` の `sfx` マッピングに追加
2. `OverlayBuilder.js` の `getBubbleOverlaysFromContext` に判定ロジックを追加（必要に応じて）

---

## デバッグ

### Console ログ

各モジュールは詳細なログを出力します：

```
[ExpressionSelector] Selected: "confused" (weights: {...})
[OverlayBuilder-Layer] enemyKey=slime, targetPart=breast, ...
[OverlayBuilder-SFX] Selected: type="rub_breast", key="...", ...
[VisualAutoFill] Ready - Integration layer initialized
```

### 内部関数へのアクセス

デバッグ・テスト用に内部関数も公開されています：

```javascript
// 台詞解析をテスト
const hint = window.ExpressionSelector._internal.analyzeDialogueEmotion("やめて...！");
console.log(hint); // → { resist: 40, confused: 0, pleasure: 0 }

// オーバーレイ解決をテスト
const result = window.OverlayBuilder._internal.resolveAttackOverlay(
  "slime", "attack", "breast", "soft"
);
console.log(result); // → { overlays: ["slime_breast_low"], layer: "overlay" }
```

---

## パフォーマンス

### ファイルサイズ比較

| バージョン | ファイル数 | 総行数 | 備考 |
|-----------|----------|-------|------|
| v1.0 (統合) | 1ファイル | 760行 | 全てが1ファイル |
| v2.0 (分割) | 3ファイル | 800行 | モジュール化 |

**増加**: 約40行（コメント・ヘッダー追加のため）

### 実行時パフォーマンス

- **影響なし**: 関数呼び出しのオーバーヘッドは無視できるレベル
- **メモリ**: ほぼ同じ（グローバル変数が3つに分散されるのみ）

---

## トラブルシューティング

### エラー: ExpressionSelector not loaded

**原因**: プラグインの読み込み順序が間違っている

**解決策**: `js/plugins.js` で `ExpressionSelector` が `VisualAutoFill` より前にあることを確認

### エラー: OverlayBuilder not loaded

**原因**: プラグインの読み込み順序が間違っている

**解決策**: `js/plugins.js` で `OverlayBuilder` が `VisualAutoFill` より前にあることを確認

### 表情が自動補完されない

**原因**: `context.isRestrained` が設定されていない

**解決策**: コンテキストに `isRestrained` フラグを追加

---

## まとめ

### メリット

1. **保守性向上**: 各モジュールが独立しており、修正が容易
2. **可読性向上**: ファイルが小さく、見通しが良い
3. **拡張性向上**: 新機能の追加が容易
4. **テスト性向上**: 各モジュールを独立してテスト可能
5. **再利用性**: 表情システムだけ他のプロジェクトに移植可能

### 後方互換性

既存のコードは変更なしで動作します。

### 推奨される使い方

- **新規コード**: 各モジュールを直接使用（`ExpressionSelector.xxx`, `OverlayBuilder.xxx`）
- **既存コード**: そのまま（`VisualAutoFill.xxx` で動作）

