// Improved BattleBustManager.js (status/bubble channels, top bubble layer)
(() => {
  // Force update check
  console.log("%c[BattleBustManager] PLUGIN UPDATED: " + new Date().toISOString(), "color: red; font-size: 20px; font-weight: bold;");
  console.log("[BattleBustManager] plugin loaded");

  //  ImageManager.isReady()をオーバーライドしてエラーを無視（クラッシュ防止）
  const _ImageManager_isReady = ImageManager.isReady;
  ImageManager.isReady = function() {
    try {
      for (const cache of [this._cache, this._system]) {
        for (const url in cache) {
          const bitmap = cache[url];
          // エラー状態のbitmapは無視（例外を投げない）
          if (bitmap.isError()) {
            console.warn(`[ImageManager] Image load error ignored: ${bitmap.url}`);
            continue;
          }
          if (!bitmap.isReady()) {
            return false;
          }
        }
      }
      return true;
    } catch (e) {
      console.error("[ImageManager] Error in isReady(), returning false:", e);
      return false;
    }
  };

  // 安全な画像読み込み関数：存在しない画像は dummy.png にフォールバック
  ImageManager.loadPictureSafe = function(filename) {
    const bitmap = ImageManager.loadPicture(filename);
    
    // エラーハンドリングを追加
    const original_onError = bitmap._onError;
    bitmap._onError = function() {
      console.warn(`[loadPictureSafe] Image load failed: ${filename} → fallback to dummy.png`);
      
      // ダミー画像をロード
      try {
        const dummy = ImageManager.loadPicture("dummy");
        if (dummy && dummy._image) {
          bitmap._image = dummy._image;
          bitmap._loadingState = "loaded";
          bitmap._createBaseTexture(dummy._image);
          bitmap._callLoadListeners();
        } else {
          // ダミー画像もない場合は空のbitmapを作成
          bitmap._loadingState = "loaded";
          bitmap._callLoadListeners();
        }
      } catch (e) {
        console.error(`[loadPictureSafe] Fallback failed for ${filename}:`, e);
        bitmap._loadingState = "loaded";
        bitmap._callLoadListeners();
      }
      
      // 元の_onErrorは呼ばない（エラーを処理したため）
    };
    
    bitmap.addLoadListener(() => {
      if (bitmap.isReady()) {
        console.log(`[loadPictureSafe] Loaded: ${filename}`);
      } else if (bitmap.isError()) {
        // エラーは_onErrorで処理済み
      }
    });
    return bitmap;
  };
  
  //  _nohighlight専用フォールバック：先に通常版をロードし、_nohighlight版が存在すれば上書き
  // 逆フォールバック方式：確実に画像を表示してから、より適切な画像があれば差し替え
  ImageManager.loadPictureSafeWithFallback = function(filename, fallbackFilename) {
    console.log(`[loadPictureSafeWithFallback] Primary: ${filename}, Fallback: ${fallbackFilename}`);
    
    //  先にフォールバック（通常版）をロードして返す
    const bitmap = ImageManager.loadPicture(fallbackFilename);
    console.log(`[loadPictureSafeWithFallback] Loading fallback first: ${fallbackFilename}`);
    
    // バックグラウンドでメイン画像（_nohighlight版）の存在をチェック
    const mainBitmap = ImageManager.loadPicture(filename);
    
    mainBitmap.addLoadListener(() => {
      if (mainBitmap.isReady() && mainBitmap._image) {
        // _nohighlight版が存在した！上書きする
        console.log(`[loadPictureSafeWithFallback] ✓ Primary exists, upgrading: ${filename}`);
        bitmap._image = mainBitmap._image;
        bitmap._url = mainBitmap._url;
        if (mainBitmap._baseTexture) {
          bitmap._baseTexture = mainBitmap._baseTexture;
        }
        // 既に画面に表示されているspriteも更新される
      }
    });
    
    // メイン画像のエラーは無視（フォールバックで十分）
    mainBitmap._onError = function() {
      console.log(`[loadPictureSafeWithFallback] ✗ Primary not found (using fallback): ${filename}`);
    };
    
    return bitmap;
  };

  // ── 汎用アニメ（既存を尊重） ────────────────────────
  function fadeIn(sprite, duration = 20) {
    sprite.alpha = 0;
    const step = 1 / duration;
    const fadeTicker = () => {
      sprite.alpha += step;
      if (sprite.alpha < 1) requestAnimationFrame(fadeTicker);
      else sprite.alpha = 1;
    };
    requestAnimationFrame(fadeTicker);
  }

  function scalePop(sprite, duration = 15) {
    sprite.scale.set(2.0, 2.0);
    const step = (2.0 - 1.0) / duration;
    let count = 0;
    const tick = () => {
      count++;
      const scale = 2.0 - step * count;
      sprite.scale.set(scale, scale);
      if (count < duration) requestAnimationFrame(tick);
      else sprite.scale.set(1.0, 1.0);
    };
    requestAnimationFrame(tick);
  }

  // ① 状態を比較してアニメーションタイプを判断（既存ロジック）
  function determineAnimationType(actor, displayInfo) {
    const currentStage = actor.getClothingStage();
    const actorId = actor.actorId();

    const manager = window.BattleBustManager;
    if (!manager._lastClothingStage) {
      manager._lastClothingStage = {};
    }

    const lastStage = manager._lastClothingStage[actorId];
    const isChanged = currentStage !== lastStage;
    manager._lastClothingStage[actorId] = currentStage;

    if (isChanged && (currentStage === "damaged" || currentStage === "destroyed")) {
      console.log(`[BBM] Clothing stage changed: ${lastStage} → ${currentStage}`);
      return "clothingDamage";  // 衣装破壊専用エフェクトを使用
    }
    return "fadeIn";
  }

  // ── レイヤ生成・描画のヘルパ ─────────────────────────
  //  fallbackFilename が指定されている場合はフォールバック機能を使用
  function createLayerSprite(filename, animationType = "fadeIn", opt = {}, fallbackFilename = null) {
    console.log(`[BBM-LOAD] Loading image from path: ${filename}`);
    
    // フォールバック指定がある場合は専用関数を使用
    const bitmap = fallbackFilename 
      ? ImageManager.loadPictureSafeWithFallback(filename, fallbackFilename)
      : ImageManager.loadPictureSafe(filename);
      
    bitmap.smooth = false;
    const sprite = new Sprite(bitmap);
    bitmap.addLoadListener(() => {
      // 既存の基準点に合わせる（レイヤごと anchor は揃える）
      sprite.anchor.set(opt.anchorX ?? 0.5, opt.anchorY ?? 1);
      sprite.x = (opt.x ?? 408);
      sprite.y = (opt.y ?? 32);
      sprite.z = (opt.z ?? 0);
      applyAnimation(sprite, animationType);
    });
    return sprite;
  }

  function applyAnimation(sprite, type) {
    switch (type) {
      case "clothingDamage":
        // 衣装破壊専用エフェクトを呼び出す（ClothingDamageEffect.js）
        // 案1：シャッター/ワイプ効果（布が縦に裂けて左右に開く）- 採用
        if (window.ClothingDamageEffect?.playTearTransition) {
          window.ClothingDamageEffect.playTearTransition(sprite);
        } else {
          // フォールバック：エフェクトファイルが読み込まれていない場合
          console.warn("[BBM] ClothingDamageEffect not loaded, using fadeIn");
          fadeIn(sprite);
        }
        break;
      case "scalePop":
        scalePop(sprite);
        break;
      case "fadeIn":
      default:
        fadeIn(sprite);
        break;
    }
  }

  /**
   * レイヤーの子要素を全削除（最適化版）
   * 後ろから削除することで配列の再インデックスを回避
   */
  function clearChildren(spriteLayer) {
    if (!spriteLayer || !spriteLayer.children) return;
    // 後ろから削除すれば配列の再インデックスが不要（O(n²) → O(n)）
    while (spriteLayer.children.length > 0) {
      spriteLayer.removeChild(spriteLayer.children[spriteLayer.children.length - 1]);
    }
  }

  /**
   * 配列の等価性チェック（差分更新用）
   */
  /**
   * 配列の内容が等しいかチェック（差分更新用）
   */
  function arraysEqual(a, b) {
    if (a === b) return true;
    if (!a || !b) return false;
    if (a.length !== b.length) return false;
    for (let i = 0; i < a.length; i++) {
      if (a[i] !== b[i]) return false;
    }
    return true;
  }

  // overlays から「状態系／バブル系」を分離（BBM側の保険：PDM未導入でもOK）
  function partitionOverlays(rawOverlays) {
    const result = { statusKeys: [], bubbleManualKeysRaw: [] };
    if (!Array.isArray(rawOverlays)) return result;
    for (const key of rawOverlays) {
      if (typeof key !== "string") continue;
      const isBubble =
        key.startsWith("bubble:") || 
        key.startsWith("bubbleR:") || 
        key.startsWith("bubbleL:") ||
        key.startsWith("default_pose_overlay_");  // 擬音キーもバブルとして扱う
      if (isBubble) result.bubbleManualKeysRaw.push(key);
      else result.statusKeys.push(key);
    }
    return result;
  }

  //  専用画像方式: 各敵フォルダに配置された画像をそのまま表示
  // 座標調整は画像自体に含まれているため、座標計算は不要

  // bubble キー → 画像名の決定（フルパスも可）
  function resolveBubblePictureName(profileName, pose, bubbleEntry) {
    if (typeof bubbleEntry === "string") {
      // 例： "overlays/bubbles/あっ_1-2_right" のようなフォルダ指定を許す
      if (bubbleEntry.includes("/")) return bubbleEntry;
      
      // すでに default_pose_overlay_ が付いている場合はそのまま使用
      if (bubbleEntry.startsWith("default_pose_overlay_")) {
        return `busts/${profileName}/${bubbleEntry}`;
      }
      
      // 付いていない場合はプレフィックスを追加
      return `busts/${profileName}/default_pose_overlay_${bubbleEntry}`;
    }
    if (bubbleEntry && typeof bubbleEntry === "object") {
      if (bubbleEntry.picture) return String(bubbleEntry.picture);
      if (bubbleEntry.key) {
        if (String(bubbleEntry.key).includes("/")) return String(bubbleEntry.key);
        return `busts/${profileName}/${bubbleEntry.key}`;
      }
    }
    return ""; // 不明時
  }
function _resolveNudge(folderName, pose, displayInfo, originalProfileName = null) {
  // 1) イベント個別（最優先）
  const byDisplay = displayInfo?.nudge?.group || null;

  // 2) プロファイル既定（任意: visuals.nudge）
  // standpose使用時(folderName="stand_pose")は元のprofileNameからnudgeを取得
  const lookupName = (folderName === "stand_pose" && originalProfileName) 
    ? originalProfileName 
    : folderName;
  
  const prof = window.PoseProfiles?.[lookupName];
  const byProfile =
    prof?.visuals?.nudge?.group ||
    prof?.visuals?.nudge?.byPose?.[pose]?.group ||
    null;

  // マージ（displayInfo が優先）
  const gx = (byProfile?.x ?? 0) + (byDisplay?.x ?? 0);
  const gy = (byProfile?.y ?? 0) + (byDisplay?.y ?? 0);
  return { x: gx, y: gy };
}
  class BattleBustManager {
    constructor() {
      this._sprites = {};                // actorId => SpriteGroup
      this._lastClothingStage = {};      // 既存
    }

    createSpriteGroup(actor) {
      const group = new Sprite();

      // レイヤ構造：base → face → overlay(状態系) → bubble(最前面)
      group._under   = new Sprite();
      group._base    = new Sprite();
      group._face    = new Sprite();
      group._overlay = new Sprite();
      group._bubble  = new Sprite(); //  追加

      group.addChild(group._under);
      group.addChild(group._base);
      group.addChild(group._face);
      group.addChild(group._overlay);
      group.addChild(group._bubble); //  最上位

      // 配置・基準（既存準拠）
      group.x = Graphics.width / 2;
      group.y = Graphics.height;
      group._base.anchor.set(0.5, 1);
      group._face.anchor.set(0.5, 1);
      group._overlay.anchor.set(0.5, 1);
      group._bubble.anchor.set(0.5, 1); //  追加
      group._under.anchor.set(0.5, 1);

      //  差分更新用の状態管理
      group._lastState = {
        profileName: null,
        pose: null,
        expression: null,
        expressionSuffix: null,  //  ハイライト状態も追跡（"" or "_nohighlight"）
        folderName: null,  // フォルダ名も追跡（tentacle_low → tentacle_restrained_low など）
        statusOverlays: [],
        underOverlays: [],
        bubbleOverlays: []
      };

      this._sprites[actor.actorId()] = group;
      return group;
    }


async show(actor, displayInfo = null) {
  const scene = SceneManager._scene;
  if (!scene || !(scene instanceof Scene_Battle)) return;
  //  追加：立ち絵をぶら下げる親コンテナ（通常は spriteset）
  const parentContainer = scene._spriteset || scene;
  if (!displayInfo) {
    displayInfo = await PleasurePoseController.getDisplayInfo(actor, "react", "pussy");
  }
  if (!displayInfo || !displayInfo.profileName) {
    const fallback = window.PleasurePoseController?.getProfileNameFromEnemy?.(actor);
    if (fallback) {
      displayInfo = Object.assign({ profileName: fallback }, displayInfo || {});
    } else {
      console.warn("[BBM] Invalid displayInfo passed to show()");
      return;
    }
  }

  // 既存の group 再利用
  let group = this._sprites[actor.actorId()];
  if (group && (!group.parent || group.parent !== parentContainer)) {
    console.warn(`[BBM] Discarding orphaned sprite group for actor=${actor.name()}`);
    group = null;
  }


  if (!group) {
    group = this.createSpriteGroup(actor);
    parentContainer.addChild(group);  //  spriteset 配下にぶら下げる
  }

  
  // 前回の状態を取得（差分更新用）
  // createSpriteGroup で既に初期化されているが、念のため確認
  if (!group._lastState) {
    group._lastState = {
      profileName: null,
      pose: null,
      expression: null,
      expressionSuffix: null,  //  ハイライト状態も追跡（"" or "_nohighlight"）
      folderName: null,
      statusOverlays: [],
      underOverlays: [],
      bubbleOverlays: []
    };
  }

  const profileName = displayInfo.profileName ?? "default";
  const pose = displayInfo.pose;
  let expression = String(displayInfo.expression);

  // 後方互換：旧 overlays を受け取り、status/bubble を分割
  const overlaysLegacy = Array.isArray(displayInfo.overlays) ? displayInfo.overlays.map(String) : [];
  const legacyPartition = partitionOverlays(overlaysLegacy);

  // 正式引数（優先）：statusOverlays / bubbleOverlays
  const statusOverlays =
    Array.isArray(displayInfo.statusOverlays)
      ? displayInfo.statusOverlays.map(String)
      : legacyPartition.statusKeys;

  // bubbleOverlays は [string | {picture,x,y,anchorX,anchorY,z}] を許可
  const bubbleOverlays =
    Array.isArray(displayInfo.bubbleOverlays)
      ? displayInfo.bubbleOverlays
      : legacyPartition.bubbleManualKeysRaw; // PDM 未導入でも最低限動かす

  //  パス解決ロジックの統合（フォルダ統廃合対応）
  let folderName = profileName;
  let useCommonStandPose = false;

  // "standpose_xxx" プロファイルは常に共通フォルダ "stand_pose" を使用
  if (profileName.startsWith("standpose_")) {
    useCommonStandPose = true;
    folderName = "stand_pose";
  }
  // "xxx_low" プロファイルで、かつ "uniform_xxx" ポーズの場合も共通フォルダを使う
  // ただし、拘束系(restrained)や挿入系(insert)のプロファイルは除外する
  else if (profileName.endsWith("_low") && 
      !profileName.includes("restrained") && 
      !profileName.includes("insert") && 
      pose.startsWith("uniform_")) {
    useCommonStandPose = true;
    folderName = "stand_pose";
  } else {
    // 拘束・挿入プロファイル等の場合、_low/_mid/_high のサフィックスを除去して統合フォルダを参照する
    // 例: "slime_restrained_low" -> "slime_restrained"
    //     "slime_restrained_insert_high" -> "slime_restrained_insert"
    folderName = profileName.replace(/_(low|mid|high)$/, "");
  }

  // フォルダ名解決完了

  // ベース画像のパス解決
  const basePath = `busts/${folderName}/${pose}`;
  const baseFile = `${basePath}_base`;

  // 表情画像のパス解決
  let faceFile = "";
  let faceFileFallback = "";  // フォールバック用（_nohighlightなし版）
  let overlayPrefix = "";
  
  // 表情名の正規化（confuse → confused への統一）
  // JSON側で "confuse" と "confused" が混在しているため、画像読み込み時に統一
  if (expression === "confuse") {
    expression = "confused";
    console.log("[BBM] Normalized expression: confuse → confused");
  }
  
  // ハイライト消失チェック
  const expressionSuffix = actor?.hasLostHighlight?.() ? "_nohighlight" : "";
  
  if (useCommonStandPose) {
    // 共通立ち絵の場合: default_pose_face_xxx を参照
    // （stand_poseフォルダには default_pose_face_normal.png 等がある前提）
    faceFile = `busts/stand_pose/default_pose_face_${expression}${expressionSuffix}`;
    faceFileFallback = `busts/stand_pose/default_pose_face_${expression}`;  // フォールバック
    
    // オーバーレイも共通フォルダを見る（必要であれば）
    overlayPrefix = `busts/stand_pose/default_pose_overlay_`;
    } else {
      // 従来通り：個別フォルダ
      faceFile = `busts/${folderName}/default_pose_face_${expression}${expressionSuffix}`;
      faceFileFallback = `busts/${folderName}/default_pose_face_${expression}`;  // フォールバック
      overlayPrefix = `busts/${folderName}/default_pose_overlay_`;
    }
  
  //  under 用も同じ規約で解決（配置だけ背面）
  const underOverlayPrefix = overlayPrefix;
  const animationType = determineAnimationType(actor, displayInfo);

  // ======== 服段階の最終判定（intact以外なら昇格モード） ========
  const rawDisplayStage = displayInfo && displayInfo.clothingStage;
  const rawActorStage = (typeof actor.getClothingStage === "function") ? actor.getClothingStage() : undefined;
  const clothingStage = String(rawDisplayStage ?? rawActorStage ?? "intact").toLowerCase();
  const elevateAllUnder = (clothingStage !== "intact");

  // clothingStage決定完了

  // 昇格から除外したい under 用の保険（床影・背景など想定。必要に応じて語を追加）
  const elevateExclusionRegex = /(?:^|_)(shadow|floor|reflection|bg|background)(?:_|$)/i;
  const isExcludedFromElevate = (key) => elevateExclusionRegex.test(String(key));

  // --- underOverlays（背面 or 自動昇格=overlay最背面） ---
  const underList = Array.isArray(displayInfo.underOverlays)
    ? displayInfo.underOverlays.map(String)
    : [];

  //  差分更新: underOverlays が変更された場合のみ更新（真の差分更新）
  const underChanged = !arraysEqual(group._lastState.underOverlays, underList);
  if (underChanged) {
    const oldSet = new Set(group._lastState.underOverlays || []);
    const newSet = new Set(underList);
    
    // 削除するもの（old にあって new にないもの）
    const toRemove = new Set([...oldSet].filter(key => !newSet.has(key)));
    
    // 追加するもの（new にあって old にないもの）
    const toAdd = [...newSet].filter(key => !oldSet.has(key));
    
    // 削除（変更されたものだけ）
    if (toRemove.size > 0) {
      // _under 層から削除
      const underChildren = group._under.children.slice();
      for (const child of underChildren) {
        if (child._overlayKey && toRemove.has(child._overlayKey)) {
          group._under.removeChild(child);
        }
      }
      
      // overlay 層から削除（elevateAllUnder の場合）
      if (elevateAllUnder) {
        const overlayChildren = group._overlay.children.slice();
        for (const child of overlayChildren) {
          if (child._isUnderOverlay && child._overlayKey && toRemove.has(child._overlayKey)) {
            group._overlay.removeChild(child);
          }
        }
      }
    }
    
    // 追加（新しいものだけ）
    for (const key of toAdd) {
      if (!key) continue;

      const safeKey = key.replace(":", "_");
      // フルパス対応（既存は prefix + key 前提だが保険）
      const spritePath = (safeKey.startsWith("busts/") || safeKey.startsWith("pictures/"))
        ? safeKey
        : (underOverlayPrefix + safeKey);

      const sprite = createLayerSprite(spritePath, animationType);
      sprite._overlayKey = key;  // キーを保存（削除時の識別用）

      if (elevateAllUnder && !isExcludedFromElevate(key)) {
        //  overlay層の最背面に差し込んで、頬染め/汗/拘束等より下に固定
        sprite._isUnderOverlay = true;  // マーカーを付ける
        group._overlay.addChildAt(sprite, 0);
      } else {
        group._under.addChild(sprite);
      }
    }
    group._lastState.underOverlays = [...underList];
    if (toRemove.size > 0 || toAdd.length > 0) {
      console.log(`[BBM-Diff] underOverlays: -${toRemove.size} +${toAdd.length}`);
    }
  }

  if (displayInfo.se) {
    AudioManager.playSe({ name: displayInfo.se, pan: 0, pitch: 100, volume: 90 });
  }

  //  差分更新: base（体）が変更された場合のみ更新
  // folderName も考慮する必要がある（tentacle_low → tentacle_restrained_low など）
  const baseChanged = (
    group._lastState.profileName !== profileName || 
    group._lastState.pose !== pose ||
    group._lastState.folderName !== folderName
  );
  if (baseChanged) {
    clearChildren(group._base);
    const baseSprite = createLayerSprite(baseFile, animationType);
    group._base.addChild(baseSprite);
    group._lastState.profileName = profileName;
    group._lastState.pose = pose;
    group._lastState.folderName = folderName;
    console.log("[BBM-Diff] base updated (profileName or pose or folderName changed)");
  } else {
    console.log("[BBM-Diff] base unchanged, skipping");
  }

  //  差分更新: face（表情）が変更された場合のみ更新
  // baseChanged の場合も face を更新（folderName が変わると faceFile のパスも変わるため）
  //  ハイライト消失状態（expressionSuffix）の変化も検知
  const faceChanged = (
    group._lastState.expression !== expression ||
    group._lastState.expressionSuffix !== expressionSuffix ||  //  ハイライト状態の変化も検知
    baseChanged  // base が変わった場合は face も更新
  );
  let faceSprite = null;
  if (faceChanged) {
    clearChildren(group._face);
    //  _nohighlight版がない場合は通常版にフォールバック
    if (expressionSuffix === "_nohighlight") {
      faceSprite = createLayerSprite(faceFile, animationType, {}, faceFileFallback);
    } else {
      faceSprite = createLayerSprite(faceFile, animationType);
    }
    group._face.addChild(faceSprite);
    group._lastState.expression = expression;
    group._lastState.expressionSuffix = expressionSuffix;  //  ハイライト状態も記録
    console.log(`[BBM-Diff] face updated (expression=${expression}, suffix=${expressionSuffix})`);
    
    // === デバッグ追記（face更新時のみ） ===
    // 表情更新完了
  } else {
    console.log(`[BBM-Diff] face unchanged (expression=${expression}, suffix=${expressionSuffix})`);
  }

  //  差分更新: statusOverlays が変更された場合のみ更新（真の差分更新）
  const overlaysChanged = !arraysEqual(group._lastState.statusOverlays, statusOverlays);
  if (overlaysChanged) {
    const oldSet = new Set(group._lastState.statusOverlays || []);
    const newSet = new Set(statusOverlays);
    
    // 削除するもの（old にあって new にないもの）
    const toRemove = new Set([...oldSet].filter(key => !newSet.has(key)));
    
    // 追加するもの（new にあって old にないもの）
    const toAdd = [...newSet].filter(key => !oldSet.has(key));
    
    // 削除（変更されたものだけ）
    if (toRemove.size > 0) {
      const overlayChildren = group._overlay.children.slice();
      for (const child of overlayChildren) {
        if (!child._isUnderOverlay && child._overlayKey && toRemove.has(child._overlayKey)) {
          group._overlay.removeChild(child);
        }
      }
    }
    
    // 追加（新しいものだけ）
    for (const overlayKey of toAdd) {
      if (!overlayKey) continue;
      const safeKey = String(overlayKey).replace(":", "_");
      const sprite = createLayerSprite(overlayPrefix + safeKey, animationType);
      sprite._overlayKey = overlayKey;  // キーを保存（削除時の識別用）
      group._overlay.addChild(sprite);
    }
    
    group._lastState.statusOverlays = [...statusOverlays];
    if (toRemove.size > 0 || toAdd.length > 0) {
      console.log(`[BBM-Diff] statusOverlays: -${toRemove.size} +${toAdd.length}`);
    }
  }

  //  差分更新: bubbleOverlays が変更された場合のみ更新（真の差分更新）
  const bubblesChanged = !arraysEqual(group._lastState.bubbleOverlays, bubbleOverlays);
  if (bubblesChanged) {
    const oldSet = new Set(group._lastState.bubbleOverlays || []);
    const newSet = new Set(bubbleOverlays);
    
    // 削除するもの（old にあって new にないもの）
    const toRemove = new Set([...oldSet].filter(key => !newSet.has(key)));
    
    // 追加するもの（new にあって old にないもの）
    const toAdd = [...newSet].filter(key => !oldSet.has(key));
    
    // 削除（変更されたものだけ）
    if (toRemove.size > 0) {
      const bubbleChildren = group._bubble.children.slice();
      for (const child of bubbleChildren) {
        if (child._overlayKey && toRemove.has(child._overlayKey)) {
          group._bubble.removeChild(child);
        }
      }
    }
    
    // 追加（新しいものだけ）
    //  バブル（最上面）
    // 専用画像方式: 各敵フォルダの画像をそのまま表示（座標は画像に含まれる）
    for (const entry of toAdd) {
      if (!entry) continue;

      // 文字列形式のみサポート（専用画像方式）
      if (typeof entry === "string") {
        const picName = resolveBubblePictureName(folderName, pose, entry);
        if (!picName) continue;
        
        // 画像をそのまま表示（座標調整なし）
        const sprite = createLayerSprite(picName, animationType, { 
          x: 408,  // デフォルト基準点
          y: 32,   // デフォルト基準点
          z: 9999,
          anchorX: 0.5,
          anchorY: 1
        });
        sprite._overlayKey = entry;  // キーを保存（削除時の識別用）
        group._bubble.addChild(sprite);
      }
    }
    group._lastState.bubbleOverlays = [...bubbleOverlays];
    if (toRemove.size > 0 || toAdd.length > 0) {
      console.log(`[BBM-Diff] bubbleOverlays: -${toRemove.size} +${toAdd.length}`);
    }
  }

  // 位置最終調整（既存準拠）
  requestAnimationFrame(() => {
    // standpose使用時はfolderName("stand_pose")とprofileName("xxx_low")の両方を渡す
    const nudge = _resolveNudge(folderName, pose, displayInfo, profileName); 
    group.x = 488 + (nudge.x || 0);
    group.y = 617 + (nudge.y || 0);
    group.visible = true;
  });
}



    clear(actor) {
      const group = this._sprites[actor.actorId()];
      if (group) {
        if (group.parent) group.parent.removeChild(group);
        delete this._sprites[actor.actorId()];
      }
    }

    clearAll() {
      for (const actorId in this._sprites) {
        const group = this._sprites[actorId];
        if (group && group.parent) {
          group.parent.removeChild(group);
        }
      }
      this._sprites = {};
    }
    
    //  SFX（bubbleOverlays）のみをクリア（フェードアウト付き）
    clearAllSfx() {
      const FADE_DURATION = 20; // フレーム数（約0.33秒 at 60fps）
      
      for (const actorId in this._sprites) {
        const group = this._sprites[actorId];
        if (group && group._bubble && group._bubble.children.length > 0) {
          // 各SFXスプライトにフェードアウト処理を設定
          const spritesToFade = [...group._bubble.children];
          
          for (const sprite of spritesToFade) {
            if (sprite._isFading) continue; // 既にフェード中ならスキップ
            
            sprite._isFading = true;
            sprite._fadeFrame = 0;
            sprite._fadeStartAlpha = sprite.opacity !== undefined ? sprite.opacity / 255 : sprite.alpha;
            
            // フェードアウト更新関数
            sprite._fadeUpdate = function() {
              this._fadeFrame++;
              const progress = this._fadeFrame / FADE_DURATION;
              
              if (progress >= 1.0) {
                // フェード完了：削除
                if (this.parent) {
                  this.parent.removeChild(this);
                }
                return true; // 完了
              } else {
                // 透明度を段階的に減らす
                const newAlpha = this._fadeStartAlpha * (1.0 - progress);
                if (this.opacity !== undefined) {
                  this.opacity = Math.floor(newAlpha * 255);
                } else {
                  this.alpha = newAlpha;
                }
                return false; // 継続中
              }
            };
          }
          
          // 状態管理をクリア（新しいSFXを即座に追加できるように）
          if (group._lastState) {
            group._lastState.bubbleOverlays = [];
          }
          
          console.log(`[BBM] Started fade-out for ${spritesToFade.length} SFX sprites (actor ${actorId})`);
        }
      }
    }
    
    //  毎フレーム更新：フェードアウト中のSFXを処理
    update() {
      for (const actorId in this._sprites) {
        const group = this._sprites[actorId];
        if (group && group._bubble) {
          const children = [...group._bubble.children];
          for (const sprite of children) {
            if (sprite._isFading && sprite._fadeUpdate) {
              sprite._fadeUpdate();
            }
          }
        }
      }
    }
  }

  if (!window.BattleBustManager) {
    window.BattleBustManager = new BattleBustManager();
  }

})();

// これを IIFE の外に置く（既存そのまま）
//  Scene_Battle 開始時に立ち絵を表示（アクターID 1 を想定）
function Window_ResolveChoice(rect, choices) {
  this._choices = choices || [];
  Window_Command.prototype.initialize.call(this, rect);
}
Window_ResolveChoice.prototype = Object.create(Window_Command.prototype);
Window_ResolveChoice.prototype.constructor = Window_ResolveChoice;
Window_ResolveChoice.prototype.makeCommandList = function() {
  if (!this._choices) return;
  this._choices.forEach(choice => this.addCommand(choice, choice));
};

const _Scene_Battle_start = Scene_Battle.prototype.start;
Scene_Battle.prototype.start = function() {
  _Scene_Battle_start.call(this);
  const actor = $gameActors.actor(1);
  if (actor && window.PleasureStateManager?.triggerBustRedraw) {
    setTimeout(() => window.PleasureStateManager.triggerBustRedraw(actor), 0);
  }
};

Scene_Battle.prototype.showResolveChoiceWindow = function(choices, callback) {
  const width = 400;
  const height = choices.length * 48 + 32;
  const x = 60;
  const y = 200;
  const rect = new Rectangle(x, y, width, height);
  const windowChoice = new Window_ResolveChoice(rect, choices);
  choices.forEach((text, i) => {
    windowChoice.setHandler(text, () => {
      windowChoice.close();
      this._windowLayer.removeChild(windowChoice);
      callback(i);
    });
  });
  windowChoice.setHandler('cancel', () => {
    windowChoice.close();
    this._windowLayer.removeChild(windowChoice);
    callback(0);
  });
  windowChoice.refresh();
  windowChoice.contentsOpacity = 255;
  this.addWindow(windowChoice);
  windowChoice.activate();
  windowChoice.select(0);
};
