// ClothingDamageEffect.js - 衣装破壊専用ビジュアルエフェクト
(() => {
  console.log("[ClothingDamageEffect] Loaded");

  /**
   * 案1：シャッター/ワイプ効果
   * 布が縦に裂けて左右に開き、新しい立ち絵（下着/全裸）が現れる
   * @param {Sprite} sprite - 新しい立ち絵のスプライト
   * @param {number} duration - エフェクトの長さ（フレーム）
   */
  function playTearTransition(sprite, duration = 50) {
    sprite.alpha = 0;
    let frame = 0;
    
    const originalX = sprite.x;
    
    // 効果音を再生（布が裂ける音）
    if (AudioManager && AudioManager.playSe) {
      AudioManager.playSe({
        name: "Damage2",
        volume: 90,
        pitch: 80,  // 低めのピッチで布が裂ける感じ
        pan: 0
      });
    }
    
    // シャッター（左右のカーテン）を作成
    let leftShutter = null;
    let rightShutter = null;
    let tearLine = null;
    
    if (sprite.parent) {
      // 左側のシャッター
      leftShutter = new PIXI.Graphics();
      leftShutter.beginFill(0x000000, 1);
      leftShutter.drawRect(0, 0, Graphics.width / 2, Graphics.height);
      leftShutter.endFill();
      leftShutter.x = 0;
      sprite.parent.addChild(leftShutter);
      
      // 右側のシャッター
      rightShutter = new PIXI.Graphics();
      rightShutter.beginFill(0x000000, 1);
      rightShutter.drawRect(0, 0, Graphics.width / 2, Graphics.height);
      rightShutter.endFill();
      rightShutter.x = Graphics.width / 2;
      sprite.parent.addChild(rightShutter);
      
      // 裂け目の線
      tearLine = new PIXI.Graphics();
      sprite.parent.addChild(tearLine);
    }
    
    const tick = () => {
      frame++;
      
      // Phase 1 (0-15): 裂け目が走る
      if (frame <= 15) {
        const progress = frame / 15;
        
        // 裂け目の線を描画（縦に走る）
        if (tearLine) {
          tearLine.clear();
          
          // 白い裂け目ライン
          tearLine.lineStyle(4, 0xFFFFFF, 1);
          const centerX = Graphics.width / 2;
          const lineLength = Graphics.height * progress;
          tearLine.moveTo(centerX, 0);
          tearLine.lineTo(centerX, lineLength);
          
          // グロー効果
          tearLine.lineStyle(10, 0xFFFFFF, 0.3);
          tearLine.moveTo(centerX, 0);
          tearLine.lineTo(centerX, lineLength);
        }
        
        // 新しい立ち絵を少し見せ始める
        sprite.alpha = progress * 0.3;
      }
      // Phase 2 (16-35): シャッターが左右に開く
      else if (frame <= 35) {
        const openProgress = (frame - 15) / 20;
        const easeOut = 1 - Math.pow(1 - openProgress, 3);  // イージング
        
        // シャッターを左右に移動
        const slideDistance = (Graphics.width / 2 + 100) * easeOut;
        if (leftShutter) {
          leftShutter.x = -slideDistance;
        }
        if (rightShutter) {
          rightShutter.x = Graphics.width / 2 + slideDistance;
        }
        
        // 裂け目の線をフェードアウト
        if (tearLine) {
          tearLine.alpha = 1 - openProgress;
        }
        
        // 新しい立ち絵をフェードイン
        sprite.alpha = 0.3 + openProgress * 0.7;
      }
      // Phase 3 (36-50): 完全に表示
      else {
        sprite.alpha = 1;
        
        // シャッターと裂け目を削除
        if (leftShutter && leftShutter.parent) {
          leftShutter.parent.removeChild(leftShutter);
          leftShutter = null;
        }
        if (rightShutter && rightShutter.parent) {
          rightShutter.parent.removeChild(rightShutter);
          rightShutter = null;
        }
        if (tearLine && tearLine.parent) {
          tearLine.parent.removeChild(tearLine);
          tearLine = null;
        }
      }
      
      if (frame < duration) {
        requestAnimationFrame(tick);
      } else {
        sprite.alpha = 1;
        sprite.x = originalX;
        
        // クリーンアップ
        if (leftShutter && leftShutter.parent) {
          leftShutter.parent.removeChild(leftShutter);
        }
        if (rightShutter && rightShutter.parent) {
          rightShutter.parent.removeChild(rightShutter);
        }
        if (tearLine && tearLine.parent) {
          tearLine.parent.removeChild(tearLine);
        }
      }
    };
    
    requestAnimationFrame(tick);
  }

  /**
   * 案2：フラグメント/シャッター効果
   * 旧立ち絵が破片のように砕けて散り、新しい立ち絵（下着/全裸）が現れる
   * @param {Sprite} sprite - 新しい立ち絵のスプライト
   * @param {number} duration - エフェクトの長さ（フレーム）
   */
  function playFragmentTransition(sprite, duration = 50) {
    sprite.alpha = 0;
    let frame = 0;
    
    const originalX = sprite.x;
    
    // 効果音を再生
    if (AudioManager && AudioManager.playSe) {
      AudioManager.playSe({
        name: "Damage2",
        volume: 90,
        pitch: 90,
        pan: 0
      });
    }
    
    // 破片用のグラフィックス配列
    const fragments = [];
    
    if (sprite.parent) {
      // 6x4 = 24個の破片を生成
      const cols = 6;
      const rows = 4;
      const fragmentWidth = Graphics.width / cols;
      const fragmentHeight = Graphics.height / rows;
      
      for (let row = 0; row < rows; row++) {
        for (let col = 0; col < cols; col++) {
          const fragment = new PIXI.Graphics();
          fragment.beginFill(0x333333, 1);  // 暗めの色（服の残骸）
          fragment.drawRect(0, 0, fragmentWidth, fragmentHeight);
          fragment.endFill();
          
          fragment.x = col * fragmentWidth;
          fragment.y = row * fragmentHeight;
          fragment.alpha = 1;
          
          // 各破片に物理パラメータを設定
          fragment._vx = (Math.random() - 0.5) * 12;  // 横方向の速度
          fragment._vy = Math.random() * -8 - 2;      // 上方向の速度
          fragment._vr = (Math.random() - 0.5) * 0.2; // 回転速度
          fragment._delay = Math.random() * 5;        // ランダムな開始遅延
          
          sprite.parent.addChild(fragment);
          fragments.push(fragment);
        }
      }
    }
    
    const tick = () => {
      frame++;
      
      // Phase 1 (0-10): 破片が出現して飛び始める
      if (frame <= 10) {
        const progress = frame / 10;
        
        fragments.forEach(frag => {
          if (frame < frag._delay) {
            // まだ開始していない
            return;
          }
          
          // 少し移動開始
          const localFrame = frame - frag._delay;
          frag.x += frag._vx * 0.5;
          frag.y += frag._vy * 0.5;
          frag._vy += 0.3; // 重力
          frag.rotation += frag._vr * 0.5;
        });
        
        // 新しい立ち絵を少し見せ始める
        sprite.alpha = progress * 0.2;
      }
      // Phase 2 (11-35): 破片が飛び散る
      else if (frame <= 35) {
        const scatterProgress = (frame - 10) / 25;
        
        fragments.forEach(frag => {
          // 破片の移動と回転
          frag.x += frag._vx;
          frag.y += frag._vy;
          frag._vy += 0.5; // 重力加速
          frag.rotation += frag._vr;
          
          // フェードアウト
          frag.alpha = 1 - scatterProgress;
        });
        
        // 新しい立ち絵をフェードイン
        sprite.alpha = 0.2 + scatterProgress * 0.8;
      }
      // Phase 3 (36-50): 完全に表示
      else {
        sprite.alpha = 1;
        
        // 破片を削除
        fragments.forEach(frag => {
          if (frag.parent) frag.parent.removeChild(frag);
        });
        fragments.length = 0;
      }
      
      if (frame < duration) {
        requestAnimationFrame(tick);
      } else {
        sprite.alpha = 1;
        sprite.x = originalX;
        
        // クリーンアップ
        fragments.forEach(frag => {
          if (frag.parent) frag.parent.removeChild(frag);
        });
      }
    };
    
    requestAnimationFrame(tick);
  }

  /**
   * 案3：マスク効果
   * 破れ目の形のマスクが走り、破れた部分から新立ち絵が見え、全体が切り替わる
   * @param {Sprite} sprite - 新しい立ち絵のスプライト
   * @param {number} duration - エフェクトの長さ（フレーム）
   */
  function playMaskTransition(sprite, duration = 50) {
    sprite.alpha = 0;
    let frame = 0;
    
    const originalX = sprite.x;
    
    // 効果音を再生
    if (AudioManager && AudioManager.playSe) {
      AudioManager.playSe({
        name: "Damage2",
        volume: 90,
        pitch: 85,
        pan: 0
      });
    }
    
    // マスク用のグラフィックスを作成
    let mask = null;
    let tearLine = null;
    
    if (sprite.parent) {
      // マスク（破れた部分を表示）
      mask = new PIXI.Graphics();
      sprite.parent.addChild(mask);
      sprite.mask = mask;
      
      // 裂け目のライン（演出用）
      tearLine = new PIXI.Graphics();
      sprite.parent.addChild(tearLine);
    }
    
    const tick = () => {
      frame++;
      
      // Phase 1 (0-20): 裂け目が走る
      if (frame <= 20) {
        const progress = frame / 20;
        
        if (mask) {
          mask.clear();
          mask.beginFill(0xFFFFFF, 1);
          
          // 不規則な裂け目を描画（ジグザグ）
          const centerX = Graphics.width / 2;
          const maxWidth = 40 * progress;  // 裂け目の幅
          const segments = 20;
          
          mask.moveTo(centerX - maxWidth / 2, 0);
          
          for (let i = 0; i <= segments; i++) {
            const y = (Graphics.height / segments) * i;
            const randomOffset = (Math.sin(i * 0.5 + frame * 0.2) * 15 + Math.random() * 10) * progress;
            const x = centerX + randomOffset;
            const width = maxWidth * (1 + Math.sin(i * 0.3) * 0.3);
            
            mask.lineTo(x - width / 2, y);
          }
          
          // 右側を描く（戻る）
          for (let i = segments; i >= 0; i--) {
            const y = (Graphics.height / segments) * i;
            const randomOffset = (Math.sin(i * 0.5 + frame * 0.2) * 15 + Math.random() * 10) * progress;
            const x = centerX + randomOffset;
            const width = maxWidth * (1 + Math.sin(i * 0.3) * 0.3);
            
            mask.lineTo(x + width / 2, y);
          }
          
          mask.closePath();
          mask.endFill();
        }
        
        // 裂け目の縁を描く（白い線）
        if (tearLine) {
          tearLine.clear();
          tearLine.lineStyle(3, 0xFFFFFF, 1);
          
          const centerX = Graphics.width / 2;
          const segments = 20;
          
          tearLine.moveTo(centerX, 0);
          for (let i = 0; i <= segments; i++) {
            const y = (Graphics.height / segments) * i;
            const randomOffset = (Math.sin(i * 0.5 + frame * 0.2) * 15 + Math.random() * 10) * progress;
            tearLine.lineTo(centerX + randomOffset, y);
          }
        }
        
        sprite.alpha = 1;
      }
      // Phase 2 (21-40): 裂け目が広がる
      else if (frame <= 40) {
        const expandProgress = (frame - 20) / 20;
        
        if (mask) {
          mask.clear();
          mask.beginFill(0xFFFFFF, 1);
          
          // 裂け目が広がる
          const centerX = Graphics.width / 2;
          const maxWidth = 40 + 200 * expandProgress;  // どんどん広がる
          const segments = 20;
          
          mask.moveTo(centerX - maxWidth / 2, 0);
          
          for (let i = 0; i <= segments; i++) {
            const y = (Graphics.height / segments) * i;
            const randomOffset = Math.sin(i * 0.5 + frame * 0.2) * 15;
            const x = centerX + randomOffset;
            const width = maxWidth * (1 + Math.sin(i * 0.3) * 0.3);
            
            mask.lineTo(x - width / 2, y);
          }
          
          for (let i = segments; i >= 0; i--) {
            const y = (Graphics.height / segments) * i;
            const randomOffset = Math.sin(i * 0.5 + frame * 0.2) * 15;
            const x = centerX + randomOffset;
            const width = maxWidth * (1 + Math.sin(i * 0.3) * 0.3);
            
            mask.lineTo(x + width / 2, y);
          }
          
          mask.closePath();
          mask.endFill();
        }
        
        // 裂け目の線をフェードアウト
        if (tearLine) {
          tearLine.alpha = 1 - expandProgress;
        }
        
        sprite.alpha = 1;
      }
      // Phase 3 (41-50): マスクを外して完全表示
      else {
        const fadeProgress = (frame - 40) / 10;
        
        if (fadeProgress >= 1) {
          sprite.mask = null;
          sprite.alpha = 1;
          
          // マスクと線を削除
          if (mask && mask.parent) {
            mask.parent.removeChild(mask);
            mask = null;
          }
          if (tearLine && tearLine.parent) {
            tearLine.parent.removeChild(tearLine);
            tearLine = null;
          }
        }
      }
      
      if (frame < duration) {
        requestAnimationFrame(tick);
      } else {
        sprite.mask = null;
        sprite.alpha = 1;
        sprite.x = originalX;
        
        // クリーンアップ
        if (mask && mask.parent) {
          mask.parent.removeChild(mask);
        }
        if (tearLine && tearLine.parent) {
          tearLine.parent.removeChild(tearLine);
        }
      }
    };
    
    requestAnimationFrame(tick);
  }

  /**
   * 案4：ディゾルブ/ティアエフェクト
   * 旧立ち絵の服部分がランダムに透明化（破れていく）して、新立ち絵が見える
   * @param {Sprite} sprite - 新しい立ち絵のスプライト
   * @param {number} duration - エフェクトの長さ（フレーム）
   */
  function playDissolveTransition(sprite, duration = 45) {
    sprite.alpha = 0;
    let frame = 0;
    
    const originalX = sprite.x;
    
    // 効果音を再生
    if (AudioManager && AudioManager.playSe) {
      AudioManager.playSe({
        name: "Damage2",
        volume: 90,
        pitch: 75,  // 低めで布がボロボロになる感じ
        pan: 0
      });
    }
    
    // ディゾルブ用のマスク配列（ランダムな破れ目）
    let dissolveMask = null;
    const tearPoints = [];  // 破れ始める点
    
    if (sprite.parent) {
      dissolveMask = new PIXI.Graphics();
      sprite.parent.addChild(dissolveMask);
      sprite.mask = dissolveMask;
      
      // ランダムな位置に破れ始める点を生成（15個）
      for (let i = 0; i < 15; i++) {
        tearPoints.push({
          x: Math.random() * Graphics.width,
          y: Math.random() * Graphics.height,
          speed: 0.5 + Math.random() * 1.5,  // 広がる速度
          delay: Math.random() * 10  // 開始遅延
        });
      }
    }
    
    const tick = () => {
      frame++;
      
      // Phase 1-2 (0-35): ディゾルブが進行
      if (frame <= 35) {
        const progress = frame / 35;
        
        if (dissolveMask) {
          dissolveMask.clear();
          dissolveMask.beginFill(0xFFFFFF, 1);
          
          // 各破れ点から円形に広がる
          tearPoints.forEach(point => {
            if (frame < point.delay) return;
            
            const localFrame = frame - point.delay;
            const radius = localFrame * point.speed * 8;
            
            if (radius > 0) {
              dissolveMask.drawCircle(point.x, point.y, radius);
            }
          });
          
          dissolveMask.endFill();
        }
        
        sprite.alpha = 1;
      }
      // Phase 3 (36-45): マスクを外して完全表示
      else {
        const fadeProgress = (frame - 35) / 10;
        
        if (fadeProgress >= 0.5 && dissolveMask) {
          // マスクを画面全体に広げる
          dissolveMask.clear();
          dissolveMask.beginFill(0xFFFFFF, 1);
          dissolveMask.drawRect(0, 0, Graphics.width, Graphics.height);
          dissolveMask.endFill();
        }
        
        if (fadeProgress >= 1) {
          sprite.mask = null;
          sprite.alpha = 1;
          
          if (dissolveMask && dissolveMask.parent) {
            dissolveMask.parent.removeChild(dissolveMask);
            dissolveMask = null;
          }
        }
      }
      
      if (frame < duration) {
        requestAnimationFrame(tick);
      } else {
        sprite.mask = null;
        sprite.alpha = 1;
        sprite.x = originalX;
        
        // クリーンアップ
        if (dissolveMask && dissolveMask.parent) {
          dissolveMask.parent.removeChild(dissolveMask);
        }
      }
    };
    
    requestAnimationFrame(tick);
  }

  /**
   * 案5候補：スラッシュライン（参考用）
   * 白いスラッシュラインが走り「ビリッと裂けた」感を演出
   * @param {Sprite} sprite - アニメーション対象のスプライト
   * @param {number} duration - エフェクトの長さ（フレーム）
   */
  function playSlashEffect(sprite, duration = 40) {
    sprite.alpha = 0;
    let frame = 0;
    
    const originalX = sprite.x;
    const originalY = sprite.y;
    
    // 効果音を再生
    if (AudioManager && AudioManager.playSe) {
      AudioManager.playSe({
        name: "Damage2",
        volume: 90,
        pitch: 100,
        pan: 0
      });
    }
    
    // スラッシュライン用のグラフィックス
    let slashLine = null;
    
    const tick = () => {
      frame++;
      
      // Phase 1 (0-15): スラッシュが走る
      if (frame <= 15) {
        const slashProgress = frame / 15;
        
        // スラッシュラインを描画
        if (frame === 1 && sprite.parent) {
          slashLine = new PIXI.Graphics();
          sprite.parent.addChild(slashLine);
        }
        
        if (slashLine) {
          slashLine.clear();
          
          // 太い白いスラッシュライン
          slashLine.lineStyle(8, 0xFFFFFF, 1);
          const angle = Math.PI / 4; // 斜め45度
          const length = 800 * slashProgress;
          const startX = sprite.x - 300;
          const startY = sprite.y - 250;
          slashLine.moveTo(startX, startY);
          slashLine.lineTo(
            startX + Math.cos(angle) * length,
            startY + Math.sin(angle) * length
          );
          
          // グロー効果（半透明の太いライン）
          slashLine.lineStyle(15, 0xFFFFFF, 0.3);
          slashLine.moveTo(startX, startY);
          slashLine.lineTo(
            startX + Math.cos(angle) * length,
            startY + Math.sin(angle) * length
          );
        }
        
        // スラッシュに沿って画像を表示
        sprite.alpha = slashProgress * 0.5;
        sprite.setColorTone([255, 255, 255, 0]);
      }
      // Phase 2 (16-30): 全体が現れる + 衝撃波
      else if (frame <= 30) {
        const revealProgress = (frame - 15) / 15;
        sprite.alpha = 0.5 + revealProgress * 0.5;
        
        // スラッシュラインをフェードアウト
        if (slashLine) {
          slashLine.alpha = 1 - revealProgress;
        }
        
        // 衝撃波的な揺れ
        const shakeIntensity = (1 - revealProgress) * 12;
        sprite.x = originalX + Math.sin(frame * 3) * shakeIntensity;
        
        // 色を戻す（ピンクがかった色から）
        const colorFade = 1 - revealProgress;
        sprite.setColorTone([255 * colorFade, 200 * colorFade, 200 * colorFade, 0]);
      }
      // Phase 3 (31-40): 安定
      else {
        sprite.alpha = 1;
        sprite.x = originalX;
        sprite.setColorTone([0, 0, 0, 0]);
        
        if (slashLine && slashLine.parent) {
          slashLine.parent.removeChild(slashLine);
          slashLine = null;
        }
      }
      
      if (frame < duration) {
        requestAnimationFrame(tick);
      } else {
        sprite.alpha = 1;
        sprite.x = originalX;
        sprite.setColorTone([0, 0, 0, 0]);
        
        if (slashLine && slashLine.parent) {
          slashLine.parent.removeChild(slashLine);
        }
      }
    };
    
    requestAnimationFrame(tick);
  }

  /**
   * 衣装破壊エフェクト：破片が飛び散る
   * 服がバラバラに裂けて破片が飛散する演出
   * @param {Sprite} sprite - アニメーション対象のスプライト
   * @param {number} duration - エフェクトの長さ（フレーム）
   */
  function playShredEffect(sprite, duration = 45) {
    sprite.alpha = 0;
    let frame = 0;
    
    const originalX = sprite.x;
    const originalY = sprite.y;
    const particles = [];
    
    // 効果音を再生
    if (AudioManager && AudioManager.playSe) {
      AudioManager.playSe({
        name: "Damage2",
        volume: 90,
        pitch: 100,
        pan: 0
      });
    }
    
    // 破片パーティクルを生成
    if (sprite.parent) {
      for (let i = 0; i < 8; i++) {
        const particle = new PIXI.Graphics();
        particle.beginFill(0xFFFFFF, 0.8);
        
        // 破片の形（ランダムな長方形）
        const size = 5 + Math.random() * 10;
        particle.drawRect(0, 0, size, size * 2);
        particle.endFill();
        
        particle.x = sprite.x + (Math.random() - 0.5) * 100;
        particle.y = sprite.y + (Math.random() - 0.5) * 100;
        particle.alpha = 0;
        
        // 飛散方向
        particle.velocityX = (Math.random() - 0.5) * 8;
        particle.velocityY = Math.random() * -5 - 3; // 上方向
        particle.rotation = Math.random() * Math.PI * 2;
        particle.rotationSpeed = (Math.random() - 0.5) * 0.3;
        
        sprite.parent.addChild(particle);
        particles.push(particle);
      }
    }
    
    const tick = () => {
      frame++;
      
      // Phase 1 (0-8): 破裂の瞬間
      if (frame <= 8) {
        sprite.alpha = frame / 8;
        sprite.setColorTone([255, 255, 255, 0]);
        
        // パーティクル出現
        particles.forEach(p => {
          p.alpha = frame / 8;
        });
      }
      // Phase 2 (9-35): パーティクルが飛び散る
      else if (frame <= 35) {
        sprite.alpha = 1;
        const particleProgress = (frame - 8) / 27;
        
        // 本体は揺れる
        const damping = 1 - particleProgress;
        sprite.x = originalX + Math.sin(frame * 2) * 8 * damping;
        sprite.setColorTone([200 * damping, 200 * damping, 200 * damping, 0]);
        
        // パーティクルアニメーション
        particles.forEach(p => {
          p.x += p.velocityX;
          p.y += p.velocityY;
          p.velocityY += 0.3; // 重力
          p.rotation += p.rotationSpeed;
          p.alpha = 1 - particleProgress;
        });
      }
      // Phase 3 (36-45): クリーンアップ
      else {
        sprite.alpha = 1;
        sprite.x = originalX;
        sprite.setColorTone([0, 0, 0, 0]);
        
        particles.forEach(p => {
          p.alpha = 0;
        });
      }
      
      if (frame < duration) {
        requestAnimationFrame(tick);
      } else {
        sprite.alpha = 1;
        sprite.x = originalX;
        sprite.y = originalY;
        sprite.setColorTone([0, 0, 0, 0]);
        
        // パーティクルを削除
        particles.forEach(p => {
          if (p.parent) p.parent.removeChild(p);
        });
      }
    };
    
    requestAnimationFrame(tick);
  }

  /**
   * 衣装破壊エフェクト：シンプル版
   * 布が激しく揺れる演出（軽量）
   * @param {Sprite} sprite - アニメーション対象のスプライト
   * @param {number} duration - エフェクトの長さ（フレーム）
   */
  function playSimpleEffect(sprite, duration = 50) {
    sprite.alpha = 0;
    let frame = 0;
    
    const originalX = sprite.x;
    const originalY = sprite.y;
    const originalRotation = sprite.rotation || 0;
    
    // 効果音を再生
    if (AudioManager && AudioManager.playSe) {
      AudioManager.playSe({
        name: "Damage2",
        volume: 90,
        pitch: 100,
        pan: 0
      });
    }
    
    const tick = () => {
      frame++;
      
      // Phase 1 (0-10): 裂ける瞬間 - 激しい揺れ
      if (frame <= 10) {
        sprite.alpha = frame / 10;
        
        // ランダムな激しい揺れ（布がビリッと裂ける）
        const intensity = 15;
        sprite.x = originalX + (Math.random() - 0.5) * intensity;
        sprite.y = originalY + (Math.random() - 0.5) * intensity * 0.5;
        sprite.rotation = originalRotation + (Math.random() - 0.5) * 0.1;
        
        // 白フラッシュ
        sprite.setColorTone([255, 255, 255, 0]);
      }
      // Phase 2 (11-30): 布が落ち着く - 減衰振動
      else if (frame <= 30) {
        sprite.alpha = 1;
        const shakeProgress = (frame - 10) / 20;
        const damping = Math.pow(1 - shakeProgress, 2); // 急速に減衰
        
        // 横揺れ（布がヒラヒラ）
        const sway = Math.sin(frame * 0.8) * 10 * damping;
        sprite.x = originalX + sway;
        
        // 微妙な回転（布が傾く）
        sprite.rotation = originalRotation + Math.sin(frame * 0.6) * 0.05 * damping;
        
        // 色を元に戻す
        const colorFade = 1 - shakeProgress;
        sprite.setColorTone([255 * colorFade, 255 * colorFade, 255 * colorFade, 0]);
      }
      // Phase 3 (31-50): 完全に安定
      else {
        const stabilizeProgress = (frame - 30) / 20;
        const easeOut = 1 - Math.pow(1 - stabilizeProgress, 3);
        
        sprite.x = originalX + (sprite.x - originalX) * (1 - easeOut);
        sprite.rotation = originalRotation + (sprite.rotation - originalRotation) * (1 - easeOut);
        sprite.setColorTone([0, 0, 0, 0]);
      }
      
      if (frame < duration) {
        requestAnimationFrame(tick);
      } else {
        sprite.alpha = 1;
        sprite.x = originalX;
        sprite.y = originalY;
        sprite.rotation = originalRotation;
        sprite.setColorTone([0, 0, 0, 0]);
      }
    };
    
    requestAnimationFrame(tick);
  }

  // グローバルAPIとして公開
  window.ClothingDamageEffect = {
    playTearTransition,       // 案1：シャッター/ワイプ効果【採用】
    playFragmentTransition,   // 案2：フラグメント/砕ける効果
    playMaskTransition,       // 案3：マスク効果
    playDissolveTransition,   // 案4：ディゾルブ/ティアエフェクト
    playSlashEffect,          // その他：スラッシュライン
    playShredEffect,          // その他：破片が飛び散る
    playSimpleEffect          // その他：シンプルな揺れ
  };

  console.log("[ClothingDamageEffect] Ready - 7 effects available (ACTIVE: playTearTransition)");
})();

