/*:
 * @target MZ
 * @plugindesc Custom UI Manager for RPGツクールMZ - カスタムUI管理プラグイン
 * @help
 * 市販コンポーネントを使用して、ウィンドウやボタンの位置を動的に設定し、
 * 不要なデフォルトウィンドウを非表示にするプラグインです。
 * 
 * - ウィンドウの位置やサイズをカスタマイズ
 * - 不要なデフォルトウィンドウを非表示化
 * - 特定キーでUIの表示/非表示を切り替え
 * 
 * @param ButtonX
 * @text ボタンX座標
 * @type number
 * @default 100
 * 
 * @param ButtonY
 * @text ボタンY座標
 * @type number
 * @default 200
 */

(() => {
  console.log("[CustomUIManager] Loaded");

  const parameters = PluginManager.parameters("CustomUIManager");
  const buttonX = Number(parameters["ButtonX"] || 100);
  const buttonY = Number(parameters["ButtonY"] || 200);

  // グローバル管理オブジェクト
  window.UIManager = {
      // 画像をロードする
      loadUI: function(name) {
          return ImageManager.loadBitmap("img/ui/", name);
      },

      // UI要素を動的に配置する
      setPosition: function(element, x, y) {
          element.x = x;
          element.y = y;
          console.log(`[UIManager] Position set: (${x}, ${y})`);
      },

      // ボタンを作成する
      createButton: function(imageName) {
          const button = new Sprite(this.loadUI("buttons/" + imageName));
          this.setPosition(button, buttonX, buttonY);
          SceneManager._scene.addChild(button);
          return button;
      },

      // ゲージを作成する
      createGauge: function(imageName, x, y, scale) {
          const gauge = new Sprite(this.loadUI("gauges/" + imageName));
          this.setPosition(gauge, x, y);
          gauge.scale.x = scale;
          SceneManager._scene.addChild(gauge);
          return gauge;
      },

      // ウィンドウ位置とサイズをカスタマイズ
      customizeWindow: function(window, x, y, width, height, opacity = 255) {
          window.move(x, y, width, height);
          window.opacity = opacity;
          console.log(`[UIManager] Customized window at (${x}, ${y}) with size (${width}, ${height})`);
      },

      // ウィンドウ非表示
      hideWindow: function(window) {
          window.hide();
          console.log("[UIManager] Window hidden");
      },

      // ウィンドウ表示
      showWindow: function(window) {
          window.show();
          console.log("[UIManager] Window shown");
      },

      // デフォルトUI非表示（例：ステータスウィンドウ）
      hideDefaultUI: function() {
          Window_StatusBase.prototype.initialize = function(rect) {
              this.opacity = 0;
              this.hide();
              console.log("[UIManager] Default UI hidden");
          };
      }
  };

  // 戦闘画面のコマンドウィンドウを右下に移動
  const _Window_ActorCommand_initialize = Window_ActorCommand.prototype.initialize;
  Window_ActorCommand.prototype.initialize = function(rect) {
      const x = Graphics.width - 220;
      const y = Graphics.height - 120;
      const width = 200;
      const height = 100;
      _Window_ActorCommand_initialize.call(this, new Rectangle(x, y, width, height));
      this.opacity = 200;
      console.log("[CustomUIManager] Actor command window repositioned");
  };

  // 特定キーでウィンドウ表示/非表示を切り替え
  const _Scene_Map_update = Scene_Map.prototype.update;
  Scene_Map.prototype.update = function() {
      _Scene_Map_update.call(this);
      if (Input.isTriggered("ok")) {  // Zキーで表示/非表示切り替え
          if (this._customWindow) {
              if (this._customWindow.visible) {
                  UIManager.hideWindow(this._customWindow);
              } else {
                  UIManager.showWindow(this._customWindow);
              }
              console.log("[CustomUIManager] Window visibility toggled");
          }
      }
  };

  // カスタムウィンドウの作成例
  Scene_Map.prototype.createCustomWindow = function() {
      const x = 50;
      const y = 50;
      const width = 300;
      const height = 100;
      this._customWindow = new Window_Base(new Rectangle(x, y, width, height));
      UIManager.customizeWindow(this._customWindow, x, y, width, height, 180);
      this._customWindow.drawText("カスタムウィンドウ", 0, 0, this._customWindow.contents.width, "center");
      this.addWindow(this._customWindow);
  };

  // シーン作成時に呼び出し
  const _Scene_Map_createAllWindows = Scene_Map.prototype.createAllWindows;
  Scene_Map.prototype.createAllWindows = function() {
      _Scene_Map_createAllWindows.call(this);
      this.createCustomWindow();
  };

})();
