// CutinManager.js — minimal cut-in overlay (frame + part closeup)
(() => {
  console.log("[CutinManager] Loaded");

  const PIC_ID_FRAME = 91;   // 競合しにくい固定ID
  const PIC_ID_PART  = 92;

  // プロファイル別の座標・デフォルト（省略可）
  const _cache = {};
  async function loadProfile(profileName) {
    if (_cache[profileName] !== undefined) return _cache[profileName];
    try {
      const res = await fetch(`dataEx/cutinProfiles/${profileName}.json`);
      if (!res.ok) throw new Error(res.statusText);
      const json = await res.json();
      _cache[profileName] = json;
      console.log(`[CutinManager] Profile loaded: ${profileName}`);
      return json;
    } catch (e) {
      console.warn(`[CutinManager] No cutin profile for ${profileName}`, e);
      _cache[profileName] = null;
      return null;
    }
  }

  // スロット座標（なければ右上寄りに出す）
  function resolveSlotXY(prof, slot = "R1") {
    const def = { x: Math.floor(Graphics.width * 0.75), y: Math.floor(Graphics.height * 0.28) };
    const slotDef = prof?.slots?.[slot];
    return { x: slotDef?.x ?? def.x, y: slotDef?.y ?? def.y };
  }

  // 画像キーをフルパスに解決（img/cutin/<profileName>/<key>.png）
  function resolvePartPath(profileName, key) {
    return `cutin/${profileName}/${key}.png`;
  }

  // 表示（duration: ms / scale: 倍率）
  function showPictures(frameName, partPath, x, y, scale = 1.0, durationMs = 160) {
    const s = Math.max(0.1, Number(scale));
    // 枠（任意）
    if (frameName) {
      $gameScreen.showPicture(PIC_ID_FRAME, `cutin_frames/${frameName}`, 0, x, y, s * 100, s * 100, 255, 0);
    }
    // 部位アップ
    $gameScreen.showPicture(PIC_ID_PART, partPath, 0, x, y, s * 100, s * 100, 255, 0);

    // 自動消去
    setTimeout(() => {
      $gameScreen.erasePicture(PIC_ID_FRAME);
      $gameScreen.erasePicture(PIC_ID_PART);
    }, durationMs);
  }

  async function show(profileName, key, opt = {}) {
    if (!(SceneManager._scene instanceof Scene_Battle)) return;
    const prof = await loadProfile(profileName);
    const { x, y } = resolveSlotXY(prof, opt.slot);
    const partPath = resolvePartPath(profileName, key);
    const duration = opt.duration ?? prof?.defaults?.duration ?? 160;
    const scale    = opt.scale    ?? prof?.defaults?.scale    ?? 1.0;
    const frame    = opt.frame    ?? prof?.defaults?.frame    ?? null;

    // オプション演出（軽く連携）
    if (opt.flash && window.PleasureFlashFX?.play) {
      window.PleasureFlashFX.play(profileName);
    }
    if (opt.sway && window.OneShotSway?.play) {
      OneShotSway.play($gameActors.actor(1), opt.sway);
    }

    showPictures(frame, partPath, x, y, scale, duration);
  }

  window.CutinManager = { show };
})();
