/*:
 * @target MZ
 * @plugindesc [EroDebug] MP / stage / choice / corruption をコンソールから即座に上書き＆取得するデバッグ補助
 * @author you
 * @help
 * ■ 目的
 *   - コンソールから MP / stage（=戦闘内の絶頂回数に由来）/ choice（resist|fall）/ corruption を
 *     任意にセット＆読み取りできる軽量デバッグユーティリティ。
 *
 * ■ 使い方（ゲーム中に F8 → Console）
 *   - 値の設定：
 *       EroDebug.set({ mp: 50 })
 *       EroDebug.set({ mpRate: 0.25 })
 *       EroDebug.set({ stageKey: "stage2" })
 *       EroDebug.set({ stageKey: "stage3", choice: "fall" })
 *       EroDebug.set({ corruption: 80 })
 *       EroDebug.set({ mpRate: 0.2, stageKey:"stage2", choice:"resist", corruption:60 })
 *
 *   - 現在値の取得：
 *       EroDebug.read()              // スナップショット（mp/mmp/mpRate, stage, choice, corruption）
 *       EroDebug.getStage()          // "stage1" | "stage2" | "stage3"
 *       EroDebug.getChoice()         // "resist" | "fall" | null
 *       EroDebug.getMp()             // { mp, mmp }
 *
 *   - すべての一時上書きを解除：
 *       EroDebug.clear()
 *
 * ■ 備考
 *   - stage は「絶頂回数（orgasmCount）」から決まります。
 *     0→stage1 / 1→stage2 / 2以上→stage3（必要があれば拡張してください）
 *   - choice は actor._resolveHistory[stageKey] を参照する想定です（resist/fall の分岐）。
 *   - corruption は Pleasure 系の実装が getCorruption() を参照する前提で一時上書きします。
 *
 * ■ プラグインコマンド
 *   - ありません（すべてコンソールから操作）。
 */

(() => {
  // ========== 小さなユーティリティ ==========

  /** number or function()->number の両対応で数値を取得 */
  function getNumeric(valueOrFn) {
    if (typeof valueOrFn === "function") {
      try { return Number(valueOrFn()); } catch (_) { return NaN; }
    }
    return Number(valueOrFn);
  }

  /** 絶頂回数 → stageKey */
  function toStageKeyFromCount(orgasmCount) {
    const n = Number(orgasmCount || 0);
    if (n >= 2) return "stage3";
    if (n === 1) return "stage2";
    return "stage1";
  }

  /** stageKey → 絶頂回数（代表値） */
  function toCountFromStageKey(stageKey) {
    if (stageKey === "stage3") return 2;
    if (stageKey === "stage2") return 1;
    return 0; // stage1
  }

  /** 上書き前のメソッドを退避 */
  function ensureDebugBackup(actor) {
    actor._eroDebug ||= { backups: {} };
    const backups = actor._eroDebug.backups;

    if (!backups.getOrgasmCount && typeof actor.getOrgasmCount === "function") {
      backups.getOrgasmCount = actor.getOrgasmCount;
    }
    if (!backups.incrementOrgasmCount && typeof actor.incrementOrgasmCount === "function") {
      backups.incrementOrgasmCount = actor.incrementOrgasmCount;
    }
    if (!backups.getCorruption && typeof actor.getCorruption === "function") {
      backups.getCorruption = actor.getCorruption;
    }
  }

  /** 絶頂回数（=stageの根拠）を一時上書き */
  function applyOrgasmOverrides(actor, orgasmCount) {
    ensureDebugBackup(actor);
    actor._eroDebug.orgasmCountOverride = Number(orgasmCount);
    actor.getOrgasmCount = () => actor._eroDebug.orgasmCountOverride;
    actor.incrementOrgasmCount = function() {
      actor._eroDebug.orgasmCountOverride = (actor._eroDebug.orgasmCountOverride ?? 0) + 1;
      return actor._eroDebug.orgasmCountOverride;
    };
  }

  /** corruption を一時上書き */
  function applyCorruptionOverride(actor, corruptionValue) {
    ensureDebugBackup(actor);
    const numeric = Number(corruptionValue);
    actor._eroDebug.corruptionOverride = numeric;
    actor.getCorruption = () => actor._eroDebug.corruptionOverride;
  }

  /** resist/fall の分岐履歴をセット */
  function setResolveChoice(actor, stageKey, choice /* "resist" | "fall" */) {
    actor._resolveHistory ||= {};
    actor._resolveHistory[stageKey] = choice;
  }

  /** actor.mp / actor.mmp に関する setter（関数/値の両対応） */
  function setActorMp(actor, mp) {
    if (typeof actor.setMp === "function") {
      actor.setMp(Number(mp));
    } else if ("_mp" in actor) {
      actor._mp = Number(mp);
    }
  }
  function readActorMp(actor) {
    const mp  = Number(getNumeric(actor.mp ?? actor.mp?.bind(actor)));
    const mmp = Number(getNumeric(actor.mmp ?? actor.mmp?.bind(actor)));
    // MV/MZの差異に配慮（関数/値）
    return {
      mp:  Number.isFinite(mp)  ? mp  : Number(actor._mp  ?? NaN),
      mmp: Number.isFinite(mmp) ? mmp : Number(actor._mmp ?? NaN),
    };
  }

  /** まとめて現在値を読む */
  function readSnapshotForActor(actor) {
    if (!actor) return null;

    const { mp, mmp } = readActorMp(actor);
    const orgasmCount = Number(getNumeric(actor.getOrgasmCount ?? actor.getOrgasmCount?.bind?.(actor)) || 0);
    const stageKey    = toStageKeyFromCount(orgasmCount);
    const resolveMap  = actor._resolveHistory || {};
    const choice      = resolveMap[stageKey] ?? null;

    let corruption = null;
    if (typeof actor.getCorruption === "function") {
      const c = Number(actor.getCorruption());
      corruption = Number.isFinite(c) ? c : null;
    }

    const actorId = (typeof actor.actorId === "function") ? actor.actorId() : (actor.actorId ?? 1);

    return {
      actorId,
      mp, mmp,
      mpRate: (Number.isFinite(mp) && Number.isFinite(mmp) && mmp > 0) ? +(mp / mmp).toFixed(3) : null,
      orgasmCount,
      stageKey,
      choice,            // "resist" | "fall" | null
      corruption         // number | null
    };
  }

  // ========== 公開API ==========

  window.EroDebug ||= {};

  /**
   * 任意セット
   * @param {object} args
   *   - actorId    : number = 1
   *   - mp         : number | null
   *   - mpRate     : number | null   // 最大MP比率（0.25 など）
   *   - stageKey   : "stage1"|"stage2"|"stage3" | null
   *   - orgasmCount: number | null   // 指定があればこちらを優先
   *   - choice     : "resist"|"fall" | null
   *   - corruption : number | null
   */
  window.EroDebug.set = function({
    actorId = 1,
    mp = null,
    mpRate = null,
    stageKey = null,
    orgasmCount = null,
    choice = null,
    corruption = null
  } = {}) {
    const actor = $gameActors.actor(actorId);
    if (!actor) {
      console.warn("[EroDebug.set] actor not found:", actorId);
      return;
    }

    // MP
    if (mpRate != null) {
      const { mmp } = readActorMp(actor);
      if (Number.isFinite(mmp)) setActorMp(actor, Math.round(mmp * Number(mpRate)));
    }
    if (mp != null) {
      setActorMp(actor, mp);
    }

    // stage（=絶頂回数）
    const counted =
      (orgasmCount != null) ? Number(orgasmCount)
      : (stageKey ? toCountFromStageKey(stageKey) : null);
    if (counted != null) {
      applyOrgasmOverrides(actor, counted);
    }

    // choice（resist/fall）
    if (choice) {
      const stageKeyResolved = stageKey ?? toStageKeyFromCount(
        (actor._eroDebug && actor._eroDebug.orgasmCountOverride != null)
          ? actor._eroDebug.orgasmCountOverride
          : Number(getNumeric(actor.getOrgasmCount ?? actor.getOrgasmCount?.bind?.(actor)) || 0)
      );
      setResolveChoice(actor, stageKeyResolved, choice);
    }

    // corruption（堕落値）
    if (corruption != null) {
      applyCorruptionOverride(actor, corruption);
    }

    const snap = readSnapshotForActor(actor);
    console.log("[EroDebug.set] done", snap);
  };

  /** 現在のスナップショットを取得・ログ出力 */
  window.EroDebug.read = function({ actorId = 1 } = {}) {
    const actor = $gameActors.actor(actorId);
    const snapshot = readSnapshotForActor(actor);
    console.log("[EroDebug.read]", snapshot);
    return snapshot;
  };

  /** 現在の stageKey だけ取得 */
  window.EroDebug.getStage = function({ actorId = 1 } = {}) {
    const actor = $gameActors.actor(actorId);
    const oc = Number(getNumeric(actor?.getOrgasmCount ?? actor?.getOrgasmCount?.bind?.(actor)) || 0);
    return toStageKeyFromCount(oc);
  };

  /** 現在の choice（resist/fall）だけ取得（stageKey省略時は現stage） */
  window.EroDebug.getChoice = function({ actorId = 1, stageKey = null } = {}) {
    const actor = $gameActors.actor(actorId);
    const oc = Number(getNumeric(actor?.getOrgasmCount ?? actor?.getOrgasmCount?.bind?.(actor)) || 0);
    const key = stageKey || toStageKeyFromCount(oc);
    return (actor?._resolveHistory || {})[key] ?? null;
  };

  /** 現在の MP / MMP を取得 */
  window.EroDebug.getMp = function({ actorId = 1 } = {}) {
    const actor = $gameActors.actor(actorId);
    const v = readActorMp(actor);
    console.log("[EroDebug.getMp]", v);
    return v;
  };

  /** すべての一時上書きを解除（元のメソッドへ戻す） */
  window.EroDebug.clear = function({ actorId = 1 } = {}) {
    const actor = $gameActors.actor(actorId);
    if (!actor?._eroDebug) {
      console.log("[EroDebug.clear] nothing to clear");
      return;
    }
    const backups = actor._eroDebug.backups || {};
    if (backups.getOrgasmCount)       actor.getOrgasmCount = backups.getOrgasmCount;
    if (backups.incrementOrgasmCount) actor.incrementOrgasmCount = backups.incrementOrgasmCount;
    if (backups.getCorruption)        actor.getCorruption = backups.getCorruption;

    delete actor._eroDebug;
    console.log("[EroDebug.clear] restored original methods");
  };

  // 便利なワンライナー（お好みで使用）
  window.EroDebug.printLine = function({ actorId = 1 } = {}) {
    const s = window.EroDebug.read({ actorId }) || {};
    if (!s) return;
    const rate = (s.mpRate != null) ? Math.round(s.mpRate * 100) + "%" : "—";
    console.log(`[EroDebug] MP ${s.mp}/${s.mmp} (${rate}) | stage=${s.stageKey} | choice=${s.choice} | corruption=${s.corruption}`);
  };

  console.log("[EroDebug] loaded");
})();

/* === append-only: safe installers for read/getters/print === */
(() => {
  // 依存関数をローカルに最小実装（外側がなくても動くようにしておく）
  const _num = v => (typeof v === "function" ? Number(v()) : Number(v));
  const _toStageKey = n => (n >= 2 ? "stage3" : n === 1 ? "stage2" : "stage1");
  const _readMp = (a) => {
    const mp  = _num(a?.mp ?? a?.mp?.bind?.(a));
    const mmp = _num(a?.mmp ?? a?.mmp?.bind?.(a));
    return {
      mp:  Number.isFinite(mp)  ? mp  : Number(a?._mp  ?? NaN),
      mmp: Number.isFinite(mmp) ? mmp : Number(a?._mmp ?? NaN),
    };
  };

  // グローバル確保
  window.EroDebug ||= {};

  // --- read() が無ければ定義 ---
  if (typeof window.EroDebug.read !== "function") {
    window.EroDebug.read = ({ actorId = 1 } = {}) => {
      const actor = $gameActors.actor(actorId);
      if (!actor) { console.warn("[EroDebug.read] actor not found", actorId); return null; }

      const { mp, mmp } = _readMp(actor);
      const oc       = Number(actor.getOrgasmCount?.() ?? 0);
      const stageKey = _toStageKey(oc);
      const choice   = (actor._resolveHistory || {})[stageKey] ?? null;
      const cRaw     = (typeof actor.getCorruption === "function") ? Number(actor.getCorruption()) : NaN;

      const snap = {
        actorId,
        mp, mmp,
        mpRate: (mmp > 0 && Number.isFinite(mp)) ? +(mp / mmp).toFixed(3) : null,
        orgasmCount: oc,
        stageKey,
        choice,
        corruption: Number.isFinite(cRaw) ? cRaw : null
      };
      console.log("[EroDebug.read]", snap);
      return snap;
    };
  }

  // --- getStage() が無ければ定義 ---
  if (typeof window.EroDebug.getStage !== "function") {
    window.EroDebug.getStage = ({ actorId = 1 } = {}) => {
      const a = $gameActors.actor(actorId);
      const oc = Number(a?.getOrgasmCount?.() ?? 0);
      return _toStageKey(oc);
    };
  }

  // --- getChoice() が無ければ定義 ---
  if (typeof window.EroDebug.getChoice !== "function") {
    window.EroDebug.getChoice = ({ actorId = 1, stageKey = null } = {}) => {
      const a = $gameActors.actor(actorId);
      const oc = Number(a?.getOrgasmCount?.() ?? 0);
      const key = stageKey || _toStageKey(oc);
      return (a?._resolveHistory || {})[key] ?? null;
    };
  }

  // --- getMp() が無ければ定義 ---
  if (typeof window.EroDebug.getMp !== "function") {
    window.EroDebug.getMp = ({ actorId = 1 } = {}) => {
      const a = $gameActors.actor(actorId);
      const v = _readMp(a);
      console.log("[EroDebug.getMp]", v);
      return v;
    };
  }

  // --- printLine() が無ければ定義（ワンライナー） ---
  if (typeof window.EroDebug.printLine !== "function") {
    window.EroDebug.printLine = ({ actorId = 1 } = {}) => {
      const s = window.EroDebug.read({ actorId }) || {};
      if (!s) return;
      const rate = (s.mpRate != null) ? Math.round(s.mpRate * 100) + "%" : "—";
      console.log(`[EroDebug] MP ${s.mp}/${s.mmp} (${rate}) | stage=${s.stageKey} | choice=${s.choice} | corruption=${s.corruption}`);
    };
  }

  console.log("[EroDebug] append-only helpers installed");
})();
