// EnemyActionManager.strict-guard.js
// ルール：未拘束状態ではエロ攻撃（Pleasure/Insert）を一切出さない
// 拘束解除ターンは通常攻撃のみ
// 修正：空のアクションリストをハンドリング
(() => {
  // ─ helpers ─
  const isRestraintSkill = s => !!s?.meta?.RestraintSkill;
  const isPleasureSkill  = s => !!s?.meta?.PleasureSkill;
  const isInsertSkill    = s => !!s?.meta?.InsertSkill;
  const getTargetActor   = () => $gameActors.actor(1);

  const isActorRestrained = actor => {
    try { return !!window.PleasureStateManager?.isActorRestrained?.(actor); }
    catch { return false; }
  };

  const isActorInserted = actor => {
    try { return !!actor?._pleasure?.inserted; }
    catch { return false; }
  };

  // 挿入解禁ポリシー（stage2+のみ、ただしモンスター特性で上書き可能）
  function canAttemptInsertNow(actor) {
    if (!actor) return false;
    if (!isActorRestrained(actor)) return false; // 拘束中のみ
    if (isActorInserted(actor)) return false;    // 既に挿入中は不可
    
    // ★ モンスター特性を確認（$gameTroop から取得）
    const enemies = $gameTroop ? $gameTroop.members() : null;
    const enemy = enemies?.[0];
    
    if (enemy && window.MonsterTraitManager) {
      const enemyKey = window.MonsterTraitManager.getEnemyKey(enemy.enemyId());
      if (enemyKey && window.MonsterTraitManager.allowsStage1Insert(enemyKey)) {
        console.log(`[EAM] ${enemyKey} allows stage1 insert (trait system)`);
        return true;
      }
    }
    
    // 通常のモンスターは stage2+ のみ
    const oc = actor.getOrgasmCount?.() || 0;
    const stageKey = window.PleasurePoseController?.determinePleasureStage?.(oc) || "stage1";
    const isStage2Plus = (stageKey === "stage2" || stageKey === "stage3");
    
    if (!isStage2Plus) {
      console.log(`[EAM] Insert blocked at ${stageKey} (default behavior)`);
    }
    
    return isStage2Plus;
  }

  // rating の重み付けを考慮したランダム選択
  const pickOne = arr => {
    if (!Array.isArray(arr) || !arr.length) return null;
    
    // rating による重み付け選択
    const totalWeight = arr.reduce((sum, action) => sum + (action._rating || 1), 0);
    let rand = Math.random() * totalWeight;
    
    for (const action of arr) {
      rand -= (action._rating || 1);
      if (rand <= 0) return action;
    }
    
    return arr[arr.length - 1];  // フォールバック
  };

  // ─ 拘束状態の記憶（敵ごと） ─
  const _Game_Enemy_setup = Game_Enemy.prototype.setup;
  Game_Enemy.prototype.setup = function(enemyId, x, y) {
    _Game_Enemy_setup.call(this, enemyId, x, y);
    this._lastTurnRestrained = false; // 前ターンの拘束状態
    this._justReleasedThisTurn = false; // このターンに拘束解除されたか
  };

  // ─ ステート解除時のフック（拘束解除検知） ─
  const RESTRAINT_STATE_IDS = [10, 11, 12, 13]; // 拘束ステートのID
  
  const _Game_Battler_removeState = Game_Battler.prototype.removeState;
  Game_Battler.prototype.removeState = function(stateId) {
    // プレイヤーの拘束ステートが解除された場合
    if (RESTRAINT_STATE_IDS.includes(stateId) && this.isActor()) {
      // 全敵に「このターンに拘束解除された」フラグを立てる
      if ($gameTroop) {
        for (const enemy of $gameTroop.members()) {
          if (enemy) {
            enemy._justReleasedThisTurn = true;
          }
        }
      }
    }
    _Game_Battler_removeState.call(this, stateId);
  };

  // ─ ターン開始時に前ターンの状態を記憶 ─
  const _BattleManager_startTurn = BattleManager.startTurn;
  BattleManager.startTurn = function() {
    const actor = getTargetActor();
    if (actor) {
      const currentRestrained = isActorRestrained(actor);
      // 全敵に前ターンの拘束状態を記憶
      for (const enemy of $gameTroop.members()) {
        if (enemy) {
          enemy._lastTurnRestrained = currentRestrained;
        }
      }
    }
    _BattleManager_startTurn.call(this);
  };

  // ─ ターン終了時に「解除フラグ」をクリア + SFXクリア ─
  const _BattleManager_endTurn = BattleManager.endTurn;
  BattleManager.endTurn = function() {
    // 全敵の「このターンに拘束解除された」フラグをクリア
    if ($gameTroop) {
      for (const enemy of $gameTroop.members()) {
        if (enemy) {
          enemy._justReleasedThisTurn = false;
        }
      }
    }
    
    // ★ ターン終了時にSFX（bubbleOverlays）を全てクリア
    if (window.BattleBustManager?.clearAllSfx) {
      window.BattleBustManager.clearAllSfx();
      console.log("[EAM] Cleared all SFX at turn end");
    }
    
    _BattleManager_endTurn.call(this);
  };

  // ─ 行動実行時チェック：エロ攻撃が無効なら通常攻撃に差し替え ─
  const _Game_Action_apply = Game_Action.prototype.apply;
  Game_Action.prototype.apply = function(target) {
    // 敵の行動の場合のみチェック
    if (this.subject().isEnemy() && this.item() && target.isActor()) {
      const skill = this.item();
      const actor = target;
      const restrained = isActorRestrained(actor);
      
      // エロ攻撃（Pleasure/Insert）なのに拘束されていない場合
      if ((isPleasureSkill(skill) || isInsertSkill(skill)) && !restrained) {
        this.setAttack();
      }
      
      // 拘束スキルなのに既に拘束されている場合
      if (isRestraintSkill(skill) && restrained) {
        this.setAttack();
      }
      
      // 挿入スキルなのに既に挿入中の場合
      if (isInsertSkill(skill) && isActorInserted(actor)) {
        this.setAttack();
      }
    }
    
    _Game_Action_apply.call(this, target);
  };

  // ─ override ─
  const _makeActions = Game_Enemy.prototype.makeActions;
  Game_Enemy.prototype.makeActions = function() {
    const actor = getTargetActor();
    if (!actor) {
      _makeActions.call(this);
      return;
    }

    // ★元の処理を使わず、敵の全アクションリストから全候補を生成
    const enemyActions = this.enemy().actions || [];
    this._actions = [];
    
    console.log(`[EAM] Enemy ${this.name()} has ${enemyActions.length} actions in database`);
    
    for (const actionData of enemyActions) {
      const skill = $dataSkills[actionData.skillId];
      if (skill) {
        if (skill.occasion === 1) {  // 戦闘中使用可能
          const action = new Game_Action(this);
          action.setSkill(actionData.skillId);
          action._rating = actionData.rating;  // rating を保存（後で使う）
          this._actions.push(action);
          console.log(`[EAM] Added skill: ${skill.name} (ID:${actionData.skillId}, rating:${actionData.rating})`);
        } else {
          console.log(`[EAM] Skipped skill: ${skill.name} (ID:${actionData.skillId}) - occasion=${skill.occasion} (not usable in battle)`);
        }
      }
    }
    
    console.log(`[EAM] Total ${this._actions.length} skills available for ${this.name()}`);
    
    
    const restrained = isActorRestrained(actor);
    const inserted   = isActorInserted(actor);
    const allowInsertNow = canAttemptInsertNow(actor);
    
    // ★ 拘束解除されたターンの検知（2つの方法）
    const wasRestrainedLastTurn = !!this._lastTurnRestrained;
    const justReleasedByTurn = wasRestrainedLastTurn && !restrained; // ターン跨ぎ検知
    const justReleasedThisTurn = !!this._justReleasedThisTurn; // ターン中の即時検知
    const justReleased = justReleasedByTurn || justReleasedThisTurn; // どちらかがtrueなら解除ターン

    // ★修正: 元の_actionsが空の場合、強制的に通常攻撃を追加
    if (!this._actions || this._actions.length === 0) {
      const attackAction = new Game_Action(this);
      attackAction.setAttack();
      this._actions = [attackAction];
      return; // 通常攻撃を実行
    }

    // 候補を分類（DBメタ：<RestraintSkill>/<PleasureSkill>/<InsertSkill>）
    const restraintPool = [];
    const pleasurePool  = [];
    const insertPool    = [];
    const otherPool     = [];

    for (const action of (this._actions || [])) {
      const skill = action?.item?.();
      if (!skill) {
        // ★修正: スキルが取得できない場合は通常攻撃として扱う
        otherPool.push(action);
        continue;
      }
      
      // ★新機能: 挿入状態に応じたスキルフィルタリング
      if (isPleasureSkill(skill)) {
        // NonInsertedOnly: 非挿入時のみ使用可能
        if (skill.meta?.NonInsertedOnly && inserted) {
          console.log(`[EAM] Skip ${skill.name}: NonInsertedOnly but actor is inserted`);
          continue;
        }
        // InsertedOnly: 挿入時のみ使用可能
        if (skill.meta?.InsertedOnly && !inserted) {
          console.log(`[EAM] Skip ${skill.name}: InsertedOnly but actor is not inserted`);
          continue;
        }
      }
      
      if (isRestraintSkill(skill)) { restraintPool.push(action); continue; }
      if (isInsertSkill(skill))    { insertPool.push(action);    continue; }
      if (isPleasureSkill(skill))  { pleasurePool.push(action);  continue; }
      otherPool.push(action);
    }

    // ─ 状態ガード ─
    // 拘束中の再拘束／挿入中の再挿入は禁止
    if (restrained) restraintPool.length = 0;
    if (inserted)   insertPool.length   = 0;

    console.log(`[EAM] Pools before filtering: restraint=${restraintPool.length}, pleasure=${pleasurePool.length}, insert=${insertPool.length}, other=${otherPool.length}`);
    console.log(`[EAM] State: restrained=${restrained}, inserted=${inserted}, allowInsertNow=${allowInsertNow}, justReleased=${justReleased}`);
    
    // ★未拘束ならエロ攻撃（Pleasure/Insert）は完全封鎖
    if (!restrained) {
      pleasurePool.length = 0;
      insertPool.length   = 0;
      console.log(`[EAM] Not restrained: cleared pleasure and insert pools`);
    }

    // ★拘束解除されたターンは通常攻撃のみ
    if (justReleased) {
      restraintPool.length = 0;
      pleasurePool.length = 0;
      insertPool.length = 0;
      console.log(`[EAM] Just released: cleared all special pools`);
    }

    // ★拘束中は通常攻撃を90%の確率で除外（エロ攻撃を優先）
    if (restrained && otherPool.length > 0 && (pleasurePool.length > 0 || insertPool.length > 0)) {
      if (Math.random() < 0.9) {
        otherPool.length = 0;
        console.log(`[EAM] Restrained: removed normal attacks (90% chance)`);
      }
    }
    
    console.log(`[EAM] Pools after filtering: restraint=${restraintPool.length}, pleasure=${pleasurePool.length}, insert=${insertPool.length}, other=${otherPool.length}`);

    // ─ 一本化ルール ─
    let selectedPool = null;
    let selectedReason = "";

    if (justReleased) {
      // 拘束解除ターン：通常攻撃のみ
      selectedPool = otherPool;
      selectedReason = "just released (normal attack only)";
    } else if (!restrained) {
      // 未拘束：拘束があれば拘束、無ければ通常（絶対にエロは出ない）
      selectedPool = restraintPool.length ? restraintPool : otherPool;
      selectedReason = restraintPool.length ? "not restrained (restraint available)" : "not restrained (normal attack)";
    } else if (allowInsertNow && insertPool.length) {
      // 拘束中 & stage2+ & 挿入可能
      selectedPool = insertPool;
      selectedReason = "restrained + stage2+ (insert available)";
    } else if (pleasurePool.length) {
      // 拘束中 & エロ攻撃可能
      selectedPool = pleasurePool;
      selectedReason = "restrained (pleasure attack)";
    } else {
      // 拘束中だがエロ攻撃がない
      selectedPool = otherPool;
      selectedReason = "restrained but no pleasure attacks (fallback to normal)";
    }

    console.log(`[EAM] Selected pool: ${selectedReason} (${selectedPool?.length || 0} options)`);
    
    const chosen = pickOne(selectedPool);
    
    // ★修正: chosenがnullの場合、強制的に通常攻撃
    if (!chosen) {
      console.log(`[EAM] No action chosen, using normal attack`);
      const attackAction = new Game_Action(this);
      attackAction.setAttack();
      this._actions = [attackAction];
    } else {
      const chosenSkill = chosen.item();
      console.log(`[EAM] Chosen action: ${chosenSkill?.name || 'Unknown'} (ID:${chosenSkill?.id || '?'})`);
      this._actions = [chosen];
    }
  };
})();
