/*:
 * @target MZ
 * @plugindesc Ero Memory (skill-like list & help) v1.1 2pane + Z/Stats toggle (menu-only)
 * @author You
 *
 * @command AddOrUpdate
 * @text 思い出を追加/更新
 * @desc IDで思い出を追加/更新します（同一IDは上書き）
 * @arg id
 * @type string
 * @text ID
 * @arg title
 * @type string
 * @text タイトル
 * @arg description
 * @type note
 * @text 説明
 * @arg iconIndex
 * @type number
 * @min 0
 * @default 0
 * @text アイコンINDEX（任意）
 * @arg thumbnail
 * @type file
 * @dir img/pictures
 * @text サムネ画像（任意）
 * @arg unlocked
 * @type boolean
 * @default false
 * @text 解放済みにする
 *
 * @command Unlock
 * @text 思い出を解放
 * @arg id
 * @type string
 * @text ID
 */

(() => {
  const PLUGIN_NAME =
    (document.currentScript && document.currentScript.src.match(/([^/]+)\.js$/)?.[1]) ||
    "EroMemory";
  console.log(`[${PLUGIN_NAME}] loaded`);

  // ---------- データ保管 ----------
  function ensureStore(){
    if (!$gameSystem._eroMemories) $gameSystem._eroMemories = [];
    return $gameSystem._eroMemories;
  }
  function findIndexById(id){
    const list = ensureStore();
    return list.findIndex(m => m && m.id === id);
  }

  // ---------- コアAPI ----------
  const EroMemory = {
    list(){ return ensureStore(); },
    addOrUpdate(mem){
      if (!mem.id) return;
      const idx = findIndexById(mem.id);
      if (idx >= 0) {
        $gameSystem._eroMemories[idx] = { ...$gameSystem._eroMemories[idx], ...mem };
      } else {
        $gameSystem._eroMemories.push(mem);
      }
    },
    unlock(id){
      const idx = findIndexById(id);
      if (idx >= 0) $gameSystem._eroMemories[idx].unlocked = true;
    },
    markSeen(id){
      const idx = findIndexById(id);
      if (idx >= 0) $gameSystem._eroMemories[idx].seen = true;
    },
    getTotals(){
      const t = { Z:0, dev:{}, tags:{} };
      for (const m of ensureStore()){
        if (!m || !m.unlocked) continue;
        const d = m.deltas || {};
        t.Z += Number(d.Z || m.score || 0);
        if (d.dev){
          for (const k in d.dev){
            t.dev[k] = (t.dev[k] || 0) + Number(d.dev[k] || 0);
          }
        }
        if (Array.isArray(m.tags)){
          for (const tag of m.tags) t.tags[tag] = (t.tags[tag]||0) + 1;
        }
      }
      return t;
    },
    applyEffects(mem){
      if (!mem) return;
      this._lastDeltaZ = (mem.deltas && Number(mem.deltas.Z||0)) || Number(mem.score||0) || 0;
    },
    _normalize(raw, idx){
      return {
        id: String(raw.id || ""),
        title: String(raw.title || ""),
        description: String(raw.description || ""),
        iconIndex: Number(raw.iconIndex || 0),
        thumbnail: String(raw.thumbnail || ""),
        unlocked: !!raw.unlocked,
        seen: !!raw.seen,
        tags: Array.isArray(raw.tags) ? raw.tags.slice() : [],
        deltas: raw.deltas || null,
        score: Number(raw.score || 0),
        order: (raw.order != null) ? Number(raw.order) : (idx + 1),
        time: Number(raw.time || 0) || (Date.now() + idx)
      };
    },
    seedFromJson({ overwrite=false } = {}){
      const arr = $dataEroMemory?.memories;
      if (!Array.isArray(arr)) return;
      if (!overwrite && ensureStore().length > 0) return;
      $gameSystem._eroMemories = arr.map((r, i) => EroMemory._normalize(r, i));
    }
  };
  window.EroMemory = EroMemory;

  // ---------- JSON 読み込み（安全版） ----------
  window.$dataEroMemory = null;
  const _DataManager_loadDatabase = DataManager.loadDatabase;
  DataManager.loadDatabase = function() {
    _DataManager_loadDatabase.call(this);
    // 既存DB読込の後に個別ロードを明示呼び出し（MZの実装差異に依存しない）
    this.loadDataFile("$dataEroMemory", "EroMemory.json");
  };

  // 新規ゲーム開始時：JSONをシード
  const _DataManager_setupNewGame = DataManager.setupNewGame;
  DataManager.setupNewGame = function(){
    _DataManager_setupNewGame.call(this);
    EroMemory.seedFromJson({ overwrite: true });
  };

  // セーブ読込時：空ならJSONをシード（互換）
  const _DataManager_extractSaveContents = DataManager.extractSaveContents;
  DataManager.extractSaveContents = function(contents){
    _DataManager_extractSaveContents.call(this, contents);
    if (!Array.isArray($gameSystem._eroMemories) || $gameSystem._eroMemories.length === 0){
      EroMemory.seedFromJson({ overwrite: true });
    }
  };

  // ---------- プラグインコマンド（閲覧系は撤去済み） ----------
  PluginManager.registerCommand(PLUGIN_NAME, "AddOrUpdate", args => {
    const mem = {
      id: String(args.id || ""),
      title: String(args.title || ""),
      description: (args.description || "").replace(/^"|"$/g, ""),
      iconIndex: Number(args.iconIndex || 0),
      thumbnail: String(args.thumbnail || ""),
      unlocked: String(args.unlocked || "false") === "true",
      seen: false,
      tags: [],
      time: Date.now()
    };
    EroMemory.addOrUpdate(mem);
  });

  PluginManager.registerCommand(PLUGIN_NAME, "Unlock", args => {
    EroMemory.unlock(String(args.id || ""));
  });

  // ---------- シーン（2窓：上=説明/統計、下=リスト） ----------
  class Scene_EroMemory extends Scene_MenuBase {
    create(){
      super.create();
      const ww = Graphics.boxWidth;
      const wh = Graphics.boxHeight;

      const helpH = Math.floor(wh * 0.30);

      this._help = new Window_EroMemoryHelp(new Rectangle(0, 0, ww, helpH));
      this._list = new Window_EroMemoryList(new Rectangle(0, helpH, ww, wh - helpH));
      this._list.setHelpWindow(this._help);

      this._list.setHandler("ok",     this.onOk.bind(this));
      this._list.setHandler("cancel", this.popScene.bind(this));

      this.addWindow(this._help);
      this.addWindow(this._list);
    }

    start(){
      super.start();
      this._list.refresh();
      this._help.refresh();
    }

    onOk(){
      const item = this._list.item();
      if (item && item.id) {
        EroMemory.markSeen(item.id);
        EroMemory.applyEffects(item);
      }
      this._list.activate();
    }
  }

  // ---------- リスト ----------
  class Window_EroMemoryList extends Window_Selectable {
    initialize(rect){
      super.initialize(rect);
      this.refresh();
      this.activate();
      this.select(0);
    }
    maxItems(){ return this._data ? this._data.length : 0; }
    item(){ return this._data?.[this.index()] || null; }
    makeItemList(){
      this._data = [...EroMemory.list()].sort((a,b)=>{
        const ao = (a.order != null) ? a.order : 999999;
        const bo = (b.order != null) ? b.order : 999999;
        if (ao !== bo) return ao - bo;
        return (b.time||0) - (a.time||0);
      });
    }
    itemHeight(){ return this.lineHeight() + 6; }

    drawItem(index){
      const rect = this.itemRectWithPadding(index);
      const item = this._data[index];
      if (!item) return;

      const unlocked = !!item.unlocked;
      const title = unlocked ? (item.title || "（無題）") : "??????";
      const icon = item.iconIndex || 0;

      this.changePaintOpacity(true);
      let x = rect.x;
      if (icon > 0) { this.drawIcon(icon, x, rect.y + 2); x += 36; }
      this.changeTextColor(unlocked ? ColorManager.normalColor() : ColorManager.textColor(7));
      this.drawText(title, x, rect.y, rect.width - (x - rect.x) - 60);
      this.resetTextColor();

      // 右端表示：Z+◯は非表示にする（NEW/未解放のみ残す）
      if (unlocked){
        if (!item.seen){
          this.changeTextColor(ColorManager.crisisColor());
          this.drawText("NEW", rect.x, rect.y, rect.width - 12, "right");
          this.resetTextColor();
        }
      } else {
        this.changeTextColor(ColorManager.textColor(7));
        this.drawText("未解放", rect.x, rect.y, rect.width - 12, "right");
        this.resetTextColor();
      }

    }

    updateHelp(){
      if (!this._helpWindow) return;
      this._helpWindow.setItem(this.item());
    }
    refresh(){ this.makeItemList(); super.refresh(); }
  }

  // ---------- 上窓：説明 ↔ 統計 切替 ----------
  class Window_EroMemoryHelp extends Window_Base {
    initialize(rect){
      super.initialize(rect);
      this._item = null;
    }
    toggleMode(){ /* 統計モード無効化 */ }
    setItem(item){ this._item = item; this.refresh(); }

    refresh(){
      this.contents.clear();
      this.drawDesc()
    }

    drawDesc(){
      const w = this.contentsWidth();
      const lh = this.lineHeight();
      const item = this._item;
      if (!item){
        this.drawText("—", 0, 0, w, "center");
        return;
      }
      const unlocked = !!item.unlocked;
      const title = unlocked ? (item.title || "（無題）") : "??????";
      const desc  = unlocked ? (item.description || "") : "条件を満たすと解放されます。";

      this.changeTextColor(ColorManager.systemColor());
      this.drawText(title, 0, 0, w);
      this.resetTextColor();
      this.drawTextEx(desc, 0, lh, w);

      this.changeTextColor(ColorManager.textColor(7));
      this.resetTextColor();
    }
  }

  // === メニューに「思い出」を追加（エロステの直下） ===
  // ★今作では非表示（内部機能は残す）
  // const _WMC_makeCommandList = Window_MenuCommand.prototype.makeCommandList;
  // Window_MenuCommand.prototype.makeCommandList = function() {
  //   _WMC_makeCommandList.call(this);
  //   const cmd = { name: "思い出", symbol: "eroMemory", enabled: true, ext: null };
  //   let idx = this._list.findIndex(e =>
  //     e && (e.symbol === "eroStatus" || e.name === "エロステータス")
  //   );
  //   const insertAt = (idx >= 0) ? idx + 1 : this._list.length;
  //   this._list.splice(insertAt, 0, cmd);
  // };

  // const _SceneMenu_createCommandWindow = Scene_Menu.prototype.createCommandWindow;
  // Scene_Menu.prototype.createCommandWindow = function() {
  //   _SceneMenu_createCommandWindow.call(this);
  //   this._commandWindow.setHandler("eroMemory", () => SceneManager.push(Scene_EroMemory));
  // };
})();
