/*:
 * @target MZ
 * @plugindesc EroStatus Core (Facade) — boot-time auto init & safe refresh
 */
(() => {
  const NS = (window.EroStatus ??= {});

  // ===== 内部フラグ（多重初期化ガード）=====
  let _wired = false;
  let _bootHooked = false;

  // ===== 初期化 =====
  function init(){
    // 依存がまだなら次フレームで再試行（読み込み順が前後しても最終的に初期化）
    if (!NS.Events || !NS.Store || !NS.Adapter || !NS.Controller || !NS.Selectors){
      return setTimeout(init, 0);
    }
    if (_wired) return;        // すでに配線済みなら何もしない
    _wired = true;

    // Controller がイベント配線（ACTOR_CHANGED / TAB_CHANGED / REQUEST_REFRESH）を担当
    NS.Controller.wire();

    // 最初のRawを投入（UIが存在する/しないに関わらず安全）
    refresh();

    // Bootフックを一度だけ仕込む（保険として起動時に必ず初期化が走る）
    hookBootOnce();
  }

  // ===== 再読込（UIからも使う）=====
  function refresh(){
    try {
      const id = NS.Store.getState().actorId;
      NS.Store.setRaw(NS.Adapter.collectRaw(id));
    } catch(e){
      console.error("[EroStatus:Core] refresh failed:", e);
    }
  }

  // ===== API: ソフトリフレッシュ要求（イベント経由）=====
  function requestRefresh(){
    if (NS.Events) NS.emit(NS.Events.REQUEST_REFRESH);
    else refresh(); // 念のため直呼びフォールバック
  }

  // ===== Bootへのワンタイムフック =====
  function hookBootOnce(){
    if (_bootHooked) return;
    _bootHooked = true;

    const _start = Scene_Boot.prototype.start;
    Scene_Boot.prototype.start = function(){
      _start.call(this);
      try {
        // 起動直後に必ず一度は初期化されるよう保険をかける
        init();
        // 表示前に最新化（メニュー直行ケースにも強い）
        refresh();
      } catch(e){ console.error(e); }
    };
  }

  // ===== 公開API =====
  NS.Core = {
    // lifecycle
    init, refresh, requestRefresh,
    // state操作
    setActor: id => NS.Events && NS.emit(NS.Events.ACTOR_CHANGED, id),
    setTab:   tab => NS.Events && NS.emit(NS.Events.TAB_CHANGED, tab),
    // 購読・選択子
    on: NS.on, off: NS.off, emit: NS.emit,
    events: NS.Events,
    selectors: NS.Selectors,
    subscribe: NS.Store.subscribe,
    getState:  NS.Store.getState,
  };

  // ===== 自動初期化起点 =====
  // 1) 直ちに試行（依存が未読込なら内部で次フレームへ遅延）
  init();
  // 2) Boot後の安全網も hookBootOnce() 内で確保
})();
