/*:
 * @target MZ
 * @plugindesc EroStatus Events (Bus)
 */
(() => {
  const NS  = (window.EroStatus ??= {});
  const TAG = "[EroStatus:Events]";

  // -----------------------------
  // 1) イベント名（定数）
  // -----------------------------
  const Events = {
    // 既存UI系
    TAB_CHANGED:        "TAB_CHANGED",
    ACTOR_CHANGED:      "ACTOR_CHANGED",
    REQUEST_REFRESH:    "REQUEST_REFRESH",

    // 既存スキル系
    SKILL_REQUEST_REFRESH: "SKILL_REQUEST_REFRESH",
    SKILL_EQUIP_TOGGLED:   "SKILL_EQUIP_TOGGLED",
    SKILL_FILTER_CHANGED:  "SKILL_FILTER_CHANGED",
    SKILL_SORT_CHANGED:    "SKILL_SORT_CHANGED",

    // ★ 追加：ストーリーゲージ（0〜3段階）更新通知
    // payload: { actorId, gauge, value }  gauge = "courage" | "bond" | "drift" | "arousal"
    STORY_GAUGE_CHANGED: "STORY_GAUGE_CHANGED",
  };

  // -----------------------------
  // 2) 超軽量 EventBus
  // -----------------------------
  const listeners = new Map(); // Map<string, Set<Function>>

  function on(evt, fn){
    if (!evt || typeof fn !== "function") return () => {};
    const set = listeners.get(evt) ?? new Set();
    set.add(fn);
    listeners.set(evt, set);
    // ← 解除関数を返す（重要）
    return () => off(evt, fn);
  }

  function off(evt, fn){
    const set = listeners.get(evt);
    if (!set) return;
    set.delete(fn);
    if (set.size === 0) listeners.delete(evt); // 空なら片付け
  }

  function emit(evt, payload){
    const set = listeners.get(evt);
    if (!set || set.size === 0) return;
    // スナップショットで安全に走らせる
    [...set].forEach(fn => {
      try { fn(payload); }
      catch(e){ console.error(TAG, evt, e); }
    });
  }

  // -----------------------------
  // 3) Scene連携ヘルパ（任意だが推奨）
  // -----------------------------
  // シーンごとに購読→終了時に一括解除でリーク対策
  if (!Scene_Base.prototype.on) {
    Scene_Base.prototype.on = function(evt, fn){
      this.__ero_unsubs ||= [];
      const unsub = on(evt, fn);
      this.__ero_unsubs.push(unsub);
      return unsub;
    };
  }
  if (!Scene_Base.prototype.off)  Scene_Base.prototype.off  = function(evt, fn){ off(evt, fn); };
  if (!Scene_Base.prototype.emit) Scene_Base.prototype.emit = function(evt, p){ emit(evt, p); };

  const _terminate = Scene_Base.prototype.terminate;
  Scene_Base.prototype.terminate = function(){
    if (this.__ero_unsubs) {
      this.__ero_unsubs.forEach(u => { try{ u(); }catch(_){} });
      this.__ero_unsubs = [];
    }
    if (_terminate) _terminate.call(this);
  };

  // -----------------------------
  // 4) 公開
  // -----------------------------
  NS.Events = Events;
  NS.on = on;
  NS.off = off;
  NS.emit = emit;
})();
