(() => {
  const NS = (window.EroStatus ??= {});
  const S  = NS.Store ?? (NS.Store = {});  // 既存Storeに混ぜる

  // ====== Store 標準API（getState / setState / subscribe）======
  // 内部状態（空でも初期化しておく）
  S._state = S._state ?? {};

  // 購読者（UIなど）
  const stateChangeListeners = (S._subs ||= new Set());

  // 購読API：解除関数を返す
  S.subscribe = S.subscribe || function subscribeEroStore(listener) {
    stateChangeListeners.add(listener);
    return () => stateChangeListeners.delete(listener);
  };

  // 変更通知（内部ユーティリティ）
  function notifyChange(eventType, payload) {
    stateChangeListeners.forEach(fn => {
      try { fn({ type: eventType, payload }); } catch (e) { console.warn(e); }
    });
  }

  // 状態取得
  S.getState = S.getState || function getCurrentEroStoreState() {
    return S._state;
  };

  // 状態更新（差分マージ + 通知）
  S.setState = S.setState || function updateEroStoreState(partialState, reason = "UIChanged") {
    S._state = Object.assign({}, S._state, partialState);
    notifyChange(reason, { state: S._state });
    return S._state;
  };

  // ====== flavor 文言ロジック（in-memory 管理）======
  let _flavorRules   = null;
  let _flavorLoaded  = false;
  let _flavorLoading = false;

  // 0..3 → %（矜持0..30、被虐度0..24）
  const stage03ToPercent = v => Math.round(Math.max(0, Math.min(3, Number(v || 0))) * (100 / 3));
  const drift24ToPercent = v => Math.round(Math.max(0, Math.min(24, Number(v || 0))) * (100 / 24)); // 被虐度用：0〜24 → 0〜100%
  const courage30ToPercent = v => Math.round(Math.max(0, Math.min(30, 30 - Number(v || 0))) * (100 / 30)); // 矜持用：0〜30（反転）→ 0〜100%

  // ルール適合
  function ruleMatches(context, rule) {
    const when = rule?.when || {};

    const checkNumericRange = (key) => {
      const cond = when[key]; if (!cond) return true;
      const value = Number(context[key] ?? 0);
      if (typeof cond.min === "number" && value < cond.min) return false;
      if (typeof cond.max === "number" && value > cond.max) return false;
      return true;
    };

    const checkInclusion = (key) => {
      const cond = when[key]; if (!cond || !Array.isArray(cond.in)) return true;
      return cond.in.includes(context[key]);
    };

    const checkBadStatuses = () => {
      const required = when.requireBadStatus;
      if (!Array.isArray(required) || required.length === 0) return true;
      const actor = $gameActors.actor(context.actorId);
      return required.every(k => actor?.hasPersistentBadStatus?.(k));
    };

    return (
      checkNumericRange("corruptionPct") &&
      checkNumericRange("devMaxPct") &&
      checkNumericRange("pridePct") &&
      checkNumericRange("choroPct") &&
      checkInclusion("topGaugeKey") &&
      checkBadStatuses()
    );
  }

  // コンテキスト生成（Actor一元化）
  function buildFlavorContext(actorId) {
    const actor = $gameActors.actor(actorId);
    const storyGauges = S.getStoryGauges?.(actorId) ?? { courage: 0, bond: 0, drift: 0, arousal: 0 };
    const pridePct     = courage30ToPercent(storyGauges.courage); // 矜持は0〜30基準（反転）
    const choroPct     = drift24ToPercent(storyGauges.drift); // 被虐度は0〜24基準
    const devMaxPct    = S.getDevelopmentMaxPercent?.(actorId) ?? 0;
    const corruptionPct = Math.max(0, Math.min(100, Number(actor?.getCorruption?.() ?? 0)));
    const topGaugeKey   = S.getTopGaugeKey?.(actorId) ?? "courage";
    return { actorId, pridePct, choroPct, devMaxPct, corruptionPct, topGaugeKey };
  }

  // JSONロード（成功・失敗をログ）
  async function loadFlavorRules(url = "dataEx/flavor/flavorRules.json") {
    if (_flavorLoading || _flavorLoaded) {
      console.log("[EroStatus.Store] flavor: skip load (loading/loaded)");
      return;
    }
    _flavorLoading = true;
    console.log("[EroStatus.Store] flavor: loading...", url);
    try {
      const res = await fetch(url);
      if (!res.ok) throw new Error("HTTP " + res.status);
      const json = await res.json();
      _flavorRules = (Array.isArray(json) ? json : []).slice()
        .sort((a, b) => (a.priority ?? 9999) - (b.priority ?? 9999));
      console.log("[EroStatus.Store] flavor: loaded rules =", _flavorRules.length);
    } catch (e) {
      console.warn("[EroStatus.Store] flavor: load failed -> fallback", e);
      _flavorRules = null;
    } finally {
      _flavorLoaded  = true;
      _flavorLoading = false;
      // UIへ通知
      try { notifyChange("FlavorRulesLoaded", { count: _flavorRules?.length | 0 }); } catch (_) {}
    }
  }

  // テスト/ホットリロード用
  function setFlavorRules(rules) {
    if (Array.isArray(rules)) {
      _flavorRules = rules.slice().sort((a, b) => (a.priority ?? 9999) - (b.priority ?? 9999));
      _flavorLoaded = true;
      console.log("[EroStatus.Store] flavor: set rules =", _flavorRules.length);
      try { notifyChange("FlavorRulesLoaded", { count: _flavorRules.length }); } catch (_) {}
    } else {
      console.warn("[EroStatus.Store] flavor: invalid setFlavorRules argument");
    }
  }

  // 実際のテキスト取得（未ロードなら遅延ロードをキック）
  function getFlavorText(actorId) {
    if (!_flavorLoaded && !_flavorLoading) {
      // 初回アクセス時にロード開始（非同期）
      loadFlavorRules("dataEx/flavor/flavorRules.json").catch(() => {});
    }
    const ctx = buildFlavorContext(actorId);
    if (!_flavorRules || _flavorRules.length === 0) {
      return "この街では誰もが"; // フォールバック
    }
    const matched = _flavorRules.find(r => {
      try { return ruleMatches(ctx, r); } catch { return false; }
    });
    return matched?.text || "この街では誰もが";
  }

  // ====== Store公開API ======
  S.loadFlavorRules = loadFlavorRules;
  S.setFlavorRules  = setFlavorRules;
  S.getFlavorText   = getFlavorText;

  // デバッグ用
  S._debugFlavor = () => ({
    loaded:  _flavorLoaded,
    loading: _flavorLoading,
    count:   _flavorRules?.length | 0
  });
})();
