/*:
 * @target MZ
 * @plugindesc EroStatusUI（表＋裏の同時表示・左右2カラム、トグル廃止） v3.0
 * @author you
 *
 * @help
 * 左：表（ストーリー数値）＋裏（開発度数値）を上下で同時表示
 * 右：立ち絵（busts）
 *
 * - 立ち絵名は 変数ID(デフォルト:13) → EroStatus.Store → "soph_1" の順で決定
 * - メニューから「エロステータス」で起動
 */

(() => {
  // ==== 設定 ====
  // ← ファイル先頭近くの UIConfig に追記
    const UIConfig = {
    // ★必須（左カラム幅%）
    leftColumnWidthPercent: 56,

    // ★立ち絵関連（参照しているなら入れておく）
    bustFitMode: "fit-height",         // "fit-height" | "fit-width"
    bustNameVariableId: 13,            // 立ち絵名を入れている変数ID
    bustPicturesSubdir: "busts",       // img/pictures/ 配下のサブフォルダ
    // ★ 追加：固定座標
    bustFixedX: 675,
    bustFixedY: 8,

    // セクション色
    sectionTitleColor: () => ColorManager.systemColor(),

    // レイアウト
    contentPaddingX: 18,
    contentStartY: 12,
    lineGap: 4,
    sectionGap: 10,
    labelColumnWidth: 160,
    valueAlign: "right",
    titleAlign: "center",
    fontSizeScale: 1.0,
  };
  // === ヘルパー：img/pictures 配下の相対サブフォルダを正規化 ===
function normalizePicturesSubdir(subdir) {
  // 空 → "." を返して「直下」を意味させる
  let s = String(subdir || "").trim();
  if (!s || s === "." || s === "./") return ".";

  // 先頭に "img/pictures" が付いていたら削る（loadPictureは自動で付けるため）
  s = s.replace(/^(\.\/)?img\/pictures\/?/i, "");

  // 先頭/末尾のスラッシュを削除
  s = s.replace(/^\/+/, "").replace(/\/+$/, "");

  // 何も残らなければ直下
  return s.length ? s : ".";
}


  // ==== 右：立ち絵 ====
  class Sprite_EroBust extends Sprite {
    constructor() {
      super();
      this._fit = UIConfig.bustFitMode;
    }
    setPicture(folder, name) {
      const subdir = normalizePicturesSubdir(folder);
      const baseName = String(name || "").trim();
      if (!baseName) {
        console.warn("[EroStatusUI] bust name empty. Skip draw.");
        this.bitmap = null;
        return;
      }
      const path = subdir === "." ? baseName : `${subdir}/${baseName}`;
      this.bitmap = ImageManager.loadPicture(path);
      this.bitmap.addLoadListener(() => this._fitToRightBox());
    }
    _fitToRightBox() {
      if (!this.bitmap) return;
      const bw = this.bitmap.width, bh = this.bitmap.height;
      const boxW = Graphics.boxWidth, boxH = Graphics.boxHeight;
      const leftW = Math.floor(boxW * (UIConfig.leftColumnWidthPercent / 100));
      const availW = boxW - leftW;
      const availH = boxH;
      let scale = 1;
      if (this._fit === "fit-height") scale = availH / bh;
      else scale = availW / bw;
      this.scale.set(scale, scale);
      const fx = UIConfig.bustFixedX;
      const fy = UIConfig.bustFixedY;
      if (Number.isFinite(fx)) this.x = Math.floor(fx);
      if (Number.isFinite(fy)) this.y = Math.floor(fy);
      this.opacity = 255;
    }
  }

  class Window_EroUnifiedPanel extends Window_Base {
  initialize(rect) {
    super.initialize(rect);
    // Storeの変更通知で自動リフレッシュ
    this._unsubscribe = window.EroStatus?.Store?.subscribe?.((evt) => {
  if (!evt) return;
  if (evt.type === "DevelopmentChanged" || evt.type === "StoryGaugesChanged" ||
      evt.type === "UIChanged" || evt.type === "FlavorRulesLoaded") {
    this.refresh();
  }
});
  }

  destroy() {
    super.destroy();
    this._unsubscribe?.();
  }

  // Window_EroUnifiedPanel 内の refresh を差し替え
refresh() {
  this.contents.clear();

  // ===== 固定レイアウト =====
  const labelX     = UIConfig.contentPaddingX ?? 18;
  const valueRight = this.contentsWidth() - (UIConfig.contentPaddingX ?? 18);
  const startY     = UIConfig.contentStartY ?? 12;
  const lineH      = this.lineHeight();
  const rowGap     = UIConfig.lineGap ?? 4;

  // 表：内部キーは courage / drift を採用（表示ラベルは矜持/被虐度）
  const storyOrder = ["courage", "drift"];
  const storyLabel = { courage: "矜持", drift: "被虐度" };

  // ===== データ取得 =====
  const actor = $gameParty.menuActor() || $gameParty.leader() || $gameActors.actor(1);
  if (!actor) return;

  const store   = window.EroStatus?.Store;
  const actorId = store?.getState?.().actorId || actor.actorId?.() || 1;
  // StoryGaugesはactorから直接取得（Store経由の実装がない場合は直接アクセス）
  const storyStage03 = (typeof actor.getStoryGauges === "function") 
    ? actor.getStoryGauges() 
    : (store?.getStoryGauges?.(actorId) || { courage:0, bond:0, drift:0, arousal:0 });

  // 0..3 → 0..100% 換算（矜持0..30、被虐度0..24）
  const stageToPct = v => Math.round(Math.max(0, Math.min(3, Number(v||0))) * (100/3));
  const driftToPct = v => Math.round(Math.max(0, Math.min(24, Number(v||0))) * (100/24)); // 被虐度用：0〜24 → 0〜100%
  const courageToPct = v => Math.round(Math.max(0, Math.min(30, 30 - Number(v||0))) * (100/30)); // 矜持用：0〜30（反転）→ 0〜100%

  // --- 表見出し ---
  this.changeTextColor(ColorManager.systemColor());
  this.drawText("セリアの評判", labelX, startY, 300, "left");
  this.resetTextColor();

  let currentY = startY + lineH;

  // --- 表：行描画 ---
  for (const key of storyOrder) {
    const label   = storyLabel[key] || key;
    // 矜持は減点式（値が低いほど良い状態 = 高いパーセンテージ）
    let percent;
    if (key === "courage") {
      // 矜持は0〜30の内部値を反転してパーセンテージに変換（0→100%, 30→0%）
      percent = courageToPct(storyStage03[key]);
    } else if (key === "drift") {
      // 被虐度は0〜24の内部値を直接パーセンテージに変換
      percent = driftToPct(storyStage03[key]);
    } else {
      percent = stageToPct(storyStage03[key]);
    }

    // 肩書き化
    const describe = (k, p) => {
      const table = {
        courage: [
          [80,"誇り高き冒険者"],
          [60,"普通の冒険者"],
          [40,"自信喪失気味"],
          [10,"折れかけの心"],
          [0,"砕けた誇り"]
        ],
        drift:   [[80,"超ちょろい"],[60,"ちょろい"],[40,"流されやすい"],[20,"慎重"],[0,"普通"]],
      }[k];
      if (!table) return "状態不明";
      for (const [th, text] of table) if (p >= th) return text;
      return table[table.length-1][1];
    };

    this.drawText(label, labelX, currentY, UIConfig.labelColumnWidth ?? 160, "left");
    this.drawText(describe(key, percent), labelX, currentY, valueRight - labelX, "right");
    currentY += lineH + rowGap;
  }

  // ===== ここから【堕落度（1行）】を挿入 =====
  currentY += (UIConfig.sectionGap ?? 10); // 評判 → 堕落度 の間隔

  const corruptionValue = Math.max(0, Math.min(100, Number(actor.getCorruption?.() ?? 0)));
  const corruptionStageLabel = (v => (v >= 80 ? "高" : v >= 40 ? "中" : "低"))(corruptionValue);

  this.changeTextColor(ColorManager.systemColor());
  this.drawText("堕落度", labelX, currentY, UIConfig.labelColumnWidth ?? 160, "left");
  this.resetTextColor();
  this.drawText(`${corruptionValue|0}%（${corruptionStageLabel}）`, labelX, currentY, valueRight - labelX, "right");
  currentY += lineH + rowGap;

  currentY += (UIConfig.sectionGap ?? 10); // 堕落度 → 開発度 の間隔
  // ===== ここまで【堕落度】 =====

  // --- 見出し（裏：開発度） ---
  this.changeTextColor(ColorManager.textColor(2)); // ピンク寄り
  this.drawText("開発度", labelX, currentY, 300, "left");
  this.resetTextColor();
  currentY += lineH;

  // 絶頂回数
  const totalOrgasmCount = Number(actor.getTotalOrgasmCount?.() ?? 0);
  this.drawText("絶頂回数", labelX, currentY, UIConfig.labelColumnWidth ?? 160, "left");
  this.drawText(`${totalOrgasmCount}回`, labelX, currentY, valueRight - labelX, "right");
  currentY += lineH + rowGap;

  // 挿入回数
  const totalInsertCount = Number(actor.getTotalInsertCount?.() ?? 0);
  this.drawText("挿入回数", labelX, currentY, UIConfig.labelColumnWidth ?? 160, "left");
  this.drawText(`${totalInsertCount}回`, labelX, currentY, valueRight - labelX, "right");
  currentY += lineH + rowGap;

  // 開発度（代表として）
  const pussyDev = Number(actor.getDevelopment?.("pussy") ?? 0);
  this.drawText("開発度", labelX, currentY, UIConfig.labelColumnWidth ?? 160, "left");
  this.drawText(`${pussyDev|0}/100`, labelX, currentY, valueRight - labelX, "right");
  currentY += lineH + rowGap;

  // --- 見出し：フレーバー ---
  this.changeTextColor(ColorManager.systemColor());
  this.drawText("心身の記録", labelX, currentY, this.contentsWidth() - labelX*2, "left");
  this.resetTextColor();
  currentY += lineH;

  const flavorText = store?.getFlavorText?.(actorId) || "";
  const rect = new Rectangle(labelX, currentY, this.contentsWidth() - labelX*2, this.contentsHeight() - currentY);
  this.drawTextEx(flavorText, rect.x, rect.y);
} // ← refresh 終わり

  }

  // ==== シーン本体 ====
  class Scene_EroStatus extends Scene_MenuBase {
    create() {
      super.create();

      const leftWidth = Math.floor(Graphics.boxWidth * (UIConfig.leftColumnWidthPercent / 100));
      const leftRect = new Rectangle(0, 0, leftWidth, Graphics.boxHeight);

      // 左：統合ウィンドウ（表＋裏）
      this._unifiedPanelWindow = new Window_EroUnifiedPanel(leftRect);
      this.addWindow(this._unifiedPanelWindow);

      // 右：立ち絵
      this._bustSprite = new Sprite_EroBust();
      this.addChild(this._bustSprite);
      this._refreshBustOnce();
      this._bustSprite.x += 24;   // 右へ +24px
      this._bustSprite.y -= 12;   // 上へ -12px
    }

    start() {
      super.start();
        try { window.EroStatus?.Store?.loadFlavorRules?.("dataEx/flavor/flavorRules.json"); } catch(_) {}
  this._unifiedPanelWindow.refresh();
      this._unifiedPanelWindow.refresh();
    }

    update() {
      super.update();
      // ESC / 右クリックで閉じる
      if (Input.isTriggered("cancel") || TouchInput.isCancelled()) {
        this.popScene();
      }
    }

    _refreshBustOnce() {
      // 1) 変数 → 2) Store → 3) "soph_1"
      const varName = String($gameVariables.value(UIConfig.bustNameVariableId) || "").trim();
      const stateName = String(window.EroStatus?.Store?.getState()?.raw?.actor?.bustName || "").trim();
      const finalName = varName || stateName || "soph_1";
      const folder = normalizePicturesSubdir(UIConfig.bustPicturesSubdir || "busts");
      this._bustSprite.setPicture(folder, finalName);
    }
  }
  // 公開
  window.Scene_EroStatus = Scene_EroStatus;

  // ==== メニュー常設 ====
  const MENU_NAME = "エロステータス";
  if (Window_MenuCommand && !Window_MenuCommand.prototype.__EroMenuInjectedUnified) {
    Window_MenuCommand.prototype.__EroMenuInjectedUnified = true;

    const _addOriginalCommands = Window_MenuCommand.prototype.addOriginalCommands;
    Window_MenuCommand.prototype.addOriginalCommands = function () {
      _addOriginalCommands.call(this);
      this.addCommand(MENU_NAME, "EroStatus", true);
    };

    const _createCommandWindow = Scene_Menu.prototype.createCommandWindow;
    Scene_Menu.prototype.createCommandWindow = function () {
      _createCommandWindow.call(this);
      this._commandWindow.setHandler("EroStatus", () => SceneManager.push(Scene_EroStatus));
    };
  }
})();
