// ExpressionSelector.js - 表情自動選択システム
(() => {
  console.log("[ExpressionSelector] Loading...");

  const DEBUG_MODE = false;

  // =============================================================================
  // データ定義
  // =============================================================================
  
  // 状況別の表情出現確率（%）
  // 基本表情: endure（耐える）/ confused（混乱）/ ahe（快楽に屈する）
  const EXPRESSION_WEIGHTS = {
    stage1: {
      standby: { endure: 60, confused: 35, ahe: 5,  normal: 0 },
      react:   { endure: 60, confused: 35, ahe: 5,  normal: 0 },
      danger:  { endure: 70, confused: 25, ahe: 5,  normal: 0 },
      climax:  { endure: 40, confused: 45, ahe: 15, normal: 0 },
      down:    { endure: 50, confused: 40, ahe: 10, normal: 0 },
      down_attack: { endure: 55, confused: 35, ahe: 10, normal: 0 },
      down_attack_pre: { endure: 55, confused: 40, ahe: 5, normal: 0 }
    },
    stage2: {
      standby: { endure: 40, confused: 45, ahe: 15, normal: 0 },
      react:   { endure: 45, confused: 40, ahe: 15, normal: 0 },
      danger:  { endure: 55, confused: 30, ahe: 15, normal: 0 },
      climax:  { endure: 25, confused: 40, ahe: 35, normal: 0 },
      down:    { endure: 35, confused: 45, ahe: 20, normal: 0 },
      down_attack: { endure: 40, confused: 35, ahe: 25, normal: 0 },
      down_attack_pre: { endure: 45, confused: 40, ahe: 15, normal: 0 }
    },
    stage3: {
      standby: { endure: 20, confused: 40, ahe: 40, normal: 0 },
      react:   { endure: 25, confused: 35, ahe: 40, normal: 0 },
      danger:  { endure: 30, confused: 30, ahe: 40, normal: 0 },
      climax:  { endure: 10, confused: 30, ahe: 60, normal: 0 },
      down:    { endure: 15, confused: 35, ahe: 50, normal: 0 },
      down_attack: { endure: 20, confused: 30, ahe: 50, normal: 0 },
      down_attack_pre: { endure: 25, confused: 35, ahe: 40, normal: 0 }
    }
  };

  // 表情遷移ルール（前の表情に基づく次の表情の傾向）
  const EXPRESSION_TRANSITIONS = {
    endure: {
      endure: 40,
      confused: 50,
      ahe: 10
    },
    confused: {
      endure: 20,
      confused: 50,
      ahe: 30
    },
    ahe: {
      endure: 5,
      confused: 25,
      ahe: 70
    }
  };

  // =============================================================================
  // 内部ヘルパー関数
  // =============================================================================
  
  /**
   * 重み付きランダム選択
   * @param {Object} weights - { key: weight, ... }
   * @returns {string} - 選択されたキー
   */
  function weightedRandom(weights) {
    const total = Object.values(weights).reduce((a, b) => a + b, 0);
    if (total <= 0) {
      if (DEBUG_MODE) console.log("[ExpressionSelector] Total weight 0. Fallback to 'endure'.");
      return "endure";
    }
    
    let random = Math.random() * total;
    let selected = null;
    
    for (const [key, weight] of Object.entries(weights)) {
      random -= weight;
      if (random <= 0) {
        selected = key;
        break;
      }
    }
    
    if (!selected) selected = Object.keys(weights)[0] || "endure";

    console.log(`[ExpressionSelector] Selected: "${selected}" (weights: ${JSON.stringify(weights)})`);

    return selected;
  }

  /**
   * 台詞解析：感情傾向の抽出
   * @param {string} text - 台詞テキスト
   * @returns {Object|null} - { resist, confused, pleasure } または null
   */
  function analyzeDialogueEmotion(text) {
    if (!text) return null;
    
    const hints = {
      resist: 0,    // 抵抗・耐える系
      confused: 0,  // 混乱・戸惑い系
      pleasure: 0   // 快楽・堕落系
    };
    
    // === 抵抗・耐える系キーワード ===
    if (/やめ|止め|だめ|いや|やだ|離して|許さ|くっ/.test(text)) hints.resist += 40;
    if (/痛|怖|無理|嫌/.test(text)) hints.resist += 30;
    if (/まだ|耐え|我慢/.test(text)) hints.resist += 35;
    
    // === 混乱・戸惑い系キーワード ===
    if (/なんで|どうして|おかしい|変|わからない|うそ/.test(text)) hints.confused += 50;
    if (/[あぁ][っ…]|んっ|はぁ/.test(text)) hints.confused += 15;
    if (/来[るちそ]|なっちゃ|されちゃ/.test(text)) hints.confused += 20;
    
    // === 快楽・堕落系キーワード ===
    const heartCount = (text.match(/♡/g) || []).length;
    hints.pleasure += 25 * heartCount; // ♡1つで25pt
    
    if (/気持ち|いい|すごい|もっと/.test(text)) hints.pleasure += 40;
    if (/イ[くっ]|イっちゃ|絶頂/.test(text)) hints.pleasure += 60;
    
    // 全てゼロなら解析不能
    const total = hints.resist + hints.confused + hints.pleasure;
    if (total === 0) return null;
    
    return hints;
  }

  /**
   * 堕落度 + イベント補正 + 台詞解析による表情ウェイト計算
   * @param {string} stage - "stage1", "stage2", "stage3"
   * @param {string} eventType - "danger", "climax", "insert", etc.
   * @param {string} dialogueText - 台詞テキスト（オプション）
   * @param {boolean} isRestrained - 拘束状態かどうか
   * @returns {Object} - { endure, confused, ahe, normal, embarrassed }
   */
  function getExpressionWeightsAdvanced(stage, eventType, dialogueText = null, isRestrained = false) {
    // 1. 非拘束（立ち絵）モードの場合の特別処理
    //    使用可能表情: normal のみ（embarrassed は画像がないモンスターが多いため無効化）
    if (!isRestrained) {
      // embarrassed を完全に無効化し、normal のみ使用
      return { 
        normal: 100, 
        embarrassed: 0,
        endure: 0, confused: 0, ahe: 0 
      };
    }

    // 2. 拘束モード（既存ロジック）
    // 拘束絵には embarrassed が存在しないため、ウェイトをゼロにする
    let weights = { endure: 0, confused: 0, ahe: 0, normal: 0, embarrassed: 0 };
    
    if (stage === "stage1") {
      weights = { endure: 60, confused: 30, ahe: 5, normal: 5 };
    } else if (stage === "stage2") {
      weights = { endure: 30, confused: 50, ahe: 20, normal: 0 };
    } else { // stage3
      weights = { endure: 10, confused: 30, ahe: 60, normal: 0 };
    }
    
    // 2. イベント種別による補正
    if (eventType === "insert") {
      // 挿入時: 混乱が増える（異物感・驚き）
      weights.confused += 30;
      weights.endure += 10;
      weights.ahe *= 0.8;
    } else if (eventType === "orgasm") {
      // 絶頂時: stage依存で表情が変わる
      if (stage === "stage1") {
        weights.endure += 40;   // 耐え顔絶頂
        weights.confused += 30;
        weights.ahe -= 3;
      } else if (stage === "stage2") {
        weights.confused += 40; // 混乱絶頂
        weights.ahe += 30;
      } else {
        weights.ahe += 60;      // アヘ絶頂
      }
    } else if (eventType === "danger") {
      weights.endure += 30;
      weights.confused += 10;
    } else if (eventType === "climax") {
      weights.ahe += 20;
      weights.confused += 10;
    }
    
    // 3. 台詞解析による補正（あれば）
    const textHint = analyzeDialogueEmotion(dialogueText);
    if (textHint) {
      if (textHint.resist > 0) {
        weights.endure += textHint.resist;
        
        // 裏腹係数（Stage3の場合、拒絶セリフでも快楽顔が出る）
        if (stage === "stage3") {
          weights.ahe += textHint.resist * 0.6;  // 抵抗の60%が快楽に流れる
          weights.confused += textHint.resist * 0.3;
          weights.endure *= 0.5; // 耐える顔が出にくくなる
        }
      }
      
      if (textHint.confused > 0) {
        weights.confused += textHint.confused;
      }
      
      if (textHint.pleasure > 0) {
        weights.ahe += textHint.pleasure;
        weights.confused += textHint.pleasure * 0.3;
        weights.endure -= textHint.pleasure * 0.5;
      }
    }
    
    // 4. 負の値補正
    for (const key of Object.keys(weights)) {
      if (weights[key] < 0) weights[key] = 0;
    }
    
    return weights;
  }

  // =============================================================================
  // 公開API
  // =============================================================================
  
  /**
   * 重み付きランダムで表情を選択
   * @param {string} stage - "stage1", "stage2", "stage3"
   * @param {string} eventType - イベントタイプ
   * @param {string} dialogueText - 台詞テキスト（オプション）
   * @param {boolean} isRestrained - 拘束状態かどうか
   * @returns {string} - 選択された表情
   */
  function weightedRandomExpression(stage, eventType, dialogueText = null, isRestrained = false) {
    const weights = getExpressionWeightsAdvanced(stage, eventType, dialogueText, isRestrained);
    return weightedRandom(weights);
  }

  /**
   * 前の表情に基づいて次の表情を選択（遷移ベース）
   * @param {string} currentExpression - 現在の表情
   * @param {string} stage - "stage1", "stage2", "stage3"
   * @param {string} eventType - イベントタイプ
   * @returns {string} - 次の表情
   */
  function getNextExpression(currentExpression, stage, eventType) {
    // 遷移ルールがある表情のみ適用
    if (!EXPRESSION_TRANSITIONS[currentExpression]) {
      return weightedRandomExpression(stage, eventType);
    }
    
    // 基本遷移ウェイトを取得
    const transitionWeights = { ...EXPRESSION_TRANSITIONS[currentExpression] };
    
    // ステージによる補正
    if (stage === "stage3") {
      transitionWeights.ahe *= 1.5;
      transitionWeights.endure *= 0.5;
    } else if (stage === "stage1") {
      transitionWeights.endure *= 1.3;
      transitionWeights.ahe *= 0.7;
    }
    
    return weightedRandom(transitionWeights);
  }

  // =============================================================================
  // グローバルに公開
  // =============================================================================
  
  window.ExpressionSelector = {
    // データ
    EXPRESSION_WEIGHTS,
    EXPRESSION_TRANSITIONS,
    
    // 公開関数
    weightedRandomExpression,
    getNextExpression,
    
    // 内部関数（デバッグ・テスト用）
    _internal: {
      weightedRandom,
      analyzeDialogueEmotion,
      getExpressionWeightsAdvanced
    }
  };

  console.log("[ExpressionSelector] Ready");
})();

