/*:
 * @target MZ
 * @plugindesc v1.0 マップ右下の常時立ち絵オーバーレイ（TRP_Skit併用可 / 右下固定 / 自動スケール）
 * @author you
 *
 * @help
 * ■概要
 *  マップ画面に「img/pictures/<Folder>/＜name＞.png」を右下固定で表示します。
 *  ・メッセージ表示中は自動で隠す（設定で無効化可）
 *  ・画面サイズ変更に追従（右下に再レイアウト）
 *  ・TRP_Skit 再生中は自動非表示
 *  ・ゲームからは MapBust API かプラグインコマンドで操作
 *
 * ■画像の置き場所
 *  img/pictures/<Folder>/ 直下。拡張子は不要で、コマンドにはファイル名のみを渡します。
 *  例）Folderが "busts"、nameが "uniform_intact_base"
 *      → img/pictures/busts/uniform_intact_base.png
 *
 * ■プラグインコマンド
 *  - SetBust     : 表示する画像名（拡張子なし）を指定
 *  - ClearBust   : 画像をクリア
 *  - SetVisible  : 表示/非表示の切り替え
 *  - SetOpacity  : 不透明度（0～255）
 *
 * ■スクリプトAPI（イベントの「スクリプト」等から）
 *   MapBust.setBust("uniform_intact_base");
 *   MapBust.clear();
 *   MapBust.setVisible(true);
 *   MapBust.setOpacity(200);
 *   MapBust.refresh(); // 手動再レイアウト
 *
 * @param EnabledSwitchId
 * @type switch
 * @text 有効化スイッチ
 * @desc このスイッチがONのときだけ表示（0 で常時有効）
 * @default 0
 *
 * @param AutoHideDuringMessage
 * @type boolean
 * @text メッセージ中は隠す
 * @desc $gameMessage が表示中は自動的に非表示にします
 * @default true
 *
 * @param Folder
 * @type string
 * @text 画像フォルダ（pictures配下）
 * @desc img/pictures/<Folder>/ に画像を置く
 * @default busts
 *
 * @param Align
 * @type select
 * @option right-bottom
 * @option right-top
 * @option left-bottom
 * @option left-top
 * @text 基準位置
 * @default right-bottom
 *
 * @param MarginX
 * @type number
 * @min 0
 * @text 余白X
 * @default 16
 *
 * @param MarginY
 * @type number
 * @min 0
 * @text 余白Y
 * @default 8
 *
 * @param FitMode
 * @type select
 * @option fit-height
 * @option fit-width
 * @option fixed
 * @text フィット方式
 * @desc fit-height: 画面高さに収める / fit-width: 幅に収める / fixed: 固定スケール
 * @default fit-height
 *
 * @param Scale
 * @type number
 * @decimals 2
 * @text 固定スケール（fixed時）
 * @default 1.0
 *
 * @param MaxScale
 * @type number
 * @decimals 2
 * @text 上限スケール
 * @default 1.0
 *
 * @param Opacity
 * @type number
 * @min 0
 * @max 255
 * @text 既定の不透明度
 * @default 255
 *
 * @param UpdateInterval
 * @type number
 * @min 1
 * @text 更新間隔（フレーム）
 * @desc 表示状態の再チェック間隔。重い場合は増やす
 * @default 10
 *
 * @command SetBust
 * @text 画像セット
 * @arg name
 * @type string
 * @text 画像名（拡張子なし）
 * @desc 例: uniform_intact_base
 *
 * @command ClearBust
 * @text 画像クリア
 *
 * @command SetVisible
 * @text 表示/非表示
 * @arg visible
 * @type boolean
 * @default true
 *
 * @command SetOpacity
 * @text 不透明度
 * @arg opacity
 * @type number
 * @min 0
 * @max 255
 * @default 255
 */

(() => {
  const PLUGIN_NAME = "MapBustOverlay";

  const params = PluginManager.parameters(PLUGIN_NAME);
  const EnabledSwitchId       = Number(params.EnabledSwitchId || 0);
  const AutoHideDuringMessage = String(params.AutoHideDuringMessage || "true") === "true";
  const Folder                = String(params.Folder || "busts").replace(/^\/|\/$/g, "");
  const Align                 = String(params.Align || "right-bottom");
  const MarginX               = Number(params.MarginX || 16);
  const MarginY               = Number(params.MarginY || 8);
  const FitMode               = String(params.FitMode || "fit-height");
  const ScaleFixed            = Number(params.Scale || 1.0);
  const MaxScale              = Number(params.MaxScale || 1.0);
  const DefaultOpacity        = Number(params.Opacity || 255);
  const UpdateInterval        = Math.max(1, Number(params.UpdateInterval || 10));

  // ---- 状態（セーブ可：$gameSystem に保存）----
  const SystemKey = "_mapBustOverlay";
  function ensureState() {
    $gameSystem[SystemKey] = $gameSystem[SystemKey] || {
      name: "",           // 画像ファイル名（拡張子なし）
      visible: true,      // 論理表示
      opacity: DefaultOpacity
    };
    return $gameSystem[SystemKey];
  }

  // ---- 画像ロード（pictures配下）----
  function loadBustBitmap(name) {
    if (!name) return null;
    const path = (Folder ? Folder + "/" : "") + name;
    return ImageManager.loadBitmap("img/pictures/", path);
  }

  // ---- グローバルAPI ----
  window.MapBust = {
    setBust(name) { ensureState().name = name || ""; },
    clear() { ensureState().name = ""; },
    setVisible(v) { ensureState().visible = !!v; },
    setOpacity(o) { ensureState().opacity = Number(o).clamp(0,255); },
    refresh() { SceneManager._scene?._mapBustOverlay?.refresh(true); }
  };

  // ---- プラグインコマンド ----
  PluginManager.registerCommand(PLUGIN_NAME, "SetBust", args => {
    MapBust.setBust(String(args.name || ""));
    MapBust.refresh();
  });
  PluginManager.registerCommand(PLUGIN_NAME, "ClearBust", () => {
    MapBust.clear();
    MapBust.refresh();
  });
  PluginManager.registerCommand(PLUGIN_NAME, "SetVisible", args => {
    MapBust.setVisible(String(args.visible) === "true");
    MapBust.refresh();
  });
  PluginManager.registerCommand(PLUGIN_NAME, "SetOpacity", args => {
    MapBust.setOpacity(Number(args.opacity || 255));
    MapBust.refresh();
  });

  // ---- 表示本体スプライト ----
  class Sprite_MapBustOverlay extends Sprite {
    constructor() {
      super();
      this._bitmapName = "";
      this._lastTick = 0;
      this._sprite = new Sprite();
      // 右下基準で扱う（座標で吸収）
      this._sprite.anchor.set(1,1);
      this.addChild(this._sprite);
      this.visible = false;
    }

    shouldDisplay() {
      const s = ensureState();
      if (!s.visible) return false;
      if (EnabledSwitchId > 0 && !$gameSwitches.value(EnabledSwitchId)) return false;
      if (AutoHideDuringMessage && $gameMessage.isBusy()) return false;
      if (window.TRP_Skit && TRP_Skit.isPlaying && TRP_Skit.isPlaying()) return false;
      if (!s.name) return false;
      return true;
    }

    currentBitmapReady() {
      return !!(this._sprite.bitmap && this._sprite.bitmap.isReady());
    }

    update() {
      super.update();
      if (Graphics.frameCount - this._lastTick < UpdateInterval) return;
      this._lastTick = Graphics.frameCount;
      this.refresh();
    }

    refresh(force = false) {
      const s = ensureState();
      const wantVisible = this.shouldDisplay();
      this.visible = wantVisible;

      if (!wantVisible) return;

      // 画像差し替え
      if (force || this._bitmapName !== s.name) {
        this._bitmapName = s.name;
        this._sprite.bitmap = loadBustBitmap(this._bitmapName);
        this._sprite.opacity = s.opacity;
        if (this._sprite.bitmap) {
          this._sprite.bitmap.addLoadListener(() => this._applyLayout());
        }
      } else if (this.currentBitmapReady()) {
        this._sprite.opacity = s.opacity;
        this._applyLayout();
      }
    }

    _applyLayout() {
      const spr = this._sprite;
      const bm  = spr.bitmap;
      if (!bm || !bm.isReady()) return;

      const gw = Graphics.boxWidth;
      const gh = Graphics.boxHeight;

      // 表示可能領域（マージン考慮）
      const availW = Math.max(1, gw - MarginX * 2);
      const availH = Math.max(1, gh - MarginY * 2);

      // スケール
      let scale = 1.0;
      if (FitMode === "fit-height") {
        scale = availH / bm.height;
      } else if (FitMode === "fit-width") {
        scale = availW / bm.width;
      } else { // fixed
        scale = ScaleFixed;
      }
      scale = Math.min(scale, MaxScale);
      spr.scale.set(scale, scale);

      // 右下/右上/左下/左上
      let baseX = 0, baseY = 0;
      if (Align.includes("right")) baseX = gw - MarginX;
      else                         baseX = MarginX + spr.width * spr.scale.x;

      if (Align.includes("bottom")) baseY = gh - MarginY;
      else                           baseY = MarginY + spr.height * spr.scale.y;

      spr.x = Math.floor(baseX);
      spr.y = Math.floor(baseY);
    }
  }

  // ---- Scene_Map：生成・更新・破棄・リサイズ追従 ----
  const _Scene_Map_createAllWindows = Scene_Map.prototype.createAllWindows;
  Scene_Map.prototype.createAllWindows = function() {
    _Scene_Map_createAllWindows.call(this);
    this._mapBustOverlay = new Sprite_MapBustOverlay();
    // ウィンドウ層の「下」に置く（会話ウィンドウや立ち絵と干渉しにくい）
    this.addChildAt(this._mapBustOverlay, this.children.indexOf(this._windowLayer));
  };

  const _Scene_Map_update = Scene_Map.prototype.update;
  Scene_Map.prototype.update = function() {
    _Scene_Map_update.call(this);
    this._mapBustOverlay?.update();
  };

  const _Scene_Map_terminate = Scene_Map.prototype.terminate;
  Scene_Map.prototype.terminate = function() {
    _Scene_Map_terminate.call(this);
    this._mapBustOverlay = null;
  };

  const _Scene_Map_onResize = Scene_Map.prototype.onResize;
  Scene_Map.prototype.onResize = function() {
    _Scene_Map_onResize.call(this);
    this._mapBustOverlay?._applyLayout?.();
  };

})();
