/*:
 * @target MZ
 * @plugindesc モンスター特性管理システム v1.0
 * @author ProjectEst
 * 
 * @help
 * モンスターごとの特殊な振る舞いを定義します。
 * データは dataEx/monsterTraits.json で管理します。
 * 
 * ■特性一覧
 * - allowStage1Insert: stage1から挿入スキルを使用可能にする
 * 
 * ■次回作への移植手順
 * 1. このプラグインファイルをコピー
 * 2. monsterTraits.json をコピー
 * 3. JSONファイルのモンスター名を変更
 * 完了！
 * 
 * ■敵キャラの設定
 * データベース→敵キャラ→メモ欄に以下を記述（推奨）:
 *   <MonsterKey:slime>
 *   <MonsterKey:goblin>
 *   <MonsterKey:tentacle>
 * 
 * メモ欄に記述がない場合、名前から自動推測します。
 */

(() => {
  console.log("[MonsterTraitManager] v1.0 Loading...");

  // ============================================================================
  // データロード
  // ============================================================================
  
  let MONSTER_TRAITS = null;
  
  async function loadTraits() {
    try {
      const response = await fetch('dataEx/monsterTraits.json');
      if (!response.ok) {
        console.warn("[MTM] monsterTraits.json not found. Using defaults.");
        MONSTER_TRAITS = getDefaultTraits();
        return MONSTER_TRAITS;
      }
      MONSTER_TRAITS = await response.json();
      console.log("[MTM] Monster traits loaded:", Object.keys(MONSTER_TRAITS));
      
      // 各モンスターの特性をログ出力
      for (const [key, traits] of Object.entries(MONSTER_TRAITS)) {
        console.log(`[MTM] ${key}: allowStage1Insert=${traits.allowStage1Insert}`);
      }
      
      return MONSTER_TRAITS;
    } catch (error) {
      console.error("[MTM] Failed to load traits:", error);
      MONSTER_TRAITS = getDefaultTraits();
      return MONSTER_TRAITS;
    }
  }
  
  // フォールバック（ファイルがない場合）
  function getDefaultTraits() {
    return {
      slime: { allowStage1Insert: false },
      goblin: { allowStage1Insert: false },
      tentacle: { allowStage1Insert: false },
      breastTentacle: { allowStage1Insert: false },
      hypnoticGhost: { allowStage1Insert: false }
    };
  }
  
  // ゲーム起動時に読み込み
  const _Scene_Boot_start = Scene_Boot.prototype.start;
  Scene_Boot.prototype.start = function() {
    _Scene_Boot_start.call(this);
    loadTraits();
  };

  // ============================================================================
  // API関数
  // ============================================================================
  
  /**
   * 敵IDからモンスターキーを取得
   * @param {number} enemyId - 敵キャラID
   * @returns {string} モンスターキー（例: "slime"）
   */
  function getEnemyKey(enemyId) {
    const enemy = $dataEnemies[enemyId];
    
    if (!enemy) {
      return null;
    }
    
    // メタタグに MonsterKey があればそれを使用
    if (enemy.meta?.MonsterKey) {
      console.log(`[MTM] Found MonsterKey in meta: ${enemy.meta.MonsterKey}`);
      return enemy.meta.MonsterKey;
    }
    
    // なければ名前から推測（具体的なものから先にチェック）
    const name = enemy.name;
    const fallbackMap = {
      // DB表示名 : 内部ベース名（英字）
      "スライム": "slime",
      "電撃スライム": "electSlime",
      "毒スライム": "poisonSlime",
      "媚薬スライム": "aphrodisiacSlime",
      "ゴブリン": "goblin",
      "触手":"tentacle",
      "ルストテンタケル": "breastTentacle",
      "レイダー": "reider",
      "レイダーの首領": "reiderBoss",
      "掌打師": "slapper",
      "孕ませ狂い": "insaneMan",
      "催眠ゴースト": "hypnoticGhost",
      "ヘーベテンタクル": "tentacle_hebe"
    };
    
    // 完全一致チェック
    if (fallbackMap[name]) {
      console.log(`[MTM] Detected enemy key from name "${enemy.name}": ${fallbackMap[name]}`);
      return fallbackMap[name];
    }
    
    // 部分一致チェック（後方互換性のため）
    const nameLower = name.toLowerCase();
    let key = null;
    
    if (nameLower.includes("スライム") || nameLower.includes("slime")) {
      key = "slime";
    } else if (nameLower.includes("ルストテンタケル") || nameLower.includes("breasttentacle")) {
      key = "breastTentacle";
    } else if (nameLower.includes("ゴブリン") || nameLower.includes("goblin")) {
      key = "goblin";
    } else if (nameLower.includes("触手") || nameLower.includes("tentacle")) {
      key = "tentacle";
    } else {
      // デフォルトは名前の最初の単語
      key = nameLower.split(/[\s_]/)[0];
    }
    
    console.log(`[MTM] Detected enemy key from name "${enemy.name}": ${key}`);
    return key;
  }
  
  /**
   * 特性を取得
   * @param {string} enemyKey - モンスターキー
   * @param {string} traitName - 特性名
   * @returns {*} 特性の値
   */
  function getTrait(enemyKey, traitName) {
    if (!MONSTER_TRAITS) {
      console.warn("[MTM] Traits not loaded yet");
      return null;
    }
    
    const traits = MONSTER_TRAITS[enemyKey];
    if (!traits) {
      console.warn(`[MTM] No traits found for: ${enemyKey}`);
      return null;
    }
    
    return traits[traitName];
  }
  
  /**
   * stage1 から挿入可能か
   * @param {string} enemyKey - モンスターキー
   * @returns {boolean}
   */
  function allowsStage1Insert(enemyKey) {
    const value = getTrait(enemyKey, 'allowStage1Insert');
    const result = value === true;
    console.log(`[MTM] allowsStage1Insert("${enemyKey}") = ${result}`);
    return result;
  }
  
  /**
   * すべての特性を取得
   * @param {string} enemyKey - モンスターキー
   * @returns {Object}
   */
  function getAllTraits(enemyKey) {
    if (!MONSTER_TRAITS) return {};
    return MONSTER_TRAITS[enemyKey] || {};
  }

  // ============================================================================
  // グローバルAPI
  // ============================================================================

  window.MonsterTraitManager = {
    loadTraits,
    getEnemyKey,
    getTrait,
    allowsStage1Insert,
    getAllTraits,
    getTraits: () => MONSTER_TRAITS || {}
  };

  console.log("[MonsterTraitManager] Ready");
})();

