// OverlayBuilder.js - オーバーレイ・SFX自動構築システム
(() => {
  console.log("[OverlayBuilder] Loading...");

  // =============================================================================
  // データ定義
  // =============================================================================
  
  // ステージ別の頬染めマッピング
  const BLUSH_BY_STAGE = {
    stage1: "blush_low",
    stage2: "blush_mid",
    stage3: "blush_high"
  };

  // =============================================================================
  // 内部ヘルパー関数
  // =============================================================================
  
  /**
   * オーバーレイをスキップすべきかチェック（汎用機能）
   * @param {string} enemyKey - 敵タイプ
   * @param {string} overlayKey - オーバーレイキー
   * @returns {boolean} - trueならスキップ
   */
  function shouldSkipOverlay(enemyKey, overlayKey) {
    const enemySets = window.OverlaySets?.[enemyKey];
    if (!enemySets?.skipOverlays || !Array.isArray(enemySets.skipOverlays)) {
      return false;
    }
    
    // skipOverlays配列内のパターンと前方一致でチェック
    return enemySets.skipOverlays.some(pattern => {
      if (typeof pattern === 'string') {
        // 前方一致: "attack_breast" は "attack_breast_soft" や "attack_breast_hard" もマッチ
        return overlayKey.startsWith(pattern);
      }
      return false;
    });
  }
  
  /**
   * 攻撃エフェクトオーバーレイを解決
   * @param {string} enemyKey - 敵タイプ
   * @param {string} actionType - "attack" または "insert"
   * @param {string} targetPart - "pussy", "breast", etc.
   * @param {string} intensity - "soft" または "hard"
   * @returns {Object} - { overlays: [], layer: null, intensity: "soft"|"hard" }
   */
  function resolveAttackOverlay(enemyKey, actionType, targetPart, intensity) {
    // 1. キャッシュされたマージ済みマッピングを取得（パフォーマンス最適化）
    const mapping = window._mergedOverlaySets?.[enemyKey] || window.DEFAULT_OVERLAY;
    
    console.log(`[OverlayBuilder-Debug] resolveAttackOverlay called: enemyKey="${enemyKey}", actionType="${actionType}", targetPart="${targetPart}", intensity="${intensity}"`);
    console.log(`[OverlayBuilder-Debug] mapping found:`, mapping ? "yes" : "no", mapping ? Object.keys(mapping) : "N/A");
    
    if (!mapping) {
      console.warn(`[OverlayBuilder] No overlay mapping found for enemy: ${enemyKey}`);
      return { overlays: [], layer: null, intensity: intensity };
    }
    const intensityKey = intensity === "hard" ? "hard" : "soft";
    
    // 2. アクションタイプに応じたスロットを解決
    const actionSlot = mapping[actionType];
    console.log(`[OverlayBuilder-Debug] actionSlot[${actionType}]:`, actionSlot ? Object.keys(actionSlot) : "not found");
    if (!actionSlot) return { overlays: [], layer: null, intensity: intensityKey };
    
    // 3. ターゲット部位のオーバーレイを取得
    const partSlot = actionSlot[targetPart];
    console.log(`[OverlayBuilder-Debug] partSlot[${targetPart}]:`, partSlot ? (typeof partSlot === "object" ? Object.keys(partSlot) : partSlot) : "not found");
    if (!partSlot) return { overlays: [], layer: null, intensity: intensityKey };
    
    const overlays = [];
    let layerHint = null;
    
    // 4. スロットの構造に応じて解決
    if (typeof partSlot === "string") {
      // 単純な文字列（強度なし）
      overlays.push(partSlot);
    } else if (typeof partSlot === "object") {
      // レイヤー情報を取得（部位固有の設定）
      layerHint = partSlot.layer || null;
      
      // 階層構造の場合（front/backなど）
      if (partSlot.front || partSlot.back) {
        // 前後がある場合（slimeなど）
        if (partSlot.front) {
          const frontKey = partSlot.front[intensityKey] || partSlot.front.soft || partSlot.front;
          if (frontKey) overlays.push(frontKey);
        }
        if (partSlot.back) {
          const backKey = partSlot.back[intensityKey] || partSlot.back.soft || partSlot.back;
          if (backKey) overlays.push(backKey);
        }
      } else {
        // 強度だけがある場合（goblinなど）
        const key = partSlot[intensityKey] || partSlot.soft || partSlot;
        if (typeof key === "string") overlays.push(key);
      }
    }
    
    return { overlays, layer: layerHint, intensity: intensityKey };
  }

  /**
   * シーケンス用のオーバーレイ取得（フレームベース）
   * @param {string} enemyKey - 敵タイプ
   * @param {string} targetPart - 部位
   * @param {boolean} isInserted - 挿入状態
   * @param {number} frameIndex - 現在のフレーム（0始まり）
   * @param {number} totalFrames - 総フレーム数
   * @param {string} eventType - イベントタイプ（danger, climax, orgasm, insert など）
   * @param {string} phase - フェーズ（pre, main）
   * @returns {Object} - { overlays: [], layer: null, intensity: "soft"|"hard" }
   */
  function getAttackOverlayForFrame(enemyKey, targetPart, isInserted, frameIndex, totalFrames, eventType = null, phase = null) {
    const actionType = isInserted ? "insert" : "attack";
    
    // キャッシュされたマージ済みマッピングを取得（パフォーマンス最適化）
    const mapping = window._mergedOverlaySets?.[enemyKey] || window.DEFAULT_OVERLAY;
    
    if (!mapping) {
      return { overlays: [], layer: null };
    }
    const actionSlot = mapping[actionType];
    if (!actionSlot) return { overlays: [], layer: null };
    
    const partSlot = actionSlot[targetPart];
    if (!partSlot) return { overlays: [], layer: null };
    
    // シーケンスパターンの解決（優先順位順）
    let sequence = null;
    
    // 1. sequences（複数形）がある場合: eventType + phase で解決
    if (partSlot.sequences && typeof partSlot.sequences === "object") {
      // 1-1. eventType:phase で検索（挿入イベント専用）
      if (eventType && phase) {
        const key = `${eventType}:${phase}`;
        sequence = partSlot.sequences[key];
        if (sequence) {
          console.log(`[OverlayBuilder-Sequence] Resolved by "${key}"`);
        }
      }
      
      // 1-2. eventType のみで検索（通常エロ攻撃、絶頂）
      if (!sequence && eventType) {
        sequence = partSlot.sequences[eventType];
        if (sequence) {
          console.log(`[OverlayBuilder-Sequence] Resolved by eventType="${eventType}"`);
        }
      }
    }
    
    // 2. 旧形式の sequence（単数形）がある場合（後方互換）
    if (!sequence && partSlot.sequence && Array.isArray(partSlot.sequence)) {
      sequence = partSlot.sequence;
      console.log(`[OverlayBuilder-Sequence] Using legacy sequence (single)`);
    }
    
    // 3. シーケンスパターンが見つかった場合
    if (sequence && Array.isArray(sequence)) {
      // フレームインデックスを正規化（0.0 ~ 1.0）
      const progress = frameIndex / Math.max(1, totalFrames - 1);
      
      // シーケンスパターンから最適なフレームを選択
      const totalPatternFrames = sequence.length;
      const targetIndex = Math.floor(progress * totalPatternFrames);
      const clampedIndex = Math.min(targetIndex, totalPatternFrames - 1);
      const sequenceFrame = sequence[clampedIndex];
      
      console.log(`[OverlayBuilder-Sequence] Frame ${frameIndex+1}/${totalFrames} (progress=${progress.toFixed(2)}) → pattern[${clampedIndex}] intensity=${sequenceFrame.intensity}`);
      
      if (sequenceFrame.intensity === null) {
        // このフレームではオーバーレイなし
        return { overlays: [], layer: partSlot.layer, intensity: null };
      }
      
      // 指定された強度のオーバーレイを取得
      const overlayKey = partSlot[sequenceFrame.intensity];
      if (overlayKey) {
        return {
          overlays: [overlayKey],
          layer: partSlot.layer || null,
          intensity: sequenceFrame.intensity  // ★強度情報を追加
        };
      }
    }
    
    // 4. シーケンスパターンがない場合は従来通り（progressベース）
    const progress = frameIndex / Math.max(1, totalFrames - 1);
    const intensity = progress > 0.6 ? "hard" : "soft";
    const overlayKey = partSlot[intensity];
    
    console.log(`[OverlayBuilder-Sequence] No sequence found, using progress-based: intensity="${intensity}"`);
    
    return {
      overlays: overlayKey ? [overlayKey] : [],
      layer: partSlot.layer || null,
      intensity: intensity  // ★強度情報を追加
    };
  }

  // =============================================================================
  // 公開API
  // =============================================================================
  
  /**
   * 攻撃エフェクトオーバーレイを取得
   * @param {string} enemyKey - 敵タイプ
   * @param {string} targetPart - 部位
   * @param {boolean} isInserted - 挿入状態
   * @param {string} intensity - "soft" または "hard"
   * @returns {Object} - { overlays: [], layer: null, intensity: "soft"|"hard" }
   */
  function getAttackOverlays(enemyKey, targetPart, isInserted, intensity) {
    const actionType = isInserted ? "insert" : "attack";
    return resolveAttackOverlay(enemyKey, actionType, targetPart, intensity);
  }

  /**
   * 頬染めオーバーレイを取得
   * @param {string} stage - "stage1", "stage2", "stage3"
   * @param {number} progress - 0.0-1.0 の進行度
   * @returns {string} - 頬染めキー
   */
  function getBlushOverlays(stage, progress = 0.5) {
    const baseBlush = BLUSH_BY_STAGE[stage] || "blush_low";
    
    // 後半でワンランク上がる可能性
    if (progress > 0.7 && Math.random() > 0.6) {
      if (baseBlush === "blush_low") {
        return "blush_mid";
      } else if (baseBlush === "blush_mid") {
        return "blush_high";
      }
    }
    
    return baseBlush;
  }

  // カテゴリ別に直前の選択を記憶（連続回避用）
  let _lastSelectedSfxByCategory = {};

  /**
   * 配列からランダム選択（連続同一回避・カテゴリ別管理）
   * @param {Array} sfxArray - SFX配列
   * @param {string} category - カテゴリ名（"inserted", "breast", "other"など）
   * @returns {string|null} - 選択されたSFXキー
   */
  function selectRandomSfx(sfxArray, category = "default") {
    if (!Array.isArray(sfxArray) || sfxArray.length === 0) {
      return null;
    }
    
    // 1つしかない場合はそのまま
    if (sfxArray.length === 1) {
      _lastSelectedSfxByCategory[category] = sfxArray[0];
      return sfxArray[0];
    }
    
    // カテゴリごとの前回選択を取得
    const lastSelected = _lastSelectedSfxByCategory[category];
    
    // 前回と違うものを選択
    let candidates = sfxArray;
    if (lastSelected && sfxArray.includes(lastSelected) && sfxArray.length > 1) {
      candidates = sfxArray.filter(sfx => sfx !== lastSelected);
      // フィルタ後が空なら全体から選択
      if (candidates.length === 0) {
        candidates = sfxArray;
      }
    }
    
    // ランダム選択
    const selected = candidates[Math.floor(Math.random() * candidates.length)];
    _lastSelectedSfxByCategory[category] = selected;
    return selected;
  }

  /**
   * 状況に応じたSFX（擬音）オーバーレイを自動選択
   * ★シンプル3カテゴリ版 + 強弱対応 + 連続回避 + SFX抑制対応
   * 
   * @param {Object} context - { enemyKey, eventType, targetPart, isInserted, phase, currentIntensity, suppressSfx, suppressSfxCategories }
   * @param {number} intensity - 0.0-1.0 の進行度（progressベース判定用）
   * @param {string} forcedCategory - 強制的に使用するカテゴリ（"breast", "inserted", "other"など）
   * @returns {Array|null} - SFXキーの配列 ["key_left", "key_right"] または null
   */
  function getBubbleOverlaysFromContext(context, intensity, forcedCategory = null) {
    const { enemyKey, eventType, targetPart, isInserted, phase } = context;
    
    // =============================================================================
    // ★SFX抑制チェック（優先順位順）
    // =============================================================================
    
    // 1. グローバル抑制: context.suppressSfx = true
    if (context.suppressSfx === true) {
      console.log(`[OverlayBuilder-SFX] Skipped: global suppressSfx flag enabled`);
      return null;
    }
    
    // 2. 挿入イベントのpre時は非表示（既存の動作を維持）
    if (eventType === "insert" && phase === "pre") {
      console.log(`[OverlayBuilder-SFX] Skipped: insert:pre phase (default behavior)`);
      return null;
    }
    
    if (!window.OverlaySets) {
      console.error("[OverlayBuilder] window.OverlaySets is undefined!");
      return null;
    }

    // SFXマップ取得: 敵固有 → デフォルト の順でフォールバック
    const enemySfx = window.OverlaySets?.[enemyKey]?.sfx;
    const defaultSfx = window.DEFAULT_SFX;
    const sfxMap = enemySfx ? { ...defaultSfx, ...enemySfx } : defaultSfx;
    
    if (!sfxMap) {
      console.warn(`[OverlayBuilder] No sfx map found for enemyKey="${enemyKey}"`);
      return null;
    }

    // カテゴリ選択（3分岐 + 強制指定対応）
    let categoryMap = null;
    let categoryName = "";
    
    if (forcedCategory) {
      // 強制的に指定されたカテゴリを使用
      categoryMap = sfxMap[forcedCategory];
      categoryName = forcedCategory;
      console.log(`[OverlayBuilder-SFX] Using forced category: ${forcedCategory}`);
    } else if (isInserted) {
      categoryMap = sfxMap.inserted;
      categoryName = "inserted";
    } else if (targetPart === "breast") {
      categoryMap = sfxMap.breast;
      categoryName = "breast";
    } else {
      categoryMap = sfxMap.other;
      categoryName = "other";
    }
    
    // 3. カテゴリ別抑制: context.suppressSfxCategories = ["breast", "inserted", ...]
    if (Array.isArray(context.suppressSfxCategories) && 
        context.suppressSfxCategories.includes(categoryName)) {
      console.log(`[OverlayBuilder-SFX] Skipped: category "${categoryName}" is in suppressSfxCategories`);
      return null;
    }
    
    if (!categoryMap) {
      console.warn(`[OverlayBuilder-SFX] No category map found for: ${categoryName}`);
      return null;
    }
    
    // ★強弱に応じた配列選択
    let sfxArray = null;
    let intensityKey = "soft";
    
    // contextから直接intensityを取得（シーケンスパターンから来る場合）
    if (context.currentIntensity) {
      intensityKey = context.currentIntensity;
    } 
    // 数値のintensityから判定（progress値から来る場合）
    else if (typeof intensity === "number") {
      intensityKey = intensity > 0.6 ? "hard" : "soft";
    }
    
    // 新形式: { soft: [...], hard: [...] }
    if (categoryMap.soft || categoryMap.hard) {
      sfxArray = categoryMap[intensityKey] || categoryMap.soft || [];
      console.log(`[OverlayBuilder-SFX] Using intensity-based selection: ${intensityKey}`);
    } 
    // 旧形式: [...] 配列のみ（後方互換）
    else if (Array.isArray(categoryMap)) {
      sfxArray = categoryMap;
      console.log(`[OverlayBuilder-SFX] Using legacy array format (no intensity)`);
    }
    
    // ランダム選択（連続回避・カテゴリ別）
    const selectedKey = selectRandomSfx(sfxArray, categoryName);
    if (selectedKey) {
      console.log(`[OverlayBuilder-SFX] Selected: category="${categoryName}", intensity="${intensityKey}", key="${selectedKey}"`);
      return [selectedKey];  // 単一ファイル（左右の擬音が両方描かれている）
    }
    
    console.warn(`[OverlayBuilder-SFX] No SFX selected for context:`, context);
    return null;
  }

  /**
   * モンスター設定から挿入時のデフォルトレイヤーを取得
   * @param {Object} enemySets - モンスターの設定
   * @param {string} targetPart - 対象部位
   * @param {boolean} isInserted - 挿入状態か
   * @returns {string} - "overlay" または "underOverlay"
   */
  function getDefaultInsertLayer(enemySets, targetPart, isInserted) {
    if (!isInserted) return "overlay";  // 非挿入時は常にoverlay
    
    // 1. insertedBehavior.layer から取得（全部位共通）
    const behaviorLayer = enemySets?.insertedBehavior?.layer;
    if (behaviorLayer) {
      console.log(`[OverlayBuilder] Using insertedBehavior.layer: ${behaviorLayer}`);
      return behaviorLayer;
    }
    
    // 2. insertedBehavior.layerSettings から部位別に取得
    const layerSettings = enemySets?.insertedBehavior?.layerSettings;
    if (layerSettings && layerSettings[targetPart]) {
      console.log(`[OverlayBuilder] Using layerSettings[${targetPart}]: ${layerSettings[targetPart]}`);
      return layerSettings[targetPart];
    }
    
    // 3. デフォルト: 挿入時は overlay（干渉しない場合が多いため）
    return "overlay";
  }

  /**
   * 1行分のオーバーレイを構築
   * @param {Object} context - コンテキスト情報
   * @param {number} progress - 0.0-1.0 の進行度
   * @param {number} frameIndex - フレームインデックス（オプション）
   * @param {number} totalFrames - 総フレーム数（オプション）
   * @returns {Object} - { overlays: [], underOverlays: [], currentIntensity: "soft"|"hard"|null, additionalSfxCategories: [] }
   */
  function buildOverlaysForLine(context, progress, frameIndex = null, totalFrames = null) {
    const { stage, targetPart, enemyKey, isInserted, isRestrained, eventType, phase } = context;
    const overlays = [];
    const underOverlays = [];
    const additionalSfxCategories = [];  // 追加のSFXカテゴリ
    
    // ★enemySetsを先に取得（全処理で使用）
    const enemySets = window.OverlaySets?.[enemyKey];
    
    // ★stage番号を抽出（"stage1" → 1）
    const stageNum = parseInt(stage?.replace(/\D/g, "") || "1", 10);
    
    // 1. 拘束エフェクト（マッピングから取得）
    if (isRestrained) {
      const restraintKey = enemySets?.mapping?.restrain;
      if (restraintKey) {
        // ★スキップチェック
        if (!shouldSkipOverlay(enemyKey, restraintKey)) {
          overlays.push(restraintKey);
        } else {
          console.log(`[OverlayBuilder] Skipped overlay: ${restraintKey} (skipOverlays setting)`);
        }
      } else {
        // フォールバック：互換性のため
        if (!shouldSkipOverlay(enemyKey, "restrain")) {
          overlays.push("restrain");
        } else {
          console.log(`[OverlayBuilder] Skipped overlay: restrain (skipOverlays setting)`);
        }
      }
    }
    
    // 2. 頬の赤み（overlays - 前面）
    // 非拘束時（立ち絵）は赤面差分がないためスキップ
    // ★stand_pose（汎用立ち絵）使用時もスキップ（差分がないため）
    if (isRestrained && !context.willUseStandPose) {
      const blush = getBlushOverlays(stage, progress);
      overlays.push(blush);
    }
    
    // 3. 挿入・エロ攻撃エフェクト（★シーケンスパターン対応）
    // ★非拘束時（stand_pose等）はエロ攻撃オーバーレイをスキップ
    let attackResult = { overlays: [], layer: null, intensity: "soft" };
    let intensityInfo = null;
    let currentIntensity = "soft";  // ★追加：デフォルト値
    
    if (isRestrained || isInserted) {
      // ★拘束中または挿入中のみオーバーレイを生成
      if (frameIndex !== null && totalFrames !== null) {
        // シーケンスモード：フレームベースでパターンを適用
        // ★eventType と phase を渡して適切なシーケンスを選択
        attackResult = getAttackOverlayForFrame(
          enemyKey, 
          targetPart, 
          isInserted, 
          frameIndex, 
          totalFrames,
          eventType,  // ★追加
          phase       // ★追加
        );
        currentIntensity = attackResult.intensity || "soft";  // ★攻撃結果から強度を取得
        intensityInfo = `frame ${frameIndex+1}/${totalFrames}`;
      } else {
        // 従来モード：progressベースで決定
        currentIntensity = progress > 0.6 ? "hard" : "soft";  // ★強度を保存
        attackResult = getAttackOverlays(enemyKey, targetPart, isInserted, currentIntensity);
        intensityInfo = currentIntensity;
      }
      
      // ★レイヤー配置先を決定（優先順位: 部位個別layer > モンスターのinsertedBehavior > デフォルト）
      // 1. 部位個別の layer 設定（最優先）
      let preferredLayer = attackResult.layer;
      
      // 2. 部位個別の設定がない場合、モンスターのinsertedBehavior設定を参照
      if (!preferredLayer) {
        preferredLayer = getDefaultInsertLayer(enemySets, targetPart, isInserted);
      }
      
      // 3. それでもない場合はデフォルト
      if (!preferredLayer) {
        preferredLayer = enemySets?.mapping?.layer || "overlay";
      }
      
      console.log(`[OverlayBuilder] Layer decision: enemyKey=${enemyKey}, targetPart=${targetPart}, isInserted=${isInserted}, layer=${preferredLayer}`);
      
      // ★スキップチェックを適用してから追加
      if (attackResult.overlays.length > 0) {
        const filteredOverlays = attackResult.overlays.filter(key => {
          const skip = shouldSkipOverlay(enemyKey, key);
          if (skip) {
            console.log(`[OverlayBuilder] Skipped overlay: ${key} (skipOverlays setting)`);
          }
          return !skip;
        });
        
        // ★レイヤーに応じて配置
        if (preferredLayer === "underOverlay") {
          underOverlays.push(...filteredOverlays);
          console.log(`[OverlayBuilder] → underOverlays: ${filteredOverlays.join(", ")}`);
        } else {
          overlays.push(...filteredOverlays);
          console.log(`[OverlayBuilder] → overlays: ${filteredOverlays.join(", ")}`);
        }
      }
    } else {
      console.log(`[OverlayBuilder] Skipped attack overlays: not restrained and not inserted (stand_pose)`);
    }
    
    // 4. 挿入時の追加オーバーレイ（モンスター設定に基づく）
    if (isInserted && isRestrained && enemySets?.insertedBehavior) {
      const behavior = enemySets.insertedBehavior;
      
      // ★ 非挿入攻撃エフェクトの継続表示（媚薬スライム専用：挿入時も全身攻撃を継続）
      if (behavior.alwaysShowFullbodyAttack) {
        let fullbodyIntensity = currentIntensity;
        if (behavior.fullbodyIntensityMode === "soft" || behavior.fullbodyIntensityMode === "hard") {
          fullbodyIntensity = behavior.fullbodyIntensityMode;
        }
        
        // pussyの非挿入エフェクトを取得（fullbody画像）
        const fullbodyResult = getAttackOverlays(enemyKey, "pussy", false, fullbodyIntensity);
        if (fullbodyResult.overlays.length > 0) {
          // ★スキップチェック
          const filteredFullbody = fullbodyResult.overlays.filter(key => {
            const skip = shouldSkipOverlay(enemyKey, key);
            if (skip) {
              console.log(`[OverlayBuilder] Skipped fullbody overlay: ${key} (skipOverlays setting)`);
            }
            return !skip;
          });
          overlays.push(...filteredFullbody);
          console.log(`[OverlayBuilder] Added fullbody attack overlay for inserted state (${enemyKey}): ${filteredFullbody.join(", ")} [intensity: ${fullbodyIntensity}]`);
        }
      }
      
      // ★ 胸・膣用エフェクトとSFXの追加
      // fullbodyAttack と alwaysShowBreast で重複しないように排他制御
      if (behavior.alwaysShowFullbodyAttack) {
        // fullbodyAttack使用時：胸・膣のSFXを追加
        if (behavior.includeBreastSfx) {
          additionalSfxCategories.push({
            category: "breast",
            intensity: currentIntensity
          });
          console.log(`[OverlayBuilder] Marked breast SFX for fullbody attack [intensity: ${currentIntensity}]`);
        }
        
        if (behavior.includePussySfx) {
          additionalSfxCategories.push({
            category: "other",
            intensity: currentIntensity
          });
          console.log(`[OverlayBuilder] Marked pussy SFX for fullbody attack [intensity: ${currentIntensity}]`);
        }
      } else if (behavior.alwaysShowBreast) {
        // 旧実装：alwaysShowBreast使用時（fullbodyAttackと排他的）
        let breastIntensity = currentIntensity;
        if (behavior.breastIntensityMode === "soft" || behavior.breastIntensityMode === "hard") {
          breastIntensity = behavior.breastIntensityMode;
        }
        
        const breastResult = getAttackOverlays(enemyKey, "breast", false, breastIntensity);
        if (breastResult.overlays.length > 0) {
          // ★スキップチェック
          const filteredBreast = breastResult.overlays.filter(key => {
            const skip = shouldSkipOverlay(enemyKey, key);
            if (skip) {
              console.log(`[OverlayBuilder] Skipped breast overlay: ${key} (skipOverlays setting)`);
            }
            return !skip;
          });
          overlays.push(...filteredBreast);
          console.log(`[OverlayBuilder] Added breast overlay for inserted state (${enemyKey}): ${filteredBreast.join(", ")} [intensity: ${breastIntensity}]`);
        }
        
        // 胸用SFXの追加指示
        if (behavior.includeBreastSfx) {
          additionalSfxCategories.push({
            category: "breast",
            intensity: breastIntensity
          });
          console.log(`[OverlayBuilder] Marked breast SFX for addition [intensity: ${breastIntensity}]`);
        }
      }
    }
    
    // ★ 5. 非挿入時の追加SFX（媚薬スライム専用）
    if (!isInserted && isRestrained && enemySets?.nonInsertedBehavior) {
      const behavior = enemySets.nonInsertedBehavior;
      
      // 胸用SFXの追加
      if (behavior.alwaysShowBreastSfx) {
        let breastSfxIntensity = currentIntensity;
        if (behavior.breastSfxIntensityMode === "soft" || behavior.breastSfxIntensityMode === "hard") {
          breastSfxIntensity = behavior.breastSfxIntensityMode;
        }
        
        additionalSfxCategories.push({
          category: "breast",
          intensity: breastSfxIntensity
        });
        console.log(`[OverlayBuilder] Marked breast SFX for non-inserted state (${enemyKey}) [intensity: ${breastSfxIntensity}]`);
      }
    }
    
    // ★ 6. stage判定による複数部位攻撃（催眠ゴースト専用）
    if (!isInserted && isRestrained && enemySets?.multiPartBehavior) {
      const behavior = enemySets.multiPartBehavior;
      const enableFromStage = behavior.enableFromStage || 2;
      
      // stage2以降で膣攻撃も追加（stage1は胸のみ）
      if (stageNum >= enableFromStage && behavior.alwaysShowPussyFromStage2) {
        let pussyIntensity = currentIntensity;
        if (behavior.pussyIntensityMode === "soft" || behavior.pussyIntensityMode === "hard") {
          pussyIntensity = behavior.pussyIntensityMode;
        }
        
        // 膣攻撃オーバーレイを追加
        const pussyResult = resolveAttackOverlay(enemyKey, "attack", "pussy", pussyIntensity);
        if (pussyResult.overlays.length > 0) {
          // ★スキップチェック
          const filteredPussy = pussyResult.overlays.filter(key => {
            const skip = shouldSkipOverlay(enemyKey, key);
            if (skip) {
              console.log(`[OverlayBuilder] Skipped pussy overlay: ${key} (skipOverlays setting)`);
            }
            return !skip;
          });
          overlays.push(...filteredPussy);
          console.log(`[OverlayBuilder] Added pussy overlay for stage${stageNum} multi-part attack (${enemyKey}): ${filteredPussy.join(", ")} [intensity: ${pussyIntensity}]`);
        }
        
        // 膣用SFXの追加指示
        if (behavior.includePussySfx) {
          additionalSfxCategories.push({
            category: "other",  // pussy攻撃は"other"カテゴリ
            intensity: pussyIntensity
          });
          console.log(`[OverlayBuilder] Marked pussy SFX for stage${stageNum} multi-part attack [intensity: ${pussyIntensity}]`);
        }
      }
    }
    
    return { 
      overlays, 
      underOverlays,
      currentIntensity,        // ★強度情報を返す
      additionalSfxCategories  // ★追加のSFXカテゴリ
    };
  }

  // =============================================================================
  // グローバルに公開
  // =============================================================================
  
  window.OverlayBuilder = {
    // データ
    BLUSH_BY_STAGE,
    
    // 公開関数
    buildOverlaysForLine,
    getAttackOverlays,
    getBlushOverlays,
    getBubbleOverlaysFromContext,
    
    // 内部関数（デバッグ・テスト用）
    _internal: {
      resolveAttackOverlay,
      getAttackOverlayForFrame
    }
  };

  console.log("[OverlayBuilder] Ready");
})();

