// PleasureDisplayManager.js (per-line FX, safe & logged)
/* =========================================================
 * PleasureDisplayManager.js — 完全修正版（per-line FX + underOverlays対応）
 * ========================================================= */
(() => {
  const FILE_VERSION = "2024-11-15-v3-SPRITE-OVERLAY";
  console.log(`[PleasureDisplayManager] Loaded (per-line FX + underOverlays) - VERSION: ${FILE_VERSION}`);

  /* ===== 共通ユーティリティ ===== */

  function _resolveProfileName(actor, baseInfo) {
    return baseInfo?.profileName
      || window.PleasurePoseController?.getProfileNameFromEnemy?.(actor)
      || "default";
  }

  function playSE(nameOrObj) {
    if (!nameOrObj) return;
    const se = (typeof nameOrObj === "string")
      ? { name: nameOrObj, volume: 90, pitch: 100, pan: 0 }
      : {
          name:   String(nameOrObj.name || ""),
          volume: Number(nameOrObj.volume ?? 90),
          pitch:  Number(nameOrObj.pitch  ?? 100),
          pan:    Number(nameOrObj.pan    ?? 0),
        };
    if (!se.name) return;
    try {
      console.log("[PDM][SE] play:", se);
      AudioManager?.playSe?.(se);
    } catch (e) {
      console.warn("[PDM][SE] failed:", e);
    }
  }

  function playSway(actor, opt = {}) {
    const o = {
      amplitude: Number(opt.amplitude ?? 12),
      duration:  Number(opt.duration  ?? 18),
      cycles:    Number(opt.cycles    ?? 2),
    };
    try {
      console.log("[PDM][SWAY] play:", o);
      window.OneShotSway?.play?.(actor, o);
    } catch (e) {
      console.warn("[PDM][SWAY] failed:", e);
    }
  }

  function flashFallbackTint() {
    const pink   = [68, -34, 34, 68];
    const normal = [0, 0, 0, 0];
    console.log("[PDM][FLASH] fallback tint");
    try {
      $gameScreen.startTint(pink, 10);
      setTimeout(() => $gameScreen.startTint(normal, 10), 160);
    } catch (e) {
      console.warn("[PDM][FLASH] tint failed:", e);
    }
  }

  // 継続的なピンク背景（絶頂・挿入シーン用）
  // ピクチャベースの半透明オーバーレイを使用（Z-index問題を回避）
  const PINK_BG_PICTURE_ID = 99; // 使用するピクチャID（通常使われない番号）
  
  // ピンクオーバーレイのフェードイン
  function _fadeInPinkOverlay(overlay, frames) {
    const targetAlpha = 1.0;
    const startAlpha = overlay.alpha;
    const deltaAlpha = (targetAlpha - startAlpha) / frames;
    let currentFrame = 0;
    
    // 既存のフェードタイマーがあればクリア
    if (overlay._pinkFadeTimerId) {
      clearInterval(overlay._pinkFadeTimerId);
    }
    
    overlay._pinkFadeTimerId = setInterval(() => {
      currentFrame++;
      overlay.alpha = Math.min(targetAlpha, startAlpha + deltaAlpha * currentFrame);
      
      if (currentFrame >= frames) {
        clearInterval(overlay._pinkFadeTimerId);
        overlay._pinkFadeTimerId = null;
        overlay.alpha = targetAlpha;
        console.log("[PDM] Pink overlay fade-in complete");
      }
    }, 1000 / 60); // 60fps
  }
  
  function startPinkBackground() {
    console.log("[PDM] ★★★ Starting pink background for scene ★★★");
    try {
      if (!$gameScreen) {
        console.error("[PDM] $gameScreen is not available!");
        return;
      }
      
      // 方法1: Tint（既存／かなり淡いピンクに調整）
      const pink = [60, -30, 30, 60];
      $gameScreen.startTint(pink, 15);
      console.log("[PDM] Tint applied (soft):", pink);
      
      // 方法2: Flash（追加で視認性向上）
      const scene = SceneManager._scene;
      if (scene && scene._spriteset) {
        // 戦闘スプライトセット全体にピンクフィルタを適用
        const spriteset = scene._spriteset;
        if (!spriteset._pinkOverlay) {
          // ピンクのエロいグラデーション背景を作成（全体的に淡め）
          const overlay = new Sprite();
          const w = Graphics.width;
          const h = Graphics.height;
          overlay.bitmap = new Bitmap(w, h);

          // ベース：全画面を淡いピンクでふわっと塗る
          overlay.bitmap.fillAll("rgba(255, 180, 210, 0.22)");

          // 上部：やや濃いピンクのグラデーション（天井側が濃い）
          if (typeof overlay.bitmap.gradientFillRect === "function") {
            overlay.bitmap.gradientFillRect(
              0,
              0,
              w,
              h * 0.45,
              "rgba(255, 120, 190, 0.50)",
              "rgba(255, 180, 210, 0.0)",
              true
            );

            // 下部：太もも〜足元側を少し暗め＆濃いめに
            overlay.bitmap.gradientFillRect(
              0,
              h * 0.6,
              w,
              h * 0.4,
              "rgba(255, 180, 210, 0.02)",
              "rgba(200, 60, 140, 0.35)",
              true
            );
          }

          overlay.z = 1000; // 最前面
          overlay.alpha = 0; // フェードイン用に初期値を0に
          overlay._pinkFadeTimerId = null;
          spriteset._pinkOverlay = overlay;
          spriteset.addChild(overlay);
          console.log("[PDM] Pink overlay sprite created and added");
          
          // フェードインアニメーション（30フレーム = 約0.5秒）
          _fadeInPinkOverlay(overlay, 30);
        } else {
          // 既存オーバーレイを再利用（フェードアウト中なら止めて復活させる）
          const overlay = spriteset._pinkOverlay;
          if (overlay._pinkFadeTimerId) {
            clearInterval(overlay._pinkFadeTimerId);
            overlay._pinkFadeTimerId = null;
          }
          overlay.visible = true;
          // フェードインアニメーション
          _fadeInPinkOverlay(overlay, 30);
          console.log("[PDM] Pink overlay sprite fading in");
        }
      } else {
        console.warn("[PDM] Scene or spriteset not available for overlay");
      }
    } catch (e) {
      console.error("[PDM] Failed to start pink background:", e);
    }
  }

  function stopPinkBackground() {
    console.log("[PDM] ★★★ Stopping pink background ★★★");
    try {
      if (!$gameScreen) {
        console.error("[PDM] $gameScreen is not available!");
        return;
      }
      
      // 方法1: Tintをゆるやかにクリア
      const normal = [0, 0, 0, 0];
      $gameScreen.startTint(normal, 30);
      console.log("[PDM] Tint clearing (soft)");
      
      // 方法2: オーバーレイスプライトをフェードアウト
      const scene = SceneManager._scene;
      if (scene && scene._spriteset && scene._spriteset._pinkOverlay) {
        const overlay = scene._spriteset._pinkOverlay;
        // すでにフェード中なら一旦止める
        if (overlay._pinkFadeTimerId) {
          clearInterval(overlay._pinkFadeTimerId);
          overlay._pinkFadeTimerId = null;
        }
        const durationMs = 700;      // フェードアウト全体の時間
        const steps      = 20;       // 分割数
        const interval   = durationMs / steps;
        const startAlpha = overlay.alpha ?? 1.0;
        const stepDelta  = startAlpha / steps;

        overlay.visible = true;
        overlay._pinkFadeTimerId = setInterval(() => {
          let a = (overlay.alpha ?? 0) - stepDelta;
          if (a <= 0) {
            a = 0;
            overlay.alpha = 0;
            overlay.visible = false;
            clearInterval(overlay._pinkFadeTimerId);
            overlay._pinkFadeTimerId = null;
            console.log("[PDM] Pink overlay sprite faded out");
          } else {
            overlay.alpha = a;
          }
        }, interval);
        console.log("[PDM] Pink overlay sprite fading out");
      }
    } catch (e) {
      console.error("[PDM] Failed to stop pink background:", e);
    }
  }

  /**
   * フラッシュ発火（既定は“非同期＝同時進行”）
   * @param {"play"|"startFlash"} api
   * @param {boolean} awaitIt
   * @param {object} flashOpts
   */
  async function playFlash(actor, baseInfo, { api = "play", awaitIt = false, flashOpts = null } = {}) {
    const profileName = _resolveProfileName(actor, baseInfo);
    try {
      if (window.PleasureFlashFX) {
        if (api === "startFlash" && typeof window.PleasureFlashFX.startFlash === "function") {
          console.log("[PDM][FLASH] startFlash:", profileName, flashOpts || {});
          const r = window.PleasureFlashFX.startFlash(profileName, flashOpts || {});
          if (awaitIt && r?.then) await r;
        } else if (typeof window.PleasureFlashFX.play === "function") {
          console.log("[PDM][FLASH] play:", profileName, flashOpts || {});
          const p = window.PleasureFlashFX.play(profileName, flashOpts || {});
          if (awaitIt) await p;
        } else {
          console.warn("[PDM][FLASH] API not found -> fallback");
          flashFallbackTint();
        }
      } else {
        flashFallbackTint();
      }
    } catch (e) {
      console.warn("[PDM][FLASH] failed -> fallback", e);
      flashFallbackTint();
    }
  }

  /**
   * 行単位FX実行
   * @param {Game_Actor} actor
   * @param {Object} baseInfo  { profileName?, eventType? ... }
   * @param {Object} entryFx   例: { flash:true, sway:{}, se:"Damage2", cutin:{...}, regionView:{...} }
   * @param {Object} opts      例: { awaitFlash:false, flashApi:"play" }
   */
  async function runEntryFx(actor, baseInfo, entryFx, opts = {}) {
    if (!entryFx) return;
    const { awaitFlash = false, flashApi = "play" } = opts;

    // 1) SE
    if (entryFx.se) playSE(entryFx.se);

    // 2) FLASH（flash / flashFx どちらでも可）
    const wantFlash = !!(entryFx.flash || entryFx.flashFx);
    if (wantFlash) {
      console.log("[PDM][FX] Flash requested:", entryFx.flash || entryFx.flashFx);
      // flashOptions を併用（無ければ undefined）
      await playFlash(actor, baseInfo, { api: flashApi, awaitIt: awaitFlash, flashOpts: entryFx.flashOptions });
    }

    // 3) SWAY
    if (entryFx.sway) playSway(actor, entryFx.sway);

    // 4) CUTIN（任意）
    if (entryFx.cutin?.enabled && window.CutinManager?.showCutinImageAsync) {
      const part = String(entryFx.cutin.part || "pussy");
      const type = String(entryFx.cutin.type || baseInfo?.eventType || "react");
      const prof = _resolveProfileName(actor, baseInfo);
      try {
        console.log("[PDM][CUTIN] show:", { prof, part, type });
        await window.CutinManager.showCutinImageAsync(prof, part, type);
      } catch (e) {
        console.warn("[PDM][CUTIN] failed:", e);
      }
    }

    // 5) REGION VIEW（任意・実装依存に注意）
    if (entryFx.regionView && window.RegionViewManager?.show) {
      const part  = String(entryFx.regionView.part  || "pussy");
      const type  = String(entryFx.regionView.type  || baseInfo?.eventType || "react");
      const stage = String(entryFx.regionView.stage || "stage2");
      const prof  = _resolveProfileName(actor, baseInfo);
      try {
        console.log("[PDM][REGION] show:", { part, prof, type, stage });
        // 実装の引数順が異なる可能性あり（必要に応じて合わせてください）
        window.RegionViewManager.show(part, prof, part, type, stage);
      } catch (e) {
        console.warn("[PDM][REGION] failed:", e);
      }
    }
  }

  /* ===== overlays 正規化 ===== */

  // OverlayResolver があればそれに通す。無ければ素通し。
  function _resolveLineOverlays(rawOverlays) {
    if (rawOverlays == null) return [];
    const resolver = window.OverlayResolver?.resolveOverlays;
    if (typeof resolver === "function") {
      const resolved = resolver(rawOverlays);
      return Array.isArray(resolved) ? resolved : (resolved ? [resolved] : []);
    }
    return Array.isArray(rawOverlays) ? rawOverlays : [rawOverlays].filter(Boolean);
  }

  // 配列の内容が等しいかチェック（差分更新用）
  function _arraysEqual(a, b) {
    if (a === b) return true;
    if (!a || !b) return false;
    if (a.length !== b.length) return false;
    for (let i = 0; i < a.length; i++) {
      if (a[i] !== b[i]) return false;
    }
    return true;
  }

  /* ===== メッセージ待機フォールバック ===== */

  function _waitMessageWindowLocal() {
    return new Promise(resolve => {
      const h = setInterval(() => {
        try {
          if (!$gameMessage.isBusy()) { clearInterval(h); resolve(); }
        } catch (_) { clearInterval(h); resolve(); }
      }, 10);
    });
  }

  /* ===== メイン：PleasureDisplayManager ===== */

  window.PleasureDisplayManager = {
    /**
     * 立ち絵とセリフを表示（dialogueSequence対応・行ごとFX対応）
     * @param {Game_Actor} actor
     * @param {Object} displayInfo  pose/expression/overlays/underOverlays/dialogue/dialogueSequence/...
     */
    async show(actor, displayInfo) {
      if (!actor || !displayInfo) return;

      // 最低限の補完
      displayInfo.profileName = displayInfo.profileName || _resolveProfileName(actor, displayInfo);

      const resolvePoseAuto = () => {
        const stage = actor.getClothingStage?.() || "intact";
        const type  = actor.getClothingType?.()  || "uniform";
        return `${type}_${stage}`;
      };

      if (!displayInfo.pose || displayInfo.pose === "auto") {
        displayInfo.pose = resolvePoseAuto();
      }
      if (!displayInfo.expression) displayInfo.expression = "normal";

      if (!Array.isArray(displayInfo.overlays)) displayInfo.overlays = [];
      displayInfo.overlays = _resolveLineOverlays(displayInfo.overlays);

      if (!Array.isArray(displayInfo.underOverlays)) displayInfo.underOverlays = [];
      displayInfo.underOverlays = _resolveLineOverlays(displayInfo.underOverlays);

      const hasSequence = Array.isArray(displayInfo.dialogueSequence);

      // === sequence あり：最初の行の表情・オーバーレイで初期化（ちらつき防止） ===
      if (hasSequence) {
        // "現在状態"を持っておく（継続指定に対応）
        // ★ 最初の行の表情・オーバーレイを初期値として使用（指定がない場合はフォールバック）
        const first = displayInfo.dialogueSequence[0] || {};
        
        const resolvePoseForInit = () => {
          if ("pose" in first) return first.pose === "auto" ? resolvePoseAuto() : first.pose;
          return displayInfo.pose;
        };
        
        const initialPose       = resolvePoseForInit();
        const initialExpression = ("expression" in first) ? first.expression : displayInfo.expression || "normal";
        const initialOverlays   = ("overlays" in first) ? _resolveLineOverlays(first.overlays) : displayInfo.overlays || [];
        const initialUnder      = ("underOverlays" in first) ? _resolveLineOverlays(first.underOverlays) : displayInfo.underOverlays || [];
        
        let currentPose       = initialPose;
        let currentExpression = initialExpression;
        let currentOverlays   = initialOverlays;
        let currentUnder      = initialUnder;

        // ★ 最初の行のみ特別処理：初期表示＋FX＋メッセージ
        const firstEntry = displayInfo.dialogueSequence[0];
        if (firstEntry) {
          const entry = (typeof firstEntry === "string") ? { text: firstEntry } : (firstEntry || {});
          const lineText = (entry.text ?? entry.dialogue ?? "");
          
          // 最初の行で初期表示（ちらつき防止のため、ここで一度だけ表示）
          if (window.BattleBustManager?.show) {
            await window.BattleBustManager.show(actor, {
              profileName:   displayInfo.profileName,
              eventType:     displayInfo.eventType,
              pose:          currentPose,
              expression:    currentExpression,
              overlays:      currentOverlays,
              underOverlays: currentUnder
            });
          }
          
          // 行FX → セリフ
          await runEntryFx(actor, displayInfo, entry.fx, { awaitFlash: false, flashApi: "play" });
          if (lineText.trim()) await this.showMessage(lineText);
        }

        // ★ 2行目以降の処理（差分更新のみ）
        for (let i = 1; i < displayInfo.dialogueSequence.length; i++) {
          const rawEntry = displayInfo.dialogueSequence[i];
          const entry    = (typeof rawEntry === "string") ? { text: rawEntry } : (rawEntry || {});
          const lineText = (entry.text ?? entry.dialogue ?? "");

          // pose：未指定→継続、"auto"→衣装自動
          const candidatePose = ("pose" in entry) ? entry.pose : currentPose;
          const linePose      = (!candidatePose || candidatePose === "auto") ? resolvePoseAuto() : candidatePose;

          // expression：未指定→継続
          const lineExpression = ("expression" in entry) ? entry.expression : currentExpression;

          // overlays（前面）：未指定→継続、指定あり→置換（マージしない）
          const lineOverlays = ("overlays" in entry)
            ? (_resolveLineOverlays(entry.overlays) || [])
            : (currentOverlays || []);

          // underOverlays（背面）：未指定→継続、指定あり→置換
          const lineUnder = ("underOverlays" in entry)
            ? (_resolveLineOverlays(entry.underOverlays) || [])
            : (currentUnder || []);

          // ★ 差分更新：変更がある場合のみ BattleBustManager.show() を呼び出す
          const poseChanged = (linePose !== currentPose);
          const exprChanged = (lineExpression !== currentExpression);
          const overlaysChanged = !_arraysEqual(lineOverlays, currentOverlays);
          const underChanged = !_arraysEqual(lineUnder, currentUnder);
          
          if (poseChanged || exprChanged || overlaysChanged || underChanged) {
            if (window.BattleBustManager?.show) {
              await window.BattleBustManager.show(actor, {
                profileName:   displayInfo.profileName,
                eventType:     displayInfo.eventType,
                pose:          linePose,
                expression:    lineExpression,
                overlays:      lineOverlays,
                underOverlays: lineUnder
              });
            }
          }

          // 行FX → セリフ
          await runEntryFx(actor, displayInfo, entry.fx, { awaitFlash: false, flashApi: "play" });
          if (lineText.trim()) await this.showMessage(lineText);

          // 次行用 現在値の更新
          currentPose       = linePose;
          currentExpression = lineExpression;
          currentOverlays   = lineOverlays;
          currentUnder      = lineUnder;
        }
        return; // sequence の場合はここで終了
      }

      // === sequence なし（単発表示） ===
      if (window.BattleBustManager?.show) {
        await window.BattleBustManager.show(actor, {
          profileName:   displayInfo.profileName,
          eventType:     displayInfo.eventType,
          pose:          displayInfo.pose,
          expression:    displayInfo.expression,
          overlays:      displayInfo.overlays,
          underOverlays: displayInfo.underOverlays
        });
      }

      const singleText = (typeof displayInfo.dialogue === "object")
        ? displayInfo.dialogue?.text : displayInfo.dialogue;

      if (displayInfo.dialogueFx) {
        console.log("[PDM] ★ dialogueFx detected:", displayInfo.dialogueFx);
        await runEntryFx(actor, displayInfo, displayInfo.dialogueFx, { awaitFlash: false, flashApi: "play" });
      } else {
        console.log("[PDM] ★ NO dialogueFx in displayInfo");
      }
      if (typeof singleText === "string" && singleText.trim()) {
        await this.showMessage(singleText);
      }
    },

    /**
     * テキストを自然な文節で自動改行（残り文字数チェック付き）
     * @param {string} text - 元のテキスト
     * @param {number} maxLength - 改行する最大文字数（デフォルト: 25）
     * @param {number} minLength - 改行しない最小文字数（デフォルト: 12）
     * @param {number} minRemainingLength - 残り文字が短すぎる場合改行しない閾値（デフォルト: 10）
     * @returns {string} - 改行が挿入されたテキスト
     */
    autoBreakText(text, maxLength = 25, minLength = 12, minRemainingLength = 10) {
      // 30文字以下なら改行不要（表示可能範囲内）
      if (text.length <= 30) return text;
      
      console.log(`[PDM][autoBreakText] Processing: "${text}" (${text.length} chars)`);
      
      const lines = [];
      let current = "";
      let i = 0;
      
      while (i < text.length) {
        current += text[i];
        i++;
        
        // minLength未満なら改行しない（短すぎる行を防ぐ）
        if (current.length < minLength) continue;
        
        // 1行目の場合はmaxLength、2行目以降は30文字を超えたら改行位置を探す
        const isFirstLine = lines.length === 0;
        const breakThreshold = isFirstLine ? maxLength : 30;
        
        // 改行閾値を超えたら、次の自然な区切りを探す
        if (current.length >= breakThreshold) {
          const remaining = text.substring(i);
          
          // 残りが短すぎる場合は改行せずそのまま続ける
          if (remaining.length < minRemainingLength && remaining.length > 0) {
            current += remaining;
            lines.push(current.trim());
            current = ""; // 追加済みなのでクリア
            break;
          }
          
          const lookAhead = text.substring(i, Math.min(i + 20, text.length));
          let breakPos = -1;
          
          // 1. ♡が連続している場合は最後の♡の後で区切る
          const heartMatch = lookAhead.match(/^.{0,15}?♡+/);
          if (heartMatch) {
            breakPos = heartMatch[0].length;
            // ♡の直後に全角スペースや半角スペースがあればそれも含める
            if (lookAhead[breakPos] === '　' || lookAhead[breakPos] === ' ') {
              breakPos++;
            }
          }
          // 2. 句点・感嘆符（連続対応）
          else {
            const punctMatch = lookAhead.match(/^.{0,15}?[。！？]+/);
            if (punctMatch) {
              breakPos = punctMatch[0].length;
              if (lookAhead[breakPos] === '　' || lookAhead[breakPos] === ' ') breakPos++;
            }
          }
          // 3. 読点
          if (breakPos === -1) {
            const commaMatch = lookAhead.match(/^.{0,15}?[、]/);
            if (commaMatch) {
              breakPos = commaMatch[0].length;
              if (lookAhead[breakPos] === '　' || lookAhead[breakPos] === ' ') breakPos++;
            }
          }
          // 4. 濁点付き促音「゛っ♡」のまとまり
          if (breakPos === -1) {
            const tsuMatch = lookAhead.match(/^.{0,10}?[゛っ]+♡+/);
            if (tsuMatch) breakPos = tsuMatch[0].length;
          }
          // 5. 三点リーダーの後
          if (breakPos === -1) {
            const ellipsisMatch = lookAhead.match(/^.{0,10}?…+/);
            if (ellipsisMatch) breakPos = ellipsisMatch[0].length;
          }
          // 6. 助詞の後
          if (breakPos === -1) {
            const particleMatch = lookAhead.match(/^.{0,15}?[がをにはもでからまでと]/);
            if (particleMatch) breakPos = particleMatch[0].length;
          }
          
          // 区切り位置が見つかった場合、その後の残り文字数もチェック
          if (breakPos > 0) {
            const afterBreak = text.substring(i + breakPos);
            
            // 区切った後の残りが短すぎる場合は改行せずに続ける
            if (afterBreak.length < minRemainingLength && afterBreak.length > 0) {
              current += text.substring(i);
              lines.push(current.trim());
              current = ""; // 追加済みなのでクリア
              break;
            }
            
            current += text.substring(i, i + breakPos);
            i += breakPos;
            lines.push(current.trim());
            current = "";
          } else {
            // 区切り位置が見つからない場合は、残りをすべて追加してループを終了
            current += text.substring(i);
            lines.push(current.trim());
            current = ""; // 追加済みなのでクリア
            break;
          }
        }
      }
      
      // whileループを正常に終了した場合（breakしなかった場合）のみ、残りを追加
      if (current.trim()) {
        console.log(`[PDM][autoBreakText] Adding remaining: "${current.trim()}"`);
        lines.push(current.trim());
      }
      
      console.log(`[PDM][autoBreakText] Result: ${lines.length} lines`, lines);
      return lines.join('\\n');
    },

    /** メッセージウィンドウにセリフを表示し、閉じるのを待つ */
    async showMessage(text) {
      // 手動改行が含まれていればそのまま使用、なければ自動改行を適用
      // RPGツクールMZの表示限界: 約30文字/行
      // maxLength: 16文字で改行を探す、minLength: 8文字以上、minRemainingLength: 6文字以上残す
      const finalText = text.includes('\\n') 
        ? text 
        : this.autoBreakText(text, 16, 8, 6);
      
      console.log("[PDM][showMessage] Original:", text);
      console.log("[PDM][showMessage] Final:", finalText);
      console.log("[PDM][showMessage] Has line break:", finalText.includes('\\n'));
      
      // 改行コードで分割して、各行を別々のメッセージとして追加
      const lines = finalText.split('\\n');
      console.log("[PDM][showMessage] Lines:", lines.length, lines);
      
      for (const line of lines) {
        if (line.trim()) {
          $gameMessage.add(line.trim());
        }
      }
      
      const waiter = (typeof window.waitMessageWindow === "function")
        ? window.waitMessageWindow
        : _waitMessageWindowLocal;
      await waiter();
    },

    /** 互換：フォールバックTintを公開（必要な場所で直接呼べる） */
    flashPinkEffect: flashFallbackTint,

    /** 継続的なピンク背景を開始（絶頂・挿入シーン用） */
    startPinkBackground: startPinkBackground,

    /** 継続的なピンク背景を終了 */
    stopPinkBackground: stopPinkBackground,

    clear(actor) {
      if (!actor) return;
      console.log(`[PDM] Clear: actor=${actor.name()}`);
      if (window.BattleBustManager?.clear) window.BattleBustManager.clear(actor);
    },

    clearAll() {
      console.log("[PDM] Clear all battle busts");
      if (window.BattleBustManager?.clearAll) window.BattleBustManager.clearAll();
    }
  };
})();

// --- PleasureDisplayManager.js 末尾 or デバッグ用ブロック ---
/* === Debug Preview: “書いたまま”を即再生 === */
// --- PleasureDisplayManager.js の debug ブロック直後に追加 ---
// 依存：PDM.show / PPC.getProfileNameFromEnemy / loadPoseProfile
// 依存：OrgasmEventHandler.loadOrgasmDetailProfile / getOrgasmDetailSequence
//      InsertEventHandler.loadInsertDetailProfile / getInsertPreSequence / getInsertSequence
//      RestraintEventHandler.loadProfile

(function extendDebugForSpecialEvents(PDM, PPC){
  // 読み込み順の保険：PDM/PPCどちらか無いなら諦める
  PDM = PDM || window.PleasureDisplayManager;
  PPC = PPC || window.PleasurePoseController;
  if (!PDM || !PPC) return;

  // どこから呼ばれても PDM を使えるように
  window.PDM = PDM;

  /* ---------- 共通ヘルパ ---------- */
  function _resolveActor(actorId){
    if (actorId && $gameActors?.actor) return $gameActors.actor(actorId);
    return $gameParty?.leader?.() || ($gameActors?.actor ? $gameActors.actor(1) : null);
  }
  function _resolveProfileName(actor, override){
    if (override) return override;
    return PPC.getProfileNameFromEnemy?.(actor) || "default";
  }

  // ステージ/腐敗/resolve を一時的に強制して runner を実行 → その後すぐ復元
  function _withActorOverrides(actor, {
    stageKey=null,                 // "stage1"|"stage2"|"stage3"
    orgasmCountOverride=null,      // 数値（stageKey と両方指定時はこっち優先）
    corruptionValueOverride=null,  // 数値（0-100想定）
    resolveTone=null               // "resist"|"fall"（stage2/3のみ意味あり）
  }, runner){
    const backups = {};
    try {
      // stageKey → orgasmCount へ変換
      if (orgasmCountOverride == null && stageKey) {
        orgasmCountOverride =
          stageKey === "stage1" ? 0 :
          stageKey === "stage2" ? 1 : 2;
      }
      if (typeof actor.getOrgasmCount === "function" && orgasmCountOverride != null) {
        backups.getOrgasmCount = actor.getOrgasmCount;
        actor.getOrgasmCount = () => Number(orgasmCountOverride);
      }
      if (typeof actor.getCorruption === "function" && corruptionValueOverride != null) {
        backups.getCorruption = actor.getCorruption;
        actor.getCorruption = () => Number(corruptionValueOverride);
      }
      if (resolveTone) {
        const forcedStage =
          stageKey ?? (orgasmCountOverride != null
            ? (orgasmCountOverride >= 2 ? "stage3" : (orgasmCountOverride === 1 ? "stage2" : "stage1"))
            : null);
        if (forcedStage === "stage2" || forcedStage === "stage3") {
          actor._resolveHistory = actor._resolveHistory || {};
          actor._resolveHistory[forcedStage] = resolveTone;
        }
      }
      return runner();
    } finally {
      if (backups.getOrgasmCount) actor.getOrgasmCount = backups.getOrgasmCount;
      if (backups.getCorruption)  actor.getCorruption  = backups.getCorruption;
    }
  }

  // 任意の dialogueSequence を即再生（pose/expression/overlays は各行指定が優先）
  async function _playSequence(actor, profileName, eventType, sequence){
    const displayInfo = {
      profileName,
      eventType: eventType || "react",
      pose: "auto",
      expression: "normal",
      overlays: [],
      dialogueSequence: sequence
    };
    await PDM.show(actor, displayInfo);
  }

  /* ---------- 1) 絶頂デバッグ ---------- */
  // 例) PDM.debugOrgasm({ section:"sequence", stageKey:"stage2", corruptionValueOverride:45 })
  PDM.debugOrgasm = async function({
    actorId=null, profileName=null,
    section="sequence",            // "sequence" | "after_resist" | "after_fall"
    stageKey="stage2",
    corruptionValueOverride=null,  // mid/highテスト用（例: 45/85）
    targetPart="pussy",            // ログ用途のみ
    stepsOverride=null,
    resolveTone=null,              // 念のため指定可（after_*は固定演出だが揃えておく）
    orgasmCountOverride=null
  }={}){
    const actor = _resolveActor(actorId);
    if (!actor) return console.warn("[debugOrgasm] actor not found");

    const prof = _resolveProfileName(actor, profileName);
    await (window.OrgasmEventHandler?.loadOrgasmDetailProfile?.(prof));

    let steps = stepsOverride;
    if (!steps) {
      steps = window.OrgasmEventHandler?.getOrgasmDetailSequence?.(prof, section, stageKey, _corruptionStageFromValue(corruptionValueOverride)) || [];
    }
    console.group("[debugOrgasm]");
    console.log({prof, section, stageKey, corruptionValueOverride, resolveTone, count: steps?.length});
    console.groupEnd();

    return _withActorOverrides(actor, { stageKey, corruptionValueOverride, resolveTone, orgasmCountOverride }, async () => {
      await _playSequence(actor, prof, "orgasm", steps);
    });
  };

  /* ---------- 2) 挿入デバッグ ---------- */
  // 例) PDM.debugInsert({ which:"pre_first", stageKey:"stage2", corruptionValueOverride:40 })
  PDM.debugInsert = async function({
    actorId=null, profileName=null,
    stageKey="stage2",
    corruptionValueOverride=null,
    which="pre_first",                 // "pre_first"|"pre_repeat"|"sequence_first"|"sequence_repeat"|"sequence_resist_success"|"sequence_resist_fail"|"sequence_accept"
    stepsOverride=null,
    resolveTone=null,                  // sequence_* が resist/fall連動する実装に備えて
    orgasmCountOverride=null
  }={}){
    const actor = _resolveActor(actorId);
    if (!actor) return console.warn("[debugInsert] actor not found");

    const prof = _resolveProfileName(actor, profileName);
    await (window.InsertEventHandler?.loadInsertDetailProfile?.(prof));

    let steps = stepsOverride;
    const cStage = _corruptionStageFromValue(corruptionValueOverride);
    if (!steps) {
      if (which === "pre_first" || which === "pre_repeat") {
        const isFirst = (which === "pre_first");
        steps = window.InsertEventHandler?.getInsertPreSequence?.(prof, stageKey, cStage, isFirst) || [];
      } else {
        steps = window.InsertEventHandler?.getInsertSequence?.(prof, stageKey, cStage, true, which)
             || window.InsertEventHandler?.getInsertSequence?.(prof, stageKey, cStage, false, which)
             || [];
      }
    }
    console.group("[debugInsert]");
    console.log({prof, which, stageKey, corruptionValueOverride, resolveTone, count: steps?.length});
    console.groupEnd();

    return _withActorOverrides(actor, { stageKey, corruptionValueOverride, resolveTone, orgasmCountOverride }, async () => {
      await _playSequence(actor, prof, "insert", steps);
    });
  };

  /* ---------- 3) 拘束デバッグ ---------- */
  // 例) PDM.debugRestraint({ countKey:"repeat", stageKey:"stage2", corruptionValueOverride:10 })
  PDM.debugRestraint = async function({
    actorId=null, profileName=null,
    stageKey="stage1",
    corruptionValueOverride=null,
    countKey="first",                 // "first" | "repeat"
    stepsOverride=null,
    resolveTone=null,                 // 互換用（拘束は通常未使用）
    orgasmCountOverride=null
  }={}){
    const actor = _resolveActor(actorId);
    if (!actor) return console.warn("[debugRestraint] actor not found");

    const prof = _resolveProfileName(actor, profileName);
    await (window.RestraintEventHandler?.loadProfile?.(prof));

    const cStage = _corruptionStageFromValue(corruptionValueOverride);

    return _withActorOverrides(actor, { stageKey, corruptionValueOverride, resolveTone, orgasmCountOverride }, async () => {
      if (stepsOverride && Array.isArray(stepsOverride)) {
        await _playSequence(actor, prof, "react", stepsOverride);
        return;
      }
      // 既存の拘束JSON：visuals + lines（単発）
      const p = window._restraintProfileCache?.[prof];
      const visuals = p?.visuals?.default?.[cStage]?.[stageKey] || {};
      const texts   = p?.lines?.default?.[stageKey]?.[cStage]?.[countKey] || [];
      const line = texts[0] || "……";
      await PDM.show(actor, {
        profileName: prof,
        eventType: "react",
        pose: visuals.pose || "auto",
        expression: visuals.expression || "normal",
        overlays: visuals.overlays || [],
        dialogue: line
      });
    });
  };

  /* ---------- 4) 既存 debugPreview を拡張：任意 sequence と stage/腐敗強制 ---------- */
  const _origPreview = typeof PDM.debugPreview === "function" ? PDM.debugPreview.bind(PDM) : null;
  PDM.debugPreview = async function(args = {}){
    const actor = _resolveActor(args.actorId);
    if (!actor) return console.warn("[debugPreview+] actor not found");

    const prof = _resolveProfileName(actor, args.profileName);
    const runner = async () => {
      if (Array.isArray(args.dialogueSequence)) {
        console.group("[DebugPreview+] • dialogueSequence override");
        console.log({eventType: args.eventType || "react", prof, lines: args.dialogueSequence.length});
        console.groupEnd();
        await _playSequence(actor, prof, args.eventType || "react", args.dialogueSequence);
        return;
      }
      if (_origPreview) return _origPreview(args);           // 元実装があれば利用
      // 元が無い場合のフォールバック
      const info = await PPC.getDisplayInfo(actor,
        args.eventType || "react",
        args.targetPart || "pussy",
        args.withDialogue ?? true,
        args.options || { phase:"pre" }
      );
      if (args.profileName) info.profileName = prof;
      console.group("[debugPreview:fallback]");
      console.log("pose:", info.pose, "expr:", info.expression, "overlays:", info.overlays);
      console.groupEnd();
      await PDM.show(actor, info);
    };

    return _withActorOverrides(actor, {
      stageKey: args.stageKey,
      orgasmCountOverride: args.orgasmCountOverride,
      corruptionValueOverride: args.corruptionValueOverride,
      resolveTone: args.resolveTone
    }, runner);
  };

  /* ---------- 補助：腐敗値→low/mid/high 推定 ---------- */
  function _corruptionStageFromValue(v){
    if (v == null) return "low";
    const n = Number(v);
    if (n >= 80) return "high";
    if (n >= 40) return "mid";
    return "low";
  }

})(window.PleasureDisplayManager, window.PleasurePoseController);



