// PleasureFlashFX.js (simple 1-pattern-per-monster, RGBA 8-digit)
(() => {
  console.log("[PleasureFlashFX] Loaded (simple RGBA)");

  // 既定値（すべて最小限）：色は #RRGGBBAA で統一
  const DEFAULT_METHOD   = "screen";      // "screen" or "picture"
  const DEFAULT_COLOR    = "#ffffff40";   // 白色（RGBA8桁）かなり薄く透明、目に優しい
  const DEFAULT_TIMES    = 2;             // 回数
  const DEFAULT_DURATION = 6;            // 1回の長さ（フレーム）適度に滑らか
  const DEFAULT_INTERVAL = 7;             // 回と回の間（フレーム）短く激しく
  const PICTURE_ID       = 98;            // 競合しづらい予約ID
  const BLANK_PICTURE    = "blank";       // img/pictures/blank.png 推奨（1px透明）

  // -------- util --------
  function sleepFrames(f) {
    return new Promise(r => setTimeout(r, Math.max(0, f) * 16));
  }

  // #RGB / #RRGGBB / #RRGGBBAA を受けて [r,g,b,a] に正規化（aは0..255）
  function parseHexColor(hex) {
    const src = String(hex || "").trim().replace("#", "");
    if (!src) return [247, 140, 201, 255]; // fallback: #f78cc9ff

    let s = src.toLowerCase();
    if (s.length === 3) { // #RGB -> #RRGGBB
      s = s.split("").map(c => c + c).join("");
    }
    if (s.length === 6) { // #RRGGBB -> #RRGGBBAA (opaque)
      s = s + "ff";
    }
    // 以降は 8桁前提
    const n = parseInt(s, 16);
    const r = (n >> 24) & 0xff;
    const g = (n >> 16) & 0xff;
    const b = (n >> 8)  & 0xff;
    const a = n & 0xff;
    return [r, g, b, a];
  }

  // -------- low-level renderers --------
  function flashScreen(rgba, duration) {
    // rgba: [r,g,b,a]（0..255）、duration: フレーム
    $gameScreen.startFlash(rgba, duration);
  }

  function flashPicture(rgba, duration) {
    // blank.png を全画面に出して、色味をTintで寄せる簡易オーバーレイ
    const [r, g, b, a] = rgba;
    const w = Graphics.width, h = Graphics.height;

    // ピクチャ表示（opacity は 0..255）
    $gameScreen.showPicture(PICTURE_ID, BLANK_PICTURE, 0, 0, 0, w, h, a, 0);
    // TintPicture はトーン（-255..+255）指定：白基準を色へ寄せる
    // 白(255,255,255) -> (r,g,b) に寄せるため、差分でトーン指定
    $gameScreen.tintPicture(PICTURE_ID, [r - 255, g - 255, b - 255, 0], 0);

    // 一定フレーム後に消す
    setTimeout(() => $gameScreen.erasePicture(PICTURE_ID), duration * 16);
  }

  // -------- profile loader (1体=1パターン) --------
  const _cache = {};
  async function loadProfile(profileName) {
    if (_cache[profileName] !== undefined) return _cache[profileName];
    try {
      const res = await fetch(`dataEx/flashFxProfiles/${profileName}.json`);
      if (!res.ok) throw new Error(res.statusText);
      const json = await res.json();
      _cache[profileName] = json;
      console.log(`[PleasureFlashFX] Loaded profile: ${profileName}`);
      return json;
    } catch (e) {
      console.warn(`[PleasureFlashFX] No profile for ${profileName}; using defaults.`, e);
      _cache[profileName] = null;
      return null;
    }
  }

  // -------- public API --------
  async function play(profileName) {
    console.log("[PleasureFlashFX] ★ play() called with:", profileName);
    
    // 戦闘シーン以外では何もしない（安全弁）
    if (!SceneManager._scene || !(SceneManager._scene instanceof Scene_Battle)) {
      console.log("[PleasureFlashFX] ★ Not in battle scene, skipping");
      return;
    }

    const prof     = await loadProfile(profileName);
    const method   = (prof?.method)   || DEFAULT_METHOD;
    const colorStr = (prof?.color)    || DEFAULT_COLOR;
    const times    = Number(prof?.times ?? DEFAULT_TIMES);
    const duration = Number(prof?.duration ?? DEFAULT_DURATION);
    const interval = Number(prof?.interval ?? DEFAULT_INTERVAL);

    console.log("[PleasureFlashFX] ★ Flash params:", { method, colorStr, times, duration, interval });

    // 8桁カラーから RGBA を抽出（JSONに alpha が残っていたら**それを優先**）
    const rgbaParsed = parseHexColor(colorStr);
    const finalA = (prof && "alpha" in prof) ? Number(prof.alpha) : rgbaParsed[3];
    const rgba = [rgbaParsed[0], rgbaParsed[1], rgbaParsed[2], Math.max(0, Math.min(255, finalA))];

    console.log("[PleasureFlashFX] ★ Final RGBA:", rgba);

    // リッチな演出：強度を変えた3段階のフラッシュ
    const flashPattern = [
      { intensity: 0.6, duration: 5, interval: 6 },   // 1回目：控えめ
      { intensity: 0.8, duration: 6, interval: 6 },   // 2回目：中くらい
      { intensity: 1.0, duration: 8, interval: 0 }    // 3回目：最も強く
    ];

    for (let i = 0; i < flashPattern.length; i++) {
      const pattern = flashPattern[i];
      console.log(`[PleasureFlashFX] ★ Flash ${i + 1}/${flashPattern.length} (intensity: ${pattern.intensity})`);
      
      // 強度に応じてアルファ値を調整
      const adjustedRgba = [
        rgba[0],
        rgba[1],
        rgba[2],
        Math.round(rgba[3] * pattern.intensity)
      ];
      
      // Flashを開始
      if (method === "picture") {
        flashPicture(adjustedRgba, pattern.duration);
      } else {
        flashScreen(adjustedRgba, pattern.duration);
      }
      
      // durationフレーム待つ（フェードアウトが完了するまで）
      await sleepFrames(pattern.duration);
      console.log(`[PleasureFlashFX] ★ Flash ${i + 1}/${flashPattern.length} fade complete`);
      
      // 明示的にflashをクリア
      $gameScreen.clearFlash();
      console.log(`[PleasureFlashFX] ★ Flash cleared`);
      
      // intervalがある場合は暗い時間を作る
      if (pattern.interval > 0) {
        await sleepFrames(pattern.interval);
        console.log(`[PleasureFlashFX] ★ Interval complete`);
      }
    }
    
    console.log("[PleasureFlashFX] ★ Flash complete (rich pattern)");
  }

  window.PleasureFlashFX = { play };
  window.PleasureFlashFX.startFlash = function(profileName) {
  return window.PleasureFlashFX.play(profileName);
  };
})();
