// PleasureIntroManager.js
(() => {
  console.log("[PleasureIntroManager] Loaded");

  if (!window.PleasureIntroManager) window.PleasureIntroManager = {};
  if (!window._introProfileCache) window._introProfileCache = {};

  /**
   * 指定されたプロファイルとeventTypeに基づくイントロJSONを読み込む
   */
  PleasureIntroManager.loadIntroProfile = async function(profileKey, introEventType) {
    if (!window._introProfileCache[introEventType]) window._introProfileCache[introEventType] = {};
    if (window._introProfileCache[introEventType][profileKey]) return;

    const filePath = `dataEx/introProfiles/${introEventType}/${profileKey}.json`;
    try {
      const response = await fetch(filePath);
      if (!response.ok) throw new Error("HTTP error " + response.status);
      const jsonData = await response.json();
      window._introProfileCache[introEventType][profileKey] = jsonData;
      console.log(`[IntroLoader] Loaded intro profile: ${introEventType}/${profileKey}`);
    } catch (error) {
      console.error(`[IntroLoader] Failed to load intro profile: ${introEventType}/${profileKey}`, error);
    }
  };

  /**
   * 対象の構造からランダムなイントロ文を選出
   */
  PleasureIntroManager.selectIntroLine = function(profileKey, introEventType, pleasureStageKey, corruptionStageKey) {
    const introProfile = window._introProfileCache?.[introEventType]?.[profileKey];
    if (!introProfile) {
      console.warn(`[IntroSelect] No profile loaded: ${introEventType}/${profileKey}`);
      return null;
    }

    let candidateLines = introProfile?.[pleasureStageKey]?.[corruptionStageKey];

    // fallback
    if (!candidateLines || !Array.isArray(candidateLines) || candidateLines.length === 0) {
      candidateLines = introProfile?.["stage1"]?.[corruptionStageKey] || introProfile?.["stage2"]?.["low"] || [];
      if (!candidateLines || candidateLines.length === 0) {
        console.warn(`[IntroSelect] No fallback intro lines found: ${introEventType}/${profileKey}`);
        return null;
      }
    }

    const selectedLine = candidateLines[Math.floor(Math.random() * candidateLines.length)];
    console.log(`[IntroSelect] Selected intro: ${selectedLine}`);
    return selectedLine;
  };

  /**
   * アクターとイベント種別に応じてイントロ文を表示し、完了まで待つ
   */
  PleasureIntroManager.showIntroLine = async function(targetActor, introEventType) {
    
    if (!targetActor || !introEventType) return;
    console.log(`[IntroDebug] showIntroLine called: actor=${targetActor.name()}, eventType=${introEventType}`);

    const stageKey = window.PleasurePoseController.determinePleasureStage?.(targetActor.getOrgasmCount()) || "stage1";
    const corruptionStage = window.determineStageByCorruption?.(targetActor.getCorruption?.()) || "low";
    const profileIdentifier = window.PleasurePoseController.getProfileNameFromEnemy(targetActor);

    console.log(`[IntroDebug] stageKey=${stageKey}, corruptionStage=${corruptionStage}, profile=${profileIdentifier}`);

    await this.loadIntroProfile(profileIdentifier, introEventType);
    const selectedLine = this.selectIntroLine(profileIdentifier, introEventType, stageKey, corruptionStage);

    if (selectedLine) {
      console.log(`[IntroDebug] Displaying intro line: ${selectedLine}`);
      await new Promise(resolve => setTimeout(resolve, 30));
      $gameMessage.add(selectedLine);
      await new Promise(resolve => {
        const checkIntroDone = setInterval(() => {
          if (!$gameMessage.isBusy()) {
            clearInterval(checkIntroDone);
            resolve();
          }
        }, 10);
      });
    }else{
      console.warn(`[IntroDebug] No intro line selected for ${profileIdentifier}`);
    }
  };
})();
