// ResistanceCore.js (standard rule integrated)
// 仕様：この戦闘中、挿入抵抗に成功するたびに「基礎成功率」を10ptずつ低下（下限0%）
(() => {
  console.log("[ResistanceCore] Loaded (with per-battle success penalty)");

  // ===== ユーティリティ =====
  const clamp = (v, lo, hi) => Math.max(lo, Math.min(hi, v));

  // ===== 基本定数（%を小数で保持）=====
  const BASE_RATE_DECIMAL      = 0.60; // 60%
  const CORRUPTION_MAX_PEN     = 0.30; // 屈服 最大 -30%pt（0..100 → 0..-0.30）
  const DURABILITY_MAX_BON     = 0.10; // 衣装耐久 最大 +10%pt

  // ===== 標準ルール：成功ペナルティ設定（％ポイント単位）=====
  const PENALTY_PER_SUCCESS_PT = 10;   // 1成功ごとに -10pt
  const MAX_PENALTY_STACKS     = 99;   // 念のための上限

  // ===== 内部：この戦闘での成功回数を俳優に保持 =====
  function getSuccessCountThisBattle(actor) {
    return Math.max(0, Number(actor?._insertResistSuccessCountThisBattle ?? 0) | 0);
  }
  function setSuccessCountThisBattle(actor, count) {
    if (!actor) return;
    actor._insertResistSuccessCountThisBattle = Math.max(0, count|0);
  }
  function addSuccessThisBattle(actor) {
    setSuccessCountThisBattle(actor, getSuccessCountThisBattle(actor) + 1);
  }
  function resetSuccessThisBattle(actor) {
    setSuccessCountThisBattle(actor, 0);
  }

  // 戦闘開始で自動リセット（既存 onBattleStart に安全ラップ）
  const _Game_Actor_onBattleStart = Game_Actor.prototype.onBattleStart;
  Game_Actor.prototype.onBattleStart = function() {
    _Game_Actor_onBattleStart.call(this);
    resetSuccessThisBattle(this);
  };

  // ===== 公開：成功回数→ペナルティ％、補正後の基礎％ =====
  function getPenaltyPercent(actor) {
    const stacks = Math.min(getSuccessCountThisBattle(actor), MAX_PENALTY_STACKS);
    return stacks * PENALTY_PER_SUCCESS_PT; // 0,10,20,...
  }
  function getAdjustedBasePercent(actor) {
    const basePercent = Math.round((typeof BASE_RATE_DECIMAL === "number" ? BASE_RATE_DECIMAL : 0.6) * 100);
    const after = Math.max(0, basePercent - getPenaltyPercent(actor));
    return after; // 0..100
  }

  // ===== コア：挿入抵抗成功率（0.05..0.95にクリップ）=====
  function calcInsertResistRate(actor) {
    // 1) 基礎（成功ペナルティ適用済み％ → 小数へ）
    let rate = clamp(getAdjustedBasePercent(actor) / 100, 0, 1);

    // 2) 屈服ペナルティ（0..100 → 0..-0.30）
    const corruption = actor?.getCorruption?.() ?? 0;
    rate -= clamp((corruption / 100) * CORRUPTION_MAX_PEN, 0, CORRUPTION_MAX_PEN);

    // 3) 衣装耐久ボーナス（0..100 → 0..+0.10）
    if (typeof actor?.getClothingDurability === "function") {
      const durabilityRate = clamp(actor.getClothingDurability() / 100, 0, 1);
      rate += durabilityRate * DURABILITY_MAX_BON;
    }

    // 4) 最終クリップ
    return clamp(rate, 0.05, 0.95);
  }

  // ===== UI向け：説明行を返す（InsertEventHandler があれば利用）=====
  // 返り値は配列（"基礎…", "屈服…", "衣装…", "＝ 合計…"）形式
  function explainInsertResist(actor) {
    const basePercent = Math.round(BASE_RATE_DECIMAL * 100);
    const penaltyPct  = getPenaltyPercent(actor);
    const adjusted    = getAdjustedBasePercent(actor);

    const corrRaw   = clamp((actor?.getCorruption?.() ?? 0), 0, 100);
    const corrPct   = Math.round(corrRaw * (CORRUPTION_MAX_PEN)); // 例：100 * 0.30 = 30
    const duraRaw   = typeof actor?.getClothingDurability === "function"
                    ? clamp(actor.getClothingDurability(), 0, 100) : 0;
    const duraPct   = Math.round((duraRaw / 100) * (DURABILITY_MAX_BON * 100)); // 0..10

    const rate      = Math.round(calcInsertResistRate(actor) * 100);

    const lines = [];
    // 基礎は「調整後基礎％（成功ペナ −X% を含む）」として表示
    lines.push(`基礎 ${adjusted}%（成功ペナ −${penaltyPct}%）`);
    lines.push(`堕落 −${corrPct}%（現在${Math.round(corrRaw)}）`);
    if (duraPct !== 0) lines.push(`衣装耐久 +${duraPct}%（現在${Math.round(duraRaw)}%）`);
    lines.push(`＝ 合計 ≈ ${rate}%`);
    return lines;
  }

  // ===== サマリ：外部から使うAPI =====
  window.ResistanceCore = {
    calcInsertResistRate,
    roll: (rate) => Math.random() < rate,
    // 標準ルールAPI
    noteInsertResistSuccess: addSuccessThisBattle,           // 抵抗成功時に1回呼ぶ
    getInsertResistSuccessCount: getSuccessCountThisBattle,  // 表示・デバッグ用
    getInsertResistPenaltyPercent: getPenaltyPercent,        // 表示用
    getAdjustedInsertBasePercent: getAdjustedBasePercent,    // 表示用
    explainInsertResist,                                     // UIの内訳表示用
    // 定数は参照できるように公開（互換性維持）
    _const: {
      BASE_RATE: BASE_RATE_DECIMAL,
      CORRUPTION_MAX_PEN,
      DURABILITY_MAX_BON
    }
  };
})();
