/*:
 * @target MZ
 * @plugindesc 内部用：抵抗判定ミニUI（1D100回転→確定表示／コードからのみ呼び出し）v1.0
 * @author you
 * @help
 * 本プラグインは「見せ方だけ」を担当します。プラグインコマンドはありません。
 * 判定（成功率・出目・成否）は ResistanceCore 側で決定し、ここはそれを可視化するだけ。
 *
 * 公開API:
 *   await ResistanceRollMiniUI.showAsync({
 *     rolled: 42,                // 1..100（Coreでの結果に整合する値）
 *     successRate: 47,           // 0..100（閾値%）
 *     success: true,             // 成否
 *     actorName: "セリア",       // 任意
 *     checkLabel: "抵抗判定（挿入）", // 見出しテキスト
 *     detailLines: ["基礎60%","屈服 -18%（現在60）","衣装耐久 +5%","＝ 合計 ≈ 47%"], // 任意・最大4行程度
 *     maskBackground: false,     // 半透明マスク
 *     diceFontSize: 28,
 *     spinFrames: 28,
 *     spinIntervalMs: 40,
 *     seStart: "Cursor2",
 *     seStop: "Decision3",
 *     logToBattleLog: true
 *   });
 */
var ResistanceRollMiniUI = ResistanceRollMiniUI || {};
(() => {
  "use strict";

  const clamp = (v, lo, hi) => Math.max(lo, Math.min(hi, v));
  

  class Window_ResistanceRollMini extends Window_Base {
    initialize(rect) {
      super.initialize(rect);
      this.openness = 0;
      this._actorName = "";
      this._successRate = 50;   // 0..100
      this._rolled = 0;         // 1..100
      this._finalized = false;
      this._isSuccess = false;
      this._diceFontSize = 28;
      this._checkLabel = "抵抗判定（1D100）";
      this._detailLines = [];
    }
    setDisplayPayload(p) {
      this._actorName    = p.actorName || "";
      this._successRate  = clamp(Math.round(p.successRate || 0), 0, 100);
      this._rolled       = 0;
      this._finalized    = false;
      this._isSuccess    = false;
      this._diceFontSize = clamp(p.diceFontSize || 28, 18, 48);
      this._checkLabel   = p.checkLabel || "抵抗判定（1D100）";
      this._detailLines  = Array.isArray(p.detailLines) ? p.detailLines.slice(0, 4) : [];
      this.refresh();
    }
    setRollingValue(v) {
      this._rolled = clamp(v, 1, 100);
      this.refresh();
    }
    finalizeResult(finalRoll, isSuccess) {
      this._rolled = clamp(finalRoll, 1, 100);
      this._isSuccess = !!isSuccess;
      this._finalized = true;
      this.refresh();
    }
  refresh() {
  const c = this.contents;
  c.clear();

  const lh = this.lineHeight();
  let y = 0;

  // 見出し
  this.changeTextColor(ColorManager.systemColor());
  c.drawText(this._checkLabel, 0, y, this.contentsWidth(), lh, "center");
  this.resetTextColor();
  y += lh;

  // 内訳（必要ならその下に表示）
  for (const line of this._detailLines) {
    c.drawText(line, 0, y, this.contentsWidth(), lh, "center");
    y += lh;
  }
    // 成功率とロールをまとめて表示
    this.contents.fontSize = this._diceFontSize;
    const rollText = this._rolled ? `${this._rolled}` : "??";
    const combined = `成功率 ${this._successRate}% ／ ロール ${rollText}`;
    c.drawText(combined, 0, y, this.contentsWidth(), lh, "center");
    this.resetFontSettings();
    y += lh;

  // 結果
  if (this._finalized) {
    this.changeTextColor(this._isSuccess ? ColorManager.systemColor() : ColorManager.crisisColor());
    c.drawText(this._isSuccess ? "抵抗成功！" : "抵抗失敗…", 0, y + 4, this.contentsWidth(), lh, "center");
    this.resetTextColor();
  }
}

  }

  function addMaskUnderWindowLayer(scene) {
    const maskSprite = new Sprite();
    maskSprite.bitmap = new Bitmap(Graphics.width, Graphics.height);
    maskSprite.bitmap.fillAll("rgba(0,0,0,0.4)");
    maskSprite.interactive = false;   // 入力拾わない
    maskSprite.touchable   = false;   // 念のため

    const windowLayer = scene._windowLayer;
    if (windowLayer && windowLayer.parent === scene) {
      // ウィンドウレイヤー直下に差し込み（常にウィンドウの“下”）
      const layerIndex = scene.getChildIndex(windowLayer);
      scene.addChildAt(maskSprite, Math.max(0, layerIndex));
    } else if (scene._spriteset) {
      // 退避策：スプライトセット（戦闘画面の絵）より上、ウィンドウより下
      scene._spriteset.addChild(maskSprite);
    } else {
      // 最後の保険：とりあえず追加（この場合でも windowLayer 追加後に再描画で上に出る）
      scene.addChild(maskSprite);
    }
    try { if (scene._windowLayer?.parent === scene) scene.addChild(scene._windowLayer); } catch (_) {}
    return maskSprite;
  }

  async function showRollWindowAsync(options) {
    const scene = SceneManager._scene;
    if (!scene) {
      return { success: !!options.success, rolled: options.rolled, threshold: options.successRate };
    }

    let maskSprite = null;
    if (options.maskBackground) maskSprite = addMaskUnderWindowLayer(scene);

    // ウィンドウ生成/再利用 + 高さ調整（内訳行数に応じて）
    const winWidth = 620;
    const baseLines = 3; // 見出し/成功率/メーター/ロール
    const extra = (options.detailLines?.length || 0);
    const winHeight = (baseLines + extra) * Window_Base.prototype.lineHeight.call(scene) + 38;

    if (!scene._resistMiniWindow) {
      const rect = new Rectangle(0, 177, winWidth,winHeight);
      scene._resistMiniWindow = new Window_ResistanceRollMini(rect);
      scene.addWindow(scene._resistMiniWindow);
    } else {
      scene._resistMiniWindow.move(
        0,
        177,
        winWidth,
        winHeight
      );
      scene._resistMiniWindow.createContents(); // ★これが重要
    }

    const win = scene._resistMiniWindow;
    win.setDisplayPayload({
      actorName: options.actorName || "",
      successRate: options.successRate,
      diceFontSize: options.diceFontSize,
      checkLabel: options.checkLabel,
      detailLines: options.detailLines || []
    });
    win.open();
  // 一拍置いてウィンドウを最前面に（初回だけマスクに負ける環境対策）
    requestAnimationFrame(() => {
      try {
        if (scene._windowLayer && win.parent === scene._windowLayer) {
          scene._windowLayer.addChild(win); // 末尾=最前面
        }
      } catch (_) {}
    });

    if (options.seStart) AudioManager.playSe({ name: options.seStart, volume: 60, pitch: 100, pan: 0 });

    // 停止→確定表示
    const finalRoll = clamp(Math.round(options.rolled || 1), 1, 100);
    const isSuccess = !!options.success;
    if (options.seStop) AudioManager.playSe({ name: options.seStop, volume: 90, pitch: 100, pan: 0 });
    win.finalizeResult(finalRoll, isSuccess);

    if (options.logToBattleLog && window.BattleLogManager?.push) {
      BattleLogManager.push(`抵抗判定：${isSuccess ? "成功" : "失敗"}（≤${options.successRate} / 出目${finalRoll}）`);
    }

    // ResistanceRollMiniUI.js 内 showRollWindowAsync の末尾を修正
    await new Promise(r => {
      // ① エンターキーで閉じる
      const handler = e => {
        if (e.code === "Enter" || e.code === "Space") {
          window.removeEventListener("keydown", handler);
          win.close();
          if (maskSprite) scene.removeChild(maskSprite);
          r();
        }
      };
      window.addEventListener("keydown", handler);

      // ② 一定時間経過でも閉じる（例：2.5秒）
      setTimeout(() => {
        window.removeEventListener("keydown", handler);
        win.close();
        if (maskSprite) scene.removeChild(maskSprite);
        r();
      }, 5000);
    });

    return { success: isSuccess, rolled: finalRoll, threshold: options.successRate };
  }

  ResistanceRollMiniUI.showAsync = function(userOptions = {}) {
    const opts = Object.assign({
      rolled: 50,
      successRate: 50,
      success: false,
      actorName: "",
      checkLabel: "抵抗判定（1D100）",
      detailLines: [],
      maskBackground: false,
      diceFontSize: 28,
      spinFrames: 28,
      spinIntervalMs: 40,
      seStart: "Cursor2",
      seStop: "Decision3",
      logToBattleLog: true
    }, userOptions);

    opts.rolled = clamp(Number(opts.rolled) || 1, 1, 100);
    opts.successRate = clamp(Number(opts.successRate) || 0, 0, 100);

    return showRollWindowAsync(opts);
  };
})();
