// RestraintEventHandler.js（REH：underOverlays対応 + レース緩和＆二段描画・PDM一本化・visuals二方式対応）
(() => {
  console.log("[RestraintEventHandler] Loaded (hardened + underOverlays + dual-visuals)");

  if (!window.RestraintEventHandler) window.RestraintEventHandler = {};

  // ==========================
  // 小さなユーティリティ
  // ==========================
  function resolveAutoPose(actor, candidatePose) {
    if (candidatePose && candidatePose !== "auto") return candidatePose;
    const clothingStage = actor.getClothingStage?.() || "intact";
    const clothingType  = actor.getClothingType?.() ?? "uniform";
    return `${clothingType}_${clothingStage}`;
  }
  function normalizeOverlayKey(key) {
    return (typeof key === "string") ? key.replace(/^p(\d+):/, "p$1_") : key;
  }
  function actorIdOf(actor) {
    return actor?.actorId?.() ?? actor?.actorId ?? 0;
  }

  // ==========================
  // REH内だけの直列キュー（actor単位）
  // ==========================
  const __rehQueue__ = new Map();         // actorId -> Promise
  const __rehToken__ = new Map();         // actorId -> number（拘束描画の世代）
  function enqueueRehTask(actorId, task) {
    const prev = __rehQueue__.get(actorId) || Promise.resolve();
    const next = prev.then(task).catch(()=>{}); // エラーで詰まらないよう潰す
    __rehQueue__.set(actorId, next);
    return next;
  }
  function bumpRehToken(actorId) {
    const n = (__rehToken__.get(actorId) || 0) + 1;
    __rehToken__.set(actorId, n);
    return n;
  }
  function isRehTokenCurrent(actorId, token) {
    return (__rehToken__.get(actorId) || 0) === token;
  }

  /**
   * 拘束イベント発生処理（REHのみでの安定化）
   * - visuals 二方式に対応（legacy/new）
   * - pose:auto を衣服状態に解決
   * - REH内キューで直列化
   * - 1フレーム後に“もう一度だけ”上書き（他経路の遅延上書きに勝つため）
   */
  RestraintEventHandler.handle = async function(actor) {
    if (!actor || !actor.isActor || !actor.isActor()) return;

    const profileName     = PleasurePoseController.getProfileNameFromEnemy(actor);
    const stageKey        = PleasurePoseController.determinePleasureStage(actor.getOrgasmCount()) || "stage1";
    const corruptionStage = determineStageByCorruption(actor.getCorruption());

    await this.loadProfile(profileName);
    const profile = window._restraintProfileCache?.[profileName];
    if (!profile) {
      console.warn("[REH] profile not found:", profileName);
      return;
    }

    // 拘束回数分岐
    const restraintCount = actor._pleasure?.restraintCount || 0;
    const countKey = restraintCount <= 1 ? "first" : "repeat";

    // 台詞（任意）
    const textList = profile.lines?.default?.[stageKey]?.[corruptionStage]?.[countKey] || [];
    const dialogue = textList[Math.floor(Math.random() * textList.length)] || "……";

    // visuals（preのみ採用。無ければ安全デフォルトへ）— 両方式対応
    const visuals = profile.visuals || {};
    // ① 旧方式: visuals.default.<corruptionStage>.<stageKey>.pre
    const preFromLegacy =
      visuals?.default?.[corruptionStage]?.[stageKey]?.pre ?? null;
    // ② 新方式: visuals.<stageKey>.(react|danger|climax|down_attack).pre のどれか
    const stageBranch = visuals?.[stageKey] || {};
    const preFromNewStyle =
      stageBranch?.react?.pre
      || stageBranch?.danger?.pre
      || stageBranch?.climax?.pre
      || stageBranch?.down_attack?.pre
      || null;

    const preNode = preFromLegacy ?? preFromNewStyle;

    // 安全デフォルト（初回拘束で確実に restrain を出す）
    // ★ under_attack は明示的に指定された場合のみ使用（デフォルトでは空配列）
    const defaultBlush = (stageKey === "stage1") ? "blush_mid" : "blush_high";
    let basePose        = preNode?.pose          ?? "auto";
    let baseExpression  = preNode?.expression    ?? "endure";
    let baseOverlays    = preNode?.overlays      ?? ["restrain", defaultBlush];
    let baseUnder       = preNode?.underOverlays ?? [];
    let baseSe          = preNode?.se            ?? null;

    if (window.OverlayResolver?.resolveOverlays) {
      baseOverlays = window.OverlayResolver.resolveOverlays(baseOverlays);
      baseUnder    = window.OverlayResolver.resolveOverlays(baseUnder);
    }
    baseOverlays = (baseOverlays || []).map(normalizeOverlayKey);
    baseUnder    = (baseUnder    || []).map(normalizeOverlayKey);

    basePose = resolveAutoPose(actor, basePose);
    if (!basePose) basePose = "uniform_intact";

    // 表示パケット（PDMに渡す）
    const displayInfo = {
      profileName,
      pose: basePose,
      expression: baseExpression,
      overlays: baseOverlays,
      underOverlays: baseUnder,
      dialogue,
      log: "",
      se: baseSe,

      // PDM側に伝えるヒント（PDMは無改変でも大半が効く）
      _phase: "pre",
      resetTransient: true, // 非永続レイヤをクリアして欲しい
      forceRedraw: true,    // 同一でも描き直して欲しい
      fallbackExpressionForSequence: baseExpression,
      fallbackOverlaysForSequence: baseOverlays,
      // under側のフォールバックも提示（PDMが見るなら利用される）
      fallbackUnderOverlaysForSequence: baseUnder
    };

    const aid   = actorIdOf(actor);
    const token = bumpRehToken(aid);

    // --- REH内だけ直列化して描画 ---
    await enqueueRehTask(aid, async () => {
      // 1発目：通常描画（PDM経由）
      if (window.PleasureDisplayManager?.show) {
        await window.PleasureDisplayManager.show(actor, displayInfo);
      }

      // 2発目：1フレーム後に“もう一度だけ”上書き（他経路が遅れて上書きした場合に勝ち直す）
      await new Promise(r => setTimeout(r, 16)); // ≒1frame
      // まだこの拘束描画が「最新」で、かつ拘束状態が続いているときだけ再適用
      const stillCurrent    = isRehTokenCurrent(aid, token);
      const stillRestrained = window.PleasureStateManager?.isActorRestrained?.(actor) ?? true;

      if (stillCurrent && stillRestrained && window.PleasureDisplayManager?.show) {
        await window.PleasureDisplayManager.show(actor, {
          ...displayInfo,
          dialogue: "",            // 2発目は台詞を抑止（重複発話を避ける）
          dialogueSequence: null,  // 念のため
          se: null,                // 念のためSEも抑止
          forceRedraw: true
        });
      }
    });
  };

  /**
   * JSON読み込み（そのまま）
   */
  RestraintEventHandler.loadProfile = async function(profileName) {
    if (!window._restraintProfileCache) window._restraintProfileCache = {};
    if (window._restraintProfileCache[profileName]) return;

    try {
      const res = await fetch(`dataEx/restraintProfiles/${profileName}.json`);
      if (!res.ok) throw new Error("HTTP error " + res.status);
      const json = await res.json();
      window._restraintProfileCache[profileName] = json;
      console.log(`[RestraintProfile] Loaded: ${profileName}`);
    } catch (e) {
      console.warn(`[RestraintProfile] Failed to load for ${profileName}`, e);
    }
  };
})();
