// VisualAutoFill.js - 表情とオーバーレイの自動決定システム（統合層）
// ExpressionSelector と OverlayBuilder を統合して使いやすいAPIを提供
(() => {
  console.log("[VisualAutoFill] Loading (integration layer)...");

  // =============================================================================
  // 依存関係チェック
  // =============================================================================
  
  if (!window.ExpressionSelector) {
    console.error("[VisualAutoFill] ExpressionSelector not loaded! Please load ExpressionSelector.js first.");
    return;
  }
  
  if (!window.OverlayBuilder) {
    console.error("[VisualAutoFill] OverlayBuilder not loaded! Please load OverlayBuilder.js first.");
    return;
  }

  // =============================================================================
  // 統合API: dialogueSequence の自動補完
  // =============================================================================
  
  /**
   * dialogueSequence 全体を自動補完
   * @param {Array} sequence - dialogueSequence配列
   * @param {Object} context - コンテキスト情報（forceExpressionオプション対応）
   * @returns {Array} - 補完されたsequence
   */
  function autoFillDialogueSequence(sequence, context) {
    if (!Array.isArray(sequence) || sequence.length === 0) {
      return sequence;
    }
    
    const { stage, eventType, forceExpression } = context;
    const totalFrames = sequence.length;
    
    let currentExpression = null;
    
    return sequence.map((line, frameIndex) => {
      const progress = frameIndex / Math.max(1, totalFrames - 1); // 0.0 → 1.0
      
      // ===== 表情の決定 =====
      // ★強制表情が指定されている場合は常にそれを使用（抵抗成功時など）
      if (forceExpression) {
        currentExpression = forceExpression;
      } else if (line.expression) {
        // 明示指定があればそれを使う
        currentExpression = line.expression;
      } else {
        // 自動決定（台詞内容を考慮）
        const dialogueText = line.text || line.dialogue || "";
        
        if (currentExpression) {
          // 前の表情から遷移（台詞を考慮）
          currentExpression = window.ExpressionSelector.getNextExpression(
            currentExpression, 
            stage, 
            eventType
          );
        } else {
          // 初回は重み付きランダム（台詞を考慮）
          currentExpression = window.ExpressionSelector.weightedRandomExpression(
            stage, 
            eventType, 
            dialogueText, 
            context.isRestrained
          );
        }
      }
      
      // ===== オーバーレイの決定 =====
      let overlays = [];
      let underOverlays = [];
      
      if (line.overlays) {
        // 明示指定があればそのまま
        overlays = [...line.overlays];
      } else {
        // 自動構築（★フレーム情報を渡す）
        const autoOverlays = window.OverlayBuilder.buildOverlaysForLine(
          context, 
          progress, 
          frameIndex,      // フレームインデックス
          totalFrames      // 総フレーム数
        );
        overlays = autoOverlays.overlays || [];
        underOverlays = autoOverlays.underOverlays || [];
        
        // ★強度情報をcontextに追加（SFX選択用）
        const contextWithIntensity = {
          ...context,
          currentIntensity: autoOverlays.currentIntensity
        };
        
        // 擬音の追加（オプション）
        // ※既存の擬音オーバーレイがある場合は追加しない
        const hasBubble = overlays.some(o => 
          typeof o === "string" && (o.includes("あっ") || o.includes("んん") || o.includes("はぁ") || o.includes("default_pose_overlay"))
        );
        
        // ★行単位のSFX抑制チェック（line.suppressSfx または context.suppressSfx）
        const shouldSuppressSfx = line.suppressSfx === true || context.suppressSfx === true;
        
        if (!hasBubble && !shouldSuppressSfx) {
          const allBubbles = [];
          const categoriesProcessed = new Set();
          
          // 基本SFXのカテゴリを判定
          const { isInserted, targetPart } = context;
          const mainCategory = isInserted ? "inserted" : (targetPart === "breast" ? "breast" : "other");
          
          // 基本SFXを取得
          const bubble = window.OverlayBuilder.getBubbleOverlaysFromContext(contextWithIntensity, progress);
          if (bubble) {
            if (Array.isArray(bubble)) {
              allBubbles.push(...bubble);
            } else {
              allBubbles.push(bubble);
            }
            categoriesProcessed.add(mainCategory);
            console.log(`[VisualAutoFill] Added base SFX for category: ${mainCategory}`);
          }
          
          // ★追加のSFXカテゴリ（挿入時の胸SFXなど）
          // 同じカテゴリは重複しないようにスキップ
          if (autoOverlays.additionalSfxCategories && autoOverlays.additionalSfxCategories.length > 0) {
            for (const sfxInfo of autoOverlays.additionalSfxCategories) {
              // 既に処理済みのカテゴリはスキップ
              if (categoriesProcessed.has(sfxInfo.category)) {
                console.log(`[VisualAutoFill] Skipped duplicate SFX category: ${sfxInfo.category}`);
                continue;
              }
              
              // forcedCategoryを使用して明示的にカテゴリを指定
              const additionalBubble = window.OverlayBuilder.getBubbleOverlaysFromContext(
                contextWithIntensity, 
                progress, 
                sfxInfo.category  // ★forcedCategoryパラメータ
              );
              if (additionalBubble) {
                if (Array.isArray(additionalBubble)) {
                  allBubbles.push(...additionalBubble);
                } else {
                  allBubbles.push(additionalBubble);
                }
                categoriesProcessed.add(sfxInfo.category);
                console.log(`[VisualAutoFill] Added additional SFX for category: ${sfxInfo.category}`);
              }
            }
          }
          
          // 重複排除して追加
          const uniqueBubbles = [...new Set(allBubbles)];
          overlays.push(...uniqueBubbles);
          
          if (uniqueBubbles.length > 0) {
            console.log(`[VisualAutoFill] Added SFX bubbles (${uniqueBubbles.length} unique from ${categoriesProcessed.size} categories): ${uniqueBubbles.join(", ")}`);
          }
        }
      }
      
      // underOverlaysも明示指定がなければ自動値を使用
      if (line.underOverlays) {
        underOverlays = [...line.underOverlays];
      }
      
      return {
        ...line,
        expression: currentExpression,
        overlays,
        underOverlays,
        pose: line.pose || "auto"
      };
    });
  }

  // =============================================================================
  // 統合API: 単一ビジュアルノードの自動補完（main/pre用）
  // =============================================================================
  
  /**
   * 単一ビジュアルノードを自動補完
   * @param {Object} context - コンテキスト情報
   * @param {number} progress - 進行度（0.0-1.0）
   * @param {string} dialogueText - 台詞テキスト（オプション）
   * @returns {Object} - { expression, overlays, underOverlays }
   */
  function autoFillSingleVisual(context, progress = 0.5, dialogueText = null) {
    const { stage, eventType } = context;
    
    const expression = window.ExpressionSelector.weightedRandomExpression(
      stage, 
      eventType, 
      dialogueText, 
      context.isRestrained
    );
    
    const autoOverlays = window.OverlayBuilder.buildOverlaysForLine(context, progress);
    
    return {
      expression,
      overlays: autoOverlays.overlays || [],
      underOverlays: autoOverlays.underOverlays || []
    };
  }

  // =============================================================================
  // グローバルAPIの公開
  // =============================================================================
  
  window.VisualAutoFill = {
    // ===== 統合API =====
    autoFillDialogueSequence,
    autoFillSingleVisual,
    
    // ===== 表情関連（ExpressionSelector から再エクスポート） =====
    weightedRandomExpression: window.ExpressionSelector.weightedRandomExpression,
    getNextExpression: window.ExpressionSelector.getNextExpression,
    
    // ===== オーバーレイ関連（OverlayBuilder から再エクスポート） =====
    buildOverlaysForLine: window.OverlayBuilder.buildOverlaysForLine,
    getAttackOverlays: window.OverlayBuilder.getAttackOverlays,
    getBubbleOverlaysFromContext: window.OverlayBuilder.getBubbleOverlaysFromContext,
    
    // ===== データ（後方互換性のため） =====
    BLUSH_BY_STAGE: window.OverlayBuilder.BLUSH_BY_STAGE,
    EXPRESSION_WEIGHTS: window.ExpressionSelector.EXPRESSION_WEIGHTS,
    EXPRESSION_TRANSITIONS: window.ExpressionSelector.EXPRESSION_TRANSITIONS
  };

  console.log("[VisualAutoFill] Ready - Integration layer initialized");
  console.log("[VisualAutoFill] Loaded modules: ExpressionSelector, OverlayBuilder");
})();
