#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
全てのJSONファイルのoverlaysにblush系が無い場合、blush_lowを追加するスクリプト
"""
import json
import os
import sys
from pathlib import Path

def has_blush(overlays):
    """overlays配列にblush系が含まれているかチェック"""
    if not isinstance(overlays, list):
        return True  # overlaysが配列でない場合はスキップ
    
    for item in overlays:
        if isinstance(item, str) and 'blush' in item.lower():
            return True
    return False

def add_blush_to_overlays(overlays):
    """overlays配列の先頭にblush_lowを追加"""
    if not isinstance(overlays, list):
        return overlays
    
    if has_blush(overlays):
        return overlays  # 既にblush系がある場合は何もしない
    
    # blush_lowを追加（先頭に）
    return ["blush_low"] + overlays

def process_dict(obj, modified):
    """辞書を再帰的に処理してoverlaysを更新"""
    if isinstance(obj, dict):
        if "overlays" in obj:
            original = obj["overlays"]
            updated = add_blush_to_overlays(original)
            if original != updated:
                obj["overlays"] = updated
                modified[0] = True
        
        for key, value in obj.items():
            if isinstance(value, (dict, list)):
                process_dict(value, modified)
    
    elif isinstance(obj, list):
        for item in obj:
            if isinstance(item, (dict, list)):
                process_dict(item, modified)

def process_file(filepath):
    """1つのJSONファイルを処理"""
    try:
        with open(filepath, 'r', encoding='utf-8') as f:
            data = json.load(f)
        
        modified = [False]  # リストで包むことで内部関数から変更可能に
        process_dict(data, modified)
        
        if modified[0]:
            # バックアップを作成
            backup_path = str(filepath) + ".bak"
            if not os.path.exists(backup_path):
                with open(backup_path, 'w', encoding='utf-8') as f:
                    json.dump(json.loads(open(filepath, 'r', encoding='utf-8').read()), f, ensure_ascii=False, indent=2)
            
            # 更新されたデータを保存
            with open(filepath, 'w', encoding='utf-8') as f:
                json.dump(data, f, ensure_ascii=False, indent=2)
            
            print(f"[MODIFIED] {filepath.name}")
            return True
        else:
            print(f"[SKIPPED] {filepath.name}")
            return False
            
    except Exception as e:
        print(f"[ERROR] {filepath.name}: {e}")
        return False

def main():
    dataex_dir = Path("dataEx")
    
    if not dataex_dir.exists():
        print("Error: dataEx directory not found!")
        sys.exit(1)
    
    # 処理対象外のファイル
    exclude_files = {"flavorRules.json", "overlaySets_slime.json"}
    
    json_files = []
    for filepath in dataex_dir.rglob("*.json"):
        if filepath.name not in exclude_files:
            json_files.append(filepath)
    
    print(f"Found {len(json_files)} JSON files to process\n")
    
    modified_count = 0
    for filepath in sorted(json_files):
        if process_file(filepath):
            modified_count += 1
    
    print(f"\n{'='*60}")
    print(f"Completed: {modified_count} files modified, {len(json_files) - modified_count} files unchanged")
    print(f"{'='*60}")

if __name__ == "__main__":
    main()

