import json
from pathlib import Path
import sys

# Windows環境でUTF-8出力を強制
sys.stdout.reconfigure(encoding='utf-8')

def collect_overlays(obj):
    """再帰的にoverlaysとunderOverlaysを収集"""
    overlays = set()
    if isinstance(obj, dict):
        if 'overlays' in obj and isinstance(obj['overlays'], list):
            overlays.update(obj['overlays'])
        if 'underOverlays' in obj and isinstance(obj['underOverlays'], list):
            overlays.update(obj['underOverlays'])
        for value in obj.values():
            overlays.update(collect_overlays(value))
    elif isinstance(obj, list):
        for item in obj:
            overlays.update(collect_overlays(item))
    return overlays

def get_profile_name(json_data):
    """JSONからprofileNameを取得"""
    if isinstance(json_data, dict):
        if 'meta' in json_data and isinstance(json_data['meta'], dict):
            return json_data['meta'].get('profileName')
    return None

def check_json_file(json_path):
    """個別のJSONファイルをチェック"""
    try:
        with open(json_path, encoding='utf-8') as f:
            data = json.load(f)
        
        # profileNameを取得
        profile_name = get_profile_name(data)
        if not profile_name:
            return None, None, None
        
        # overlaysを収集
        all_overlays = collect_overlays(data)
        
        # 擬音系（日本語を含むもの）をフィルタ
        onomatopoeia = [name for name in all_overlays if any(
            ord(c) > 127 for c in name  # 非ASCII文字を含む
        )]
        
        # 対応する画像ディレクトリを確認
        img_dir = Path(f"img/pictures/busts/{profile_name}")
        if not img_dir.exists():
            return profile_name, onomatopoeia, None
        
        # 存在するファイルを取得
        existing_files = {f.stem.replace('default_pose_overlay_', '') for f in img_dir.glob('*.png')}
        
        # 不在の画像を確認
        missing = [name for name in onomatopoeia if name not in existing_files]
        
        return profile_name, onomatopoeia, missing
        
    except Exception as e:
        print(f"エラー: {json_path}: {e}")
        return None, None, None

# メイン処理
print("=" * 80)
print("触手・スライム・ゴブリン関連JSONファイルの画像参照チェック")
print("=" * 80)

# 各敵タイプのJSONを検索
enemy_types = ['tentacle', 'slime', 'goblin']
all_results = {}

for enemy_type in enemy_types:
    print(f"\n{'=' * 80}")
    print(f"【{enemy_type.upper()}】")
    print('=' * 80)
    
    json_files = list(Path('dataEx').rglob(f'*{enemy_type}*.json'))
    print(f"見つかったJSONファイル: {len(json_files)}個\n")
    
    for json_file in sorted(json_files):
        profile_name, onomatopoeia, missing = check_json_file(json_file)
        
        if profile_name is None:
            continue
        
        relative_path = json_file.relative_to(Path('dataEx'))
        
        if not onomatopoeia:
            print(f"  ✓ {relative_path}")
            print(f"    → profileName: {profile_name}")
            print(f"    → 擬音オーバーレイなし")
        elif missing is None:
            print(f"  ? {relative_path}")
            print(f"    → profileName: {profile_name}")
            print(f"    → 画像ディレクトリが存在しません: img/pictures/busts/{profile_name}")
        elif len(missing) == 0:
            print(f"  ✓ {relative_path}")
            print(f"    → profileName: {profile_name}")
            print(f"    → 擬音: {len(onomatopoeia)}個 (全て存在)")
        else:
            print(f"  ✗ {relative_path}")
            print(f"    → profileName: {profile_name}")
            print(f"    → 擬音: {len(onomatopoeia)}個 (不在: {len(missing)}個)")
            for m in missing:
                print(f"       [NG] {m}")
        
        print()

print("=" * 80)
print("チェック完了")
print("=" * 80)

