#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
長すぎるテキストを検出するツール
RPGツクールMZの表示限界（約30文字/行）を超えるテキストをリスト化
"""

import json
import glob
import os
import sys
from pathlib import Path

# 設定
MAX_LENGTH = 30  # 表示限界文字数
WARNING_LENGTH = 25  # 警告を出す文字数

def count_display_length(text):
    """
    表示文字数をカウント
    全角文字は1文字、半角文字は0.5文字として計算（簡易版）
    """
    if not text:
        return 0
    
    length = 0
    for char in text:
        # ASCII文字（半角）は0.5、それ以外（全角）は1としてカウント
        if ord(char) < 128:
            length += 0.5
        else:
            length += 1
    return length

def check_text_in_dialogueSequence(data, file_path):
    """
    dialogueSequence内のtextフィールドをチェック
    """
    issues = []
    
    def traverse(obj, path=""):
        """再帰的にJSONを探索"""
        if isinstance(obj, dict):
            # dialogueSequenceを見つけた場合
            if "dialogueSequence" in obj:
                seq = obj["dialogueSequence"]
                if isinstance(seq, list):
                    for idx, entry in enumerate(seq):
                        if isinstance(entry, dict) and "text" in entry:
                            text = entry["text"]
                            length = len(text)  # 単純な文字数
                            display_length = count_display_length(text)
                            
                            if length > MAX_LENGTH:
                                issues.append({
                                    "file": file_path,
                                    "path": f"{path}.dialogueSequence[{idx}]",
                                    "length": length,
                                    "display_length": display_length,
                                    "text": text,
                                    "severity": "ERROR"
                                })
                            elif length > WARNING_LENGTH:
                                issues.append({
                                    "file": file_path,
                                    "path": f"{path}.dialogueSequence[{idx}]",
                                    "length": length,
                                    "display_length": display_length,
                                    "text": text,
                                    "severity": "WARNING"
                                })
            
            # 他のキーも再帰的にチェック
            for key, value in obj.items():
                new_path = f"{path}.{key}" if path else key
                traverse(value, new_path)
        
        elif isinstance(obj, list):
            for idx, item in enumerate(obj):
                new_path = f"{path}[{idx}]"
                traverse(item, new_path)
    
    traverse(data)
    return issues

def main():
    print("=" * 80)
    print("長すぎるテキスト検出ツール")
    print(f"表示限界: {MAX_LENGTH}文字/行（警告: {WARNING_LENGTH}文字以上）")
    print("=" * 80)
    print()
    
    # poseProfiles内のすべてのJSONファイルをチェック
    pattern = "dataEx/poseProfiles/*.json"
    files = glob.glob(pattern)
    
    if not files:
        print(f"エラー: {pattern} にファイルが見つかりません")
        return
    
    all_issues = []
    
    for file_path in sorted(files):
        # バックアップファイルをスキップ
        if any(ext in file_path for ext in ['.bak', '.backup', '.before_']):
            continue
        
        try:
            with open(file_path, 'r', encoding='utf-8') as f:
                data = json.load(f)
            
            issues = check_text_in_dialogueSequence(data, file_path)
            all_issues.extend(issues)
            
        except json.JSONDecodeError as e:
            print(f"⚠️  JSONパースエラー: {file_path}")
            print(f"   {e}")
        except Exception as e:
            print(f"⚠️  エラー: {file_path}")
            print(f"   {e}")
    
    # 結果を出力
    if not all_issues:
        print("✅ 問題のあるテキストは見つかりませんでした！")
        return
    
    # 重要度順にソート（ERROR → WARNING）
    all_issues.sort(key=lambda x: (x["severity"] != "ERROR", x["length"]), reverse=True)
    
    errors = [i for i in all_issues if i["severity"] == "ERROR"]
    warnings = [i for i in all_issues if i["severity"] == "WARNING"]
    
    print(f"🔴 エラー（{MAX_LENGTH}文字超）: {len(errors)}件")
    print(f"🟡 警告（{WARNING_LENGTH}文字以上）: {len(warnings)}件")
    print()
    
    # TSV形式でファイルに出力
    output_file = "tools/long_text_report.tsv"
    with open(output_file, 'w', encoding='utf-8') as f:
        f.write("重要度\tファイル\t位置\t文字数\t表示長\tテキスト\n")
        for issue in all_issues:
            f.write(f"{issue['severity']}\t")
            f.write(f"{issue['file']}\t")
            f.write(f"{issue['path']}\t")
            f.write(f"{issue['length']}\t")
            f.write(f"{issue['display_length']:.1f}\t")
            f.write(f"{issue['text']}\n")
    
    print(f"📄 詳細レポートを出力しました: {output_file}")
    print()
    
    # 上位10件をコンソールに表示
    print("=" * 80)
    print("上位10件（最も長いテキスト）")
    print("=" * 80)
    for idx, issue in enumerate(all_issues[:10], 1):
        severity_icon = "🔴" if issue["severity"] == "ERROR" else "🟡"
        print(f"{idx}. {severity_icon} {issue['severity']}")
        print(f"   ファイル: {os.path.basename(issue['file'])}")
        print(f"   位置: {issue['path']}")
        print(f"   文字数: {issue['length']}文字（表示長: {issue['display_length']:.1f}）")
        print(f"   テキスト: {issue['text'][:60]}{'...' if len(issue['text']) > 60 else ''}")
        print()
    
    if len(all_issues) > 10:
        print(f"... 他 {len(all_issues) - 10} 件（詳細は {output_file} を参照）")
    
    print()
    print("=" * 80)
    print("💡 修正方法:")
    print("   1. エラー（30文字超）を優先的に修正")
    print("   2. 重要な箇所（climax、stage3など）から修正")
    print("   3. テキスト内に \\n を挿入して手動で改行")
    print("=" * 80)

if __name__ == "__main__":
    main()

