import json
from pathlib import Path
import sys
from collections import defaultdict

sys.stdout.reconfigure(encoding='utf-8')

issues = {
    'json_errors': [],
    'empty_files': [],
    'missing_meta': [],
    'missing_profilename': [],
    'profilename_mismatch': [],
    'missing_directories': [],
    'missing_overlays': [],
    'down_attack_structure': [],
    'duplicate_profilenames': defaultdict(list),
    'unused_image_dirs': [],
    'inconsistent_stages': []
}

def collect_overlays(obj):
    overlays = []
    if isinstance(obj, dict):
        if 'overlays' in obj and isinstance(obj['overlays'], list):
            overlays.extend(obj['overlays'])
        for value in obj.values():
            overlays.extend(collect_overlays(value))
    elif isinstance(obj, list):
        for item in obj:
            overlays.extend(collect_overlays(item))
    return overlays

def check_down_attack_structure(data, filepath):
    """down_attack構造の問題をチェック"""
    if 'events' not in data or not isinstance(data['events'], dict):
        return
    
    down_attack = data['events'].get('down_attack')
    if not down_attack or not isinstance(down_attack, dict):
        return
    
    for stage_key, stage_data in down_attack.items():
        if not stage_key.startswith('stage'):
            continue
        
        # stage2, stage3がresist/fall構造になっていないかチェック
        if isinstance(stage_data, dict):
            if 'resist' in stage_data or 'fall' in stage_data:
                issues['down_attack_structure'].append({
                    'file': filepath,
                    'stage': stage_key,
                    'issue': 'resist/fall構造（配列に変更が必要）'
                })

def check_stage_consistency(data, filepath):
    """stage定義の一貫性をチェック"""
    if 'events' not in data or not isinstance(data['events'], dict):
        return
    
    for event_name, event_data in data['events'].items():
        if not isinstance(event_data, dict):
            continue
        
        stages = [k for k in event_data.keys() if k.startswith('stage')]
        if not stages:
            continue
        
        # stage1があるか
        if 'stage1' not in stages:
            issues['inconsistent_stages'].append({
                'file': filepath,
                'event': event_name,
                'issue': 'stage1が存在しない',
                'stages': stages
            })
        
        # stage番号が連続しているか
        stage_nums = sorted([int(s.replace('stage', '')) for s in stages])
        expected = list(range(1, max(stage_nums) + 1))
        if stage_nums != expected:
            issues['inconsistent_stages'].append({
                'file': filepath,
                'event': event_name,
                'issue': 'stage番号が不連続',
                'found': stages,
                'expected': [f'stage{i}' for i in expected]
            })

print("=" * 80)
print("徹底的なデバッグチェック")
print("=" * 80)

# 全JSONファイルを検査
all_profilenames = {}
all_json_files = list(Path('dataEx').rglob('*.json'))

print(f"\n📁 検査対象: {len(all_json_files)}個のJSONファイル")
print("=" * 80)

for json_file in all_json_files:
    try:
        # ファイルサイズチェック
        if json_file.stat().st_size == 0:
            issues['empty_files'].append(str(json_file))
            continue
        
        # JSON読み込み
        with open(json_file, encoding='utf-8') as f:
            content = f.read()
            if not content.strip():
                issues['empty_files'].append(str(json_file))
                continue
            
            data = json.loads(content)
        
        # meta情報チェック
        if not isinstance(data, dict):
            continue
        
        if 'meta' not in data:
            issues['missing_meta'].append(str(json_file))
            continue
        
        meta = data['meta']
        if 'profileName' not in meta:
            issues['missing_profilename'].append(str(json_file))
            continue
        
        profile_name = meta['profileName']
        
        # profileName重複チェック
        all_profilenames[profile_name] = all_profilenames.get(profile_name, []) + [str(json_file)]
        
        # orgasm系は除外
        if '_orgasm' in profile_name:
            continue
        
        # ディレクトリ存在チェック
        img_dir = Path(f"img/pictures/busts/{profile_name}")
        if not img_dir.exists():
            issues['missing_directories'].append({
                'file': str(json_file),
                'profileName': profile_name,
                'directory': str(img_dir)
            })
            continue
        
        # オーバーレイチェック
        all_overlays = collect_overlays(data)
        onomatopoeia = [o for o in all_overlays if any(ord(c) > 127 for c in o)]
        
        if onomatopoeia:
            existing_files = {f.stem.replace('default_pose_overlay_', '') for f in img_dir.rglob('*.png')}
            missing = [o for o in onomatopoeia if o not in existing_files]
            
            if missing:
                issues['missing_overlays'].append({
                    'file': str(json_file),
                    'profileName': profile_name,
                    'missing': missing
                })
        
        # down_attack構造チェック
        check_down_attack_structure(data, str(json_file))
        
        # stage一貫性チェック
        check_stage_consistency(data, str(json_file))
        
    except json.JSONDecodeError as e:
        issues['json_errors'].append({
            'file': str(json_file),
            'error': str(e)
        })
    except Exception as e:
        issues['json_errors'].append({
            'file': str(json_file),
            'error': f"Unexpected: {str(e)}"
        })

# profileName重複チェック
for profile_name, files in all_profilenames.items():
    if len(files) > 1 and '_orgasm' not in profile_name:
        issues['duplicate_profilenames'][profile_name] = files

# 未使用の画像ディレクトリチェック
img_bust_dir = Path('img/pictures/busts')
if img_bust_dir.exists():
    used_profiles = set(all_profilenames.keys())
    for img_dir in img_bust_dir.iterdir():
        if img_dir.is_dir() and img_dir.name not in used_profiles:
            # orgasm系は除外
            if '_orgasm' not in img_dir.name:
                issues['unused_image_dirs'].append(img_dir.name)

# 結果表示
print("\n" + "=" * 80)
print("【1】JSONエラー")
print("=" * 80)
if issues['json_errors']:
    for item in issues['json_errors']:
        rel_path = Path(item['file']).relative_to(Path('dataEx')) if 'dataEx' in item['file'] else item['file']
        print(f"  ✗ {rel_path}")
        print(f"    エラー: {item['error']}")
else:
    print("  ✓ 問題なし")

print("\n" + "=" * 80)
print("【2】空のJSONファイル")
print("=" * 80)
if issues['empty_files']:
    for file in issues['empty_files']:
        rel_path = Path(file).relative_to(Path('dataEx')) if 'dataEx' in file else file
        print(f"  ✗ {rel_path}")
else:
    print("  ✓ 問題なし")

print("\n" + "=" * 80)
print("【3】meta情報の欠落")
print("=" * 80)
if issues['missing_meta']:
    for file in issues['missing_meta']:
        rel_path = Path(file).relative_to(Path('dataEx')) if 'dataEx' in file else file
        print(f"  ✗ {rel_path}: metaが存在しない")
else:
    print("  ✓ 問題なし")

print("\n" + "=" * 80)
print("【4】profileNameの欠落")
print("=" * 80)
if issues['missing_profilename']:
    for file in issues['missing_profilename']:
        rel_path = Path(file).relative_to(Path('dataEx')) if 'dataEx' in file else file
        print(f"  ✗ {rel_path}: profileNameが存在しない")
else:
    print("  ✓ 問題なし")

print("\n" + "=" * 80)
print("【5】画像ディレクトリの不在")
print("=" * 80)
if issues['missing_directories']:
    for item in issues['missing_directories']:
        rel_path = Path(item['file']).relative_to(Path('dataEx')) if 'dataEx' in item['file'] else item['file']
        print(f"  ✗ {rel_path}")
        print(f"    profileName: {item['profileName']}")
        print(f"    不在: {item['directory']}")
else:
    print("  ✓ 問題なし")

print("\n" + "=" * 80)
print("【6】擬音オーバーレイの不在")
print("=" * 80)
if issues['missing_overlays']:
    for item in issues['missing_overlays']:
        rel_path = Path(item['file']).relative_to(Path('dataEx')) if 'dataEx' in item['file'] else item['file']
        print(f"  ✗ {rel_path}")
        print(f"    profileName: {item['profileName']}")
        print(f"    不在: {', '.join(item['missing'][:5])}" + (f"... 他{len(item['missing'])-5}個" if len(item['missing']) > 5 else ""))
else:
    print("  ✓ 問題なし")

print("\n" + "=" * 80)
print("【7】down_attack構造の問題")
print("=" * 80)
if issues['down_attack_structure']:
    for item in issues['down_attack_structure']:
        rel_path = Path(item['file']).relative_to(Path('dataEx')) if 'dataEx' in item['file'] else item['file']
        print(f"  ✗ {rel_path}")
        print(f"    {item['stage']}: {item['issue']}")
else:
    print("  ✓ 問題なし")

print("\n" + "=" * 80)
print("【8】profileNameの重複")
print("=" * 80)
if issues['duplicate_profilenames']:
    for profile_name, files in issues['duplicate_profilenames'].items():
        print(f"  ⚠️  {profile_name} ({len(files)}個)")
        for file in files:
            rel_path = Path(file).relative_to(Path('dataEx')) if 'dataEx' in file else file
            print(f"    - {rel_path}")
else:
    print("  ✓ 問題なし")

print("\n" + "=" * 80)
print("【9】未使用の画像ディレクトリ")
print("=" * 80)
if issues['unused_image_dirs']:
    for dir_name in sorted(issues['unused_image_dirs']):
        print(f"  ⚠️  {dir_name}")
else:
    print("  ✓ 問題なし")

print("\n" + "=" * 80)
print("【10】stage定義の不整合")
print("=" * 80)
if issues['inconsistent_stages']:
    for item in issues['inconsistent_stages']:
        rel_path = Path(item['file']).relative_to(Path('dataEx')) if 'dataEx' in item['file'] else item['file']
        print(f"  ⚠️  {rel_path}")
        print(f"    イベント: {item['event']}")
        print(f"    問題: {item['issue']}")
        if 'stages' in item:
            print(f"    現在: {', '.join(item['stages'])}")
        if 'found' in item and 'expected' in item:
            print(f"    検出: {', '.join(item['found'])}")
            print(f"    期待: {', '.join(item['expected'])}")
else:
    print("  ✓ 問題なし")

# サマリー
print("\n" + "=" * 80)
print("サマリー")
print("=" * 80)
total_issues = sum([
    len(issues['json_errors']),
    len(issues['empty_files']),
    len(issues['missing_meta']),
    len(issues['missing_profilename']),
    len(issues['missing_directories']),
    len(issues['missing_overlays']),
    len(issues['down_attack_structure']),
    len(issues['duplicate_profilenames']),
    len(issues['unused_image_dirs']),
    len(issues['inconsistent_stages'])
])

if total_issues == 0:
    print("  🎉 問題は検出されませんでした！")
else:
    print(f"  検出された問題: {total_issues}件")
    print(f"    - JSONエラー: {len(issues['json_errors'])}件")
    print(f"    - 空ファイル: {len(issues['empty_files'])}件")
    print(f"    - meta欠落: {len(issues['missing_meta'])}件")
    print(f"    - profileName欠落: {len(issues['missing_profilename'])}件")
    print(f"    - ディレクトリ不在: {len(issues['missing_directories'])}件")
    print(f"    - 擬音不在: {len(issues['missing_overlays'])}件")
    print(f"    - down_attack問題: {len(issues['down_attack_structure'])}件")
    print(f"    - profileName重複: {len(issues['duplicate_profilenames'])}件")
    print(f"    - 未使用ディレクトリ: {len(issues['unused_image_dirs'])}件")
    print(f"    - stage不整合: {len(issues['inconsistent_stages'])}件")

print("=" * 80)

