#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
addSlimeBreastOverlays.py

スライム関連の全dialogueSequenceに胸オーバーレイを交互に追加
- 同じdialogueSequence内で low/high を交互に配置して動きを表現
- 既に slime_breast_low/high があれば追加しない
"""

import json
import sys
from pathlib import Path

sys.stdout.reconfigure(encoding='utf-8')

def add_breast_overlays_alternating(obj, path="", added_count=0):
    """再帰的に胸オーバーレイを交互に追加"""
    if isinstance(obj, dict):
        # dialogueSequence 配列を見つけたら交互処理
        if 'dialogueSequence' in obj and isinstance(obj['dialogueSequence'], list):
            dialogue_seq = obj['dialogueSequence']
            
            for idx, line in enumerate(dialogue_seq):
                if isinstance(line, dict) and 'overlays' in line and isinstance(line['overlays'], list):
                    overlays = line['overlays']
                    
                    # 既に胸オーバーレイがあるかチェック
                    has_breast = any('slime_breast' in o for o in overlays if isinstance(o, str))
                    
                    if not has_breast:
                        # インデックスで交互に決定（偶数: low, 奇数: high）
                        breast_overlay = 'slime_breast_low' if idx % 2 == 0 else 'slime_breast_high'
                        
                        # 挿入位置を決定
                        if 'slime_bind_base' in overlays:
                            insert_pos = overlays.index('slime_bind_base') + 1
                        elif 'restrain' in overlays:
                            insert_pos = overlays.index('restrain') + 1
                        else:
                            insert_pos = 0
                        
                        overlays.insert(insert_pos, breast_overlay)
                        added_count += 1
                        print(f"  [追加] {path}[{idx}]: {breast_overlay}")
        
        # 再帰的に処理（dialogueSequence以外も探索）
        for key, value in obj.items():
            new_path = f"{path}.{key}" if path else key
            added_count = add_breast_overlays_alternating(value, new_path, added_count)
    
    elif isinstance(obj, list):
        for idx, item in enumerate(obj):
            new_path = f"{path}[{idx}]"
            added_count = add_breast_overlays_alternating(item, new_path, added_count)
    
    return added_count

def process_file(file_path):
    """単一ファイルを処理"""
    print(f"\n📄 処理中: {file_path.name}")
    
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            data = json.load(f)
        
        added_count = add_breast_overlays_alternating(data)
        
        if added_count > 0:
            # バックアップ作成
            backup_path = file_path.with_suffix(file_path.suffix + '.backup')
            with open(backup_path, 'w', encoding='utf-8') as f:
                json.dump(json.loads(open(file_path, 'r', encoding='utf-8').read()), 
                         f, ensure_ascii=False, indent=2)
            
            # 修正版を保存
            with open(file_path, 'w', encoding='utf-8') as f:
                json.dump(data, f, ensure_ascii=False, indent=2)
            
            print(f"  ✅ 完了: {added_count}箇所に追加")
            print(f"  💾 バックアップ: {backup_path.name}")
            return True
        else:
            print(f"  ℹ️  追加不要（既に完備）")
            return False
            
    except Exception as e:
        print(f"  ❌ エラー: {e}")
        return False

def main():
    print("=" * 80)
    print("🔧 スライム胸オーバーレイ自動追加ツール（交互パターン）")
    print("=" * 80)
    print("📝 処理ルール:")
    print("   - 同じdialogueSequence内で low → high → low → high... と交互に配置")
    print("   - これにより動きとエロさを演出")
    print("=" * 80)
    
    project_root = Path(__file__).parent.parent.parent
    
    # 処理対象ファイル
    target_files = [
        project_root / 'dataEx/poseProfiles/slime_restrained_low.json',
        project_root / 'dataEx/poseProfiles/slime_restrained_high.json',
        project_root / 'dataEx/poseProfiles/slime_restrained_insert_low.json',
        project_root / 'dataEx/poseProfiles/slime_restrained_insert_high.json',
    ]
    
    fixed_count = 0
    
    for file_path in target_files:
        if not file_path.exists():
            print(f"\n⚠️  ファイルが見つかりません: {file_path.name}")
            continue
        
        if process_file(file_path):
            fixed_count += 1
    
    print("\n" + "=" * 80)
    if fixed_count > 0:
        print(f"✨ 完了: {fixed_count}ファイルを修正しました")
        print(f"💡 効果: 胸オーバーレイが交互に変化し、動きとエロさが向上します")
    else:
        print("✅ すべてのファイルが既に完備されています")
    print("=" * 80)
    print()

if __name__ == "__main__":
    main()

