#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
addSlimeBreastOverlaysLowOnly.py

絶頂・挿入イベントのスライム関係ファイルに slime_breast_low のみを追加
- 全ての dialogueSequence に slime_breast_low を追加（交互ではなく統一）
- 既に slime_breast_low/high があれば追加しない
"""

import json
import sys
from pathlib import Path

sys.stdout.reconfigure(encoding='utf-8')

def add_breast_low_only(obj, path="", added_count=0):
    """再帰的に slime_breast_low のみを追加"""
    if isinstance(obj, dict):
        # dialogueSequence 配列を見つけたら処理
        if 'dialogueSequence' in obj and isinstance(obj['dialogueSequence'], list):
            dialogue_seq = obj['dialogueSequence']
            
            for idx, line in enumerate(dialogue_seq):
                if isinstance(line, dict) and 'overlays' in line and isinstance(line['overlays'], list):
                    overlays = line['overlays']
                    
                    # 既に胸オーバーレイがあるかチェック
                    has_breast = any('slime_breast' in o for o in overlays if isinstance(o, str))
                    
                    if not has_breast:
                        # 常に slime_breast_low を追加
                        breast_overlay = 'slime_breast_low'
                        
                        # 挿入位置を決定
                        if 'slime_bind_base' in overlays:
                            insert_pos = overlays.index('slime_bind_base') + 1
                        elif 'restrain' in overlays:
                            insert_pos = overlays.index('restrain') + 1
                        else:
                            insert_pos = 0
                        
                        overlays.insert(insert_pos, breast_overlay)
                        added_count += 1
                        print(f"  [追加] {path}[{idx}]: {breast_overlay}")
        
        # 再帰的に処理（dialogueSequence以外も探索）
        for key, value in obj.items():
            new_path = f"{path}.{key}" if path else key
            added_count = add_breast_low_only(value, new_path, added_count)
    
    elif isinstance(obj, list):
        for idx, item in enumerate(obj):
            new_path = f"{path}[{idx}]"
            added_count = add_breast_low_only(item, new_path, added_count)
    
    return added_count

def process_file(file_path):
    """単一ファイルを処理"""
    print(f"\n📄 処理中: {file_path.name}")
    
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            data = json.load(f)
        
        added_count = add_breast_low_only(data)
        
        if added_count > 0:
            # バックアップ作成
            backup_path = file_path.with_suffix(file_path.suffix + '.backup2')
            with open(backup_path, 'w', encoding='utf-8') as f:
                json.dump(json.loads(open(file_path, 'r', encoding='utf-8').read()), 
                         f, ensure_ascii=False, indent=2)
            
            # 修正版を保存
            with open(file_path, 'w', encoding='utf-8') as f:
                json.dump(data, f, ensure_ascii=False, indent=2)
            
            print(f"  ✅ 完了: {added_count}箇所に追加（全て low）")
            print(f"  💾 バックアップ: {backup_path.name}")
            return added_count
        else:
            print(f"  ℹ️  追加不要（既に完備）")
            return 0
            
    except Exception as e:
        print(f"  ❌ エラー: {e}")
        return 0

def find_slime_files(directory):
    """指定ディレクトリからスライム関係のJSONファイルを再帰的に検索"""
    files = []
    for json_file in Path(directory).rglob('*.json'):
        name_lower = json_file.name.lower()
        # スライム関係のファイルを検索
        if 'slime' in name_lower:
            files.append(json_file)
    return files

def main():
    print("=" * 80)
    print("🔧 スライム胸オーバーレイ自動追加ツール（Low固定版）")
    print("=" * 80)
    print("📝 処理ルール:")
    print("   - 全ての dialogueSequence に slime_breast_low を追加（統一）")
    print("   - 絶頂・挿入イベント用（交互パターンなし）")
    print("=" * 80)
    
    project_root = Path(__file__).parent.parent.parent
    
    # 処理対象ディレクトリ
    target_dirs = [
        project_root / 'dataEx/introProfiles',
        project_root / 'dataEx/orgasmDetailProfiles',
    ]
    
    # スライム関係のファイルを収集
    all_files = []
    for target_dir in target_dirs:
        if target_dir.exists():
            found_files = find_slime_files(target_dir)
            all_files.extend(found_files)
            print(f"\n📂 {target_dir.name}: {len(found_files)}ファイル発見")
    
    if not all_files:
        print("\n⚠️  スライム関係のファイルが見つかりませんでした")
        return
    
    print(f"\n合計: {len(all_files)}ファイルを処理します")
    
    total_added = 0
    fixed_count = 0
    
    for file_path in all_files:
        added = process_file(file_path)
        if added > 0:
            fixed_count += 1
            total_added += added
    
    print("\n" + "=" * 80)
    if fixed_count > 0:
        print(f"✨ 完了: {fixed_count}ファイルを修正しました")
        print(f"📊 合計: {total_added}箇所に slime_breast_low を追加")
        print(f"💡 効果: 絶頂・挿入時にも統一された胸オーバーレイが表示されます")
    else:
        print("✅ すべてのファイルが既に完備されています")
    print("=" * 80)
    print()

if __name__ == "__main__":
    main()

