#!/usr/bin/env node
/**
 * analyzeOverlays.js
 * 
 * RPGツクールMZ 快感演出システム用 JSON静的解析スクリプト
 * 
 * 【主な機能】
 * A. 同じ向きのオーバーレイが重複使用されているかチェック（例: _rightが2枚）
 * B. 挿入時sfxと非挿入時sfxの共用チェック
 * 
 * 【使い方】
 * node tools/debug/analyzeOverlays.js
 * 
 * または TypeScript の場合:
 * ts-node tools/debug/analyzeOverlays.ts
 */

const fs = require('fs');
const path = require('path');

// ==================== 設定 ====================

// 解析対象ディレクトリ
const TARGET_DIRS = [
  'dataEx/poseProfiles',
  'dataEx/overlays',
  'dataEx/restraintProfiles',
  'dataEx/introProfiles',
  'dataEx/orgasmDetailProfiles'
];

// SFX判定用キーワード（擬音語）
const SFX_KEYWORDS = [
  'ぐちゅ', 'ぬちゃ', 'ぬちょ', 'ズプッ', 'ズポ', 'ずぶ',
  'ぐちゃ', 'にゅぷ', 'にちゃ', 'じゅぽ', 'びちゃ',
  'くちゅ', 'じゅる', 'ねちゃ'
];

// Moan判定用キーワード（喘ぎ声）
const MOAN_KEYWORDS = [
  'あっ', 'んっ', 'はぁ', 'ひゃあ', 'んん', 'あん', 'うん',
  'ふぅ', 'おほ', 'ひぃ', 'やぁ', 'いや', 'んむ', 'んぐ',
  'あぁ', 'うぅ', 'ひっ', 'んぁ', 'ふん', 'んふ'
];

// 挿入系イベント判定用キーワード
const INSERT_EVENT_KEYWORDS = [
  'insert', 'inserted', 'insertion', '挿入'
];

// ==================== ユーティリティ ====================

/**
 * 指定ディレクトリ以下の全JSONファイルを再帰的に取得
 */
function getAllJsonFiles(dirPath, basePath = '') {
  const results = [];
  
  if (!fs.existsSync(dirPath)) {
    return results;
  }
  
  const entries = fs.readdirSync(dirPath, { withFileTypes: true });
  
  for (const entry of entries) {
    const fullPath = path.join(dirPath, entry.name);
    const relativePath = path.join(basePath, entry.name);
    
    if (entry.isDirectory()) {
      results.push(...getAllJsonFiles(fullPath, relativePath));
    } else if (entry.name.endsWith('.json')) {
      results.push({ fullPath, relativePath });
    }
  }
  
  return results;
}

/**
 * オーバーレイIDからベース名を抽出（_left/_rightを除去）
 */
function getBaseName(overlayId) {
  return overlayId
    .replace(/_left$/i, '')
    .replace(/_right$/i, '')
    .replace(/_左$/i, '')
    .replace(/_右$/i, '');
}

/**
 * オーバーレイIDの向きを取得
 */
function getDirection(overlayId) {
  if (/_left$/i.test(overlayId) || /_左$/i.test(overlayId)) return 'left';
  if (/_right$/i.test(overlayId) || /_右$/i.test(overlayId)) return 'right';
  return null;
}

/**
 * オーバーレイがSFXかどうか判定
 */
function isSfxOverlay(overlayId) {
  return SFX_KEYWORDS.some(keyword => overlayId.includes(keyword));
}

/**
 * オーバーレイがMoanかどうか判定
 */
function isMoanOverlay(overlayId) {
  return MOAN_KEYWORDS.some(keyword => overlayId.includes(keyword));
}

/**
 * パス文字列が挿入系イベントかどうか判定
 */
function isInsertContext(pathString, fileContent) {
  // ファイル名チェック
  if (INSERT_EVENT_KEYWORDS.some(kw => pathString.toLowerCase().includes(kw))) {
    return true;
  }
  
  // JSONキー構造チェック
  const keys = pathString.split('.');
  return keys.some(key => INSERT_EVENT_KEYWORDS.some(kw => key.toLowerCase().includes(kw)));
}

/**
 * JSONを再帰的に走査してオーバーレイ配列を収集
 */
function collectOverlayArrays(obj, currentPath = '', filePath = '', fileContent = null) {
  const results = [];
  
  if (!obj || typeof obj !== 'object') {
    return results;
  }
  
  // overlays配列を発見
  if (Array.isArray(obj) && currentPath.endsWith('overlays')) {
    const overlayIds = obj.filter(item => typeof item === 'string');
    if (overlayIds.length > 0) {
      results.push({
        path: currentPath,
        file: filePath,
        overlays: overlayIds,
        isInsert: isInsertContext(currentPath, fileContent)
      });
    }
    return results;
  }
  
  // 配列の場合
  if (Array.isArray(obj)) {
    obj.forEach((item, index) => {
      const newPath = `${currentPath}[${index}]`;
      results.push(...collectOverlayArrays(item, newPath, filePath, fileContent));
    });
  }
  // オブジェクトの場合
  else {
    for (const [key, value] of Object.entries(obj)) {
      const newPath = currentPath ? `${currentPath}.${key}` : key;
      results.push(...collectOverlayArrays(value, newPath, filePath, fileContent));
    }
  }
  
  return results;
}

// ==================== チェックA: 同じ向きのオーバーレイ重複チェック ====================

/**
 * 同じ向きのオーバーレイが重複使用されているかチェック
 * 例: "ぐちゅっ1-6_right" が同じ行に2枚以上ある場合
 */
function findLeftRightConflicts(overlayArrays) {
  const conflicts = [];
  
  for (const arrayInfo of overlayArrays) {
    const { overlays, path: arrayPath, file } = arrayInfo;
    
    // 各オーバーレイIDの出現回数をカウント
    const idCountMap = new Map();
    
    for (const overlayId of overlays) {
      const direction = getDirection(overlayId);
      if (!direction) continue; // 向きなしはスキップ
      
      idCountMap.set(overlayId, (idCountMap.get(overlayId) || 0) + 1);
    }
    
    // 2回以上出現しているオーバーレイIDを検出
    for (const [overlayId, count] of idCountMap.entries()) {
      if (count >= 2) {
        conflicts.push({
          file,
          path: arrayPath,
          overlayId,
          count,
          direction: getDirection(overlayId)
        });
      }
    }
  }
  
  return conflicts;
}

/**
 * チェックA結果をレポート
 */
function reportLeftRightConflicts(conflicts) {
  console.log('\n' + '='.repeat(80));
  console.log('【チェックA】同じ向きのオーバーレイ重複使用チェック');
  console.log('='.repeat(80));
  
  if (conflicts.length === 0) {
    console.log('\n✅ 重複なし：すべてのオーバーレイが適切に使用されています。\n');
    return;
  }
  
  console.log(`\n⚠️  ${conflicts.length}件の重複が検出されました：\n`);
  
  for (const conflict of conflicts) {
    console.log('[OVERLAY_CONFLICT][DUPLICATE]');
    console.log(`  file: ${conflict.file}`);
    console.log(`  path: ${conflict.path}`);
    console.log(`  overlayId: "${conflict.overlayId}"`);
    console.log(`  direction: ${conflict.direction}`);
    console.log(`  count: ${conflict.count}回使用`);
    console.log('');
  }
}

// ==================== チェックB: SFX挿入系/非挿入系 共用チェック ====================

/**
 * SFXオーバーレイの使用状況を分類
 */
function classifySfxOverlayUsage(overlayArrays) {
  const sfxUsage = new Map(); // sfxId -> { insertUses: [], nonInsertUses: [] }
  
  for (const arrayInfo of overlayArrays) {
    const { overlays, path: arrayPath, file, isInsert } = arrayInfo;
    
    for (const overlayId of overlays) {
      // SFXのみ対象
      if (!isSfxOverlay(overlayId)) continue;
      
      if (!sfxUsage.has(overlayId)) {
        sfxUsage.set(overlayId, {
          insertUses: [],
          nonInsertUses: []
        });
      }
      
      const usage = sfxUsage.get(overlayId);
      const useInfo = { file, path: arrayPath };
      
      if (isInsert) {
        usage.insertUses.push(useInfo);
      } else {
        usage.nonInsertUses.push(useInfo);
      }
    }
  }
  
  // 分類
  const result = {
    insertOnly: [],
    nonInsertOnly: [],
    bothUsed: []
  };
  
  for (const [sfxId, usage] of sfxUsage.entries()) {
    const hasInsert = usage.insertUses.length > 0;
    const hasNonInsert = usage.nonInsertUses.length > 0;
    
    if (hasInsert && hasNonInsert) {
      result.bothUsed.push({ sfxId, ...usage });
    } else if (hasInsert) {
      result.insertOnly.push({ sfxId, ...usage });
    } else if (hasNonInsert) {
      result.nonInsertOnly.push({ sfxId, ...usage });
    }
  }
  
  return result;
}

/**
 * チェックB結果をレポート
 */
function reportSfxUsage(classification) {
  console.log('\n' + '='.repeat(80));
  console.log('【チェックB】挿入時sfxと非挿入時sfxの共用チェック');
  console.log('='.repeat(80));
  
  const { insertOnly, nonInsertOnly, bothUsed } = classification;
  
  // サマリー
  console.log('\n=== SFX USAGE SUMMARY ===');
  console.log(`total sfx overlays: ${insertOnly.length + nonInsertOnly.length + bothUsed.length}`);
  console.log(`insertOnly: ${insertOnly.length}`);
  console.log(`nonInsertOnly: ${nonInsertOnly.length}`);
  console.log(`bothUsed: ${bothUsed.length}`);
  console.log('=========================\n');
  
  // bothUsedの詳細
  if (bothUsed.length > 0) {
    console.log(`⚠️  ${bothUsed.length}個のSFXが挿入系・非挿入系の両方で使用されています：\n`);
    
    for (const item of bothUsed) {
      console.log(`[SFX_USAGE][BOTH] ${item.sfxId}`);
      
      console.log('  - insert:');
      const insertFiles = new Set(item.insertUses.map(u => u.file));
      for (const file of insertFiles) {
        console.log(`      file: ${file}`);
        const paths = item.insertUses.filter(u => u.file === file).map(u => u.path);
        for (const p of paths.slice(0, 3)) { // 最初の3件のみ表示
          console.log(`        path: ${p}`);
        }
        if (paths.length > 3) {
          console.log(`        ... 他${paths.length - 3}件`);
        }
      }
      
      console.log('  - non-insert:');
      const nonInsertFiles = new Set(item.nonInsertUses.map(u => u.file));
      for (const file of nonInsertFiles) {
        console.log(`      file: ${file}`);
        const paths = item.nonInsertUses.filter(u => u.file === file).map(u => u.path);
        for (const p of paths.slice(0, 3)) {
          console.log(`        path: ${p}`);
        }
        if (paths.length > 3) {
          console.log(`        ... 他${paths.length - 3}件`);
        }
      }
      console.log('');
    }
  } else {
    console.log('✅ すべてのSFXが適切に分離されています。\n');
  }
  
  // insertOnlyリスト（簡易）
  if (insertOnly.length > 0) {
    console.log(`\n📋 挿入系専用SFX (${insertOnly.length}個):`);
    const sample = insertOnly.slice(0, 10);
    for (const item of sample) {
      console.log(`  - ${item.sfxId}`);
    }
    if (insertOnly.length > 10) {
      console.log(`  ... 他${insertOnly.length - 10}個`);
    }
  }
  
  // nonInsertOnlyリスト（簡易）
  if (nonInsertOnly.length > 0) {
    console.log(`\n📋 非挿入系専用SFX (${nonInsertOnly.length}個):`);
    const sample = nonInsertOnly.slice(0, 10);
    for (const item of sample) {
      console.log(`  - ${item.sfxId}`);
    }
    if (nonInsertOnly.length > 10) {
      console.log(`  ... 他${nonInsertOnly.length - 10}個`);
    }
  }
  
  console.log('');
}

// ==================== メイン処理 ====================

function main() {
  console.log('\n🔍 RPGツクールMZ 快感演出システム - JSON静的解析');
  console.log('='.repeat(80));
  
  // プロジェクトルートを推定
  const scriptDir = __dirname;
  const projectRoot = path.resolve(scriptDir, '../..');
  
  console.log(`📂 プロジェクトルート: ${projectRoot}`);
  console.log(`📂 解析対象ディレクトリ: ${TARGET_DIRS.join(', ')}`);
  
  // すべてのJSONファイルを収集
  let allJsonFiles = [];
  for (const dir of TARGET_DIRS) {
    const dirPath = path.join(projectRoot, dir);
    const files = getAllJsonFiles(dirPath, dir);
    allJsonFiles.push(...files);
  }
  
  console.log(`📄 検出されたJSONファイル: ${allJsonFiles.length}個\n`);
  
  if (allJsonFiles.length === 0) {
    console.log('⚠️  JSONファイルが見つかりませんでした。');
    return;
  }
  
  // 全JSONからオーバーレイ配列を収集
  let allOverlayArrays = [];
  let errorCount = 0;
  
  for (const { fullPath, relativePath } of allJsonFiles) {
    try {
      const content = fs.readFileSync(fullPath, 'utf-8');
      const jsonData = JSON.parse(content);
      const arrays = collectOverlayArrays(jsonData, '', relativePath, content);
      allOverlayArrays.push(...arrays);
    } catch (error) {
      errorCount++;
      if (errorCount <= 5) { // 最初の5件のみエラー表示
        console.log(`⚠️  JSONパースエラー: ${relativePath}`);
        console.log(`   ${error.message}`);
      }
    }
  }
  
  if (errorCount > 5) {
    console.log(`⚠️  ... 他${errorCount - 5}件のパースエラー\n`);
  }
  
  console.log(`📊 収集されたオーバーレイ配列: ${allOverlayArrays.length}個\n`);
  
  // チェックA: _left/_right 重複チェック
  const conflicts = findLeftRightConflicts(allOverlayArrays);
  reportLeftRightConflicts(conflicts);
  
  // チェックB: SFX挿入系/非挿入系 共用チェック
  const sfxClassification = classifySfxOverlayUsage(allOverlayArrays);
  reportSfxUsage(sfxClassification);
  
  console.log('='.repeat(80));
  console.log('✨ 解析完了\n');
}

// スクリプト実行
if (require.main === module) {
  main();
}

module.exports = {
  getAllJsonFiles,
  getBaseName,
  getDirection,
  isSfxOverlay,
  isMoanOverlay,
  isInsertContext,
  collectOverlayArrays,
  findLeftRightConflicts,
  classifySfxOverlayUsage
};

