#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
deepCheckDuplicates.py

全JSONファイルを直接読み込んで、同じオーバーレイIDが
同じoverlays配列内に2回以上出現していないか完全チェック
"""

import json
import sys
from pathlib import Path
from collections import Counter

# UTF-8出力設定
sys.stdout.reconfigure(encoding='utf-8')

def check_overlays_in_value(value, path="", results=None):
    """再帰的にoverlays配列をチェック"""
    if results is None:
        results = []
    
    if isinstance(value, dict):
        for key, val in value.items():
            new_path = f"{path}.{key}" if path else key
            
            # overlays配列を発見
            if key == "overlays" and isinstance(val, list):
                # 文字列のオーバーレイIDのみカウント
                overlay_ids = [item for item in val if isinstance(item, str)]
                if overlay_ids:
                    # 重複をチェック
                    counts = Counter(overlay_ids)
                    duplicates = {k: v for k, v in counts.items() if v > 1}
                    
                    if duplicates:
                        results.append({
                            'path': new_path,
                            'duplicates': duplicates,
                            'full_array': overlay_ids
                        })
            else:
                check_overlays_in_value(val, new_path, results)
    
    elif isinstance(value, list):
        for idx, item in enumerate(value):
            new_path = f"{path}[{idx}]"
            check_overlays_in_value(item, new_path, results)
    
    return results

def check_file(file_path):
    """単一ファイルをチェック"""
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            data = json.load(f)
        
        duplicates = check_overlays_in_value(data)
        return duplicates
    except Exception as e:
        print(f"[ERROR] {file_path}: {e}")
        return None

def main():
    print("=" * 80)
    print("🔍 オーバーレイ重複の完全チェック")
    print("=" * 80)
    print()
    
    # 対象ディレクトリ
    project_root = Path(__file__).parent.parent.parent
    target_dirs = [
        project_root / "dataEx" / "poseProfiles",
        project_root / "dataEx" / "restraintProfiles",
        project_root / "dataEx" / "introProfiles",
        project_root / "dataEx" / "orgasmDetailProfiles",
    ]
    
    all_files = []
    for target_dir in target_dirs:
        if target_dir.exists():
            # .backupファイルを除外
            json_files = [f for f in target_dir.rglob("*.json") 
                         if not f.name.endswith('.backup') 
                         and not f.name.endswith('.bak')]
            all_files.extend(json_files)
    
    print(f"📄 対象ファイル数: {len(all_files)}")
    print()
    
    total_duplicates = 0
    problem_files = []
    
    for file_path in sorted(all_files):
        relative_path = file_path.relative_to(project_root)
        duplicates = check_file(file_path)
        
        if duplicates is None:
            continue
        
        if duplicates:
            total_duplicates += len(duplicates)
            problem_files.append((relative_path, duplicates))
            
            print(f"⚠️  {relative_path}")
            for dup in duplicates:
                print(f"  パス: {dup['path']}")
                for overlay_id, count in dup['duplicates'].items():
                    print(f"    [{count}回] {overlay_id}")
                print(f"    配列: {dup['full_array']}")
                print()
    
    print("=" * 80)
    if total_duplicates == 0:
        print("✅ 重複なし！すべてのファイルが正常です。")
    else:
        print(f"❌ {len(problem_files)}ファイルで計{total_duplicates}箇所の重複が見つかりました。")
    print("=" * 80)
    print()

if __name__ == "__main__":
    main()

