#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
fixAllDuplicates.py

検出された全ての重複を自動修正
重複している2つ目以降を削除
"""

import json
import sys
from pathlib import Path
from collections import Counter

# UTF-8出力設定
sys.stdout.reconfigure(encoding='utf-8')

def fix_duplicates_in_array(overlay_array):
    """配列から重複を削除（最初の1つだけ残す）"""
    seen = set()
    fixed = []
    removed = []
    
    for item in overlay_array:
        if item in seen:
            removed.append(item)
        else:
            seen.add(item)
            fixed.append(item)
    
    return fixed, removed

def fix_overlays_in_value(value, path="", stats=None):
    """再帰的にoverlays配列を修正"""
    if stats is None:
        stats = {'fixed': 0, 'details': []}
    
    if isinstance(value, dict):
        result = {}
        for key, val in value.items():
            new_path = f"{path}.{key}" if path else key
            
            # overlays配列を発見
            if key == "overlays" and isinstance(val, list):
                overlay_ids = [item for item in val if isinstance(item, str)]
                non_overlay_items = [item for item in val if not isinstance(item, str)]
                
                if overlay_ids:
                    # 重複をチェック
                    counts = Counter(overlay_ids)
                    duplicates = {k: v for k, v in counts.items() if v > 1}
                    
                    if duplicates:
                        fixed, removed = fix_duplicates_in_array(overlay_ids)
                        result[key] = fixed + non_overlay_items
                        stats['fixed'] += 1
                        stats['details'].append({
                            'path': new_path,
                            'removed': removed,
                            'before': val,
                            'after': result[key]
                        })
                    else:
                        result[key] = val
                else:
                    result[key] = val
            else:
                result[key] = fix_overlays_in_value(val, new_path, stats)
        return result
    
    elif isinstance(value, list):
        result = []
        for idx, item in enumerate(value):
            new_path = f"{path}[{idx}]"
            result.append(fix_overlays_in_value(item, new_path, stats))
        return result
    
    else:
        return value

def fix_file(file_path):
    """単一ファイルを修正"""
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            data = json.load(f)
        
        stats = {'fixed': 0, 'details': []}
        fixed_data = fix_overlays_in_value(data, "", stats)
        
        if stats['fixed'] > 0:
            # バックアップ作成
            backup_path = file_path.with_suffix(file_path.suffix + '.backup')
            with open(backup_path, 'w', encoding='utf-8') as f:
                json.dump(data, f, ensure_ascii=False, indent=2)
            
            # 修正版を保存
            with open(file_path, 'w', encoding='utf-8') as f:
                json.dump(fixed_data, f, ensure_ascii=False, indent=2)
            
            return stats
        else:
            return None
            
    except Exception as e:
        print(f"[ERROR] {file_path}: {e}")
        return None

def main():
    print("=" * 80)
    print("🔧 オーバーレイ重複の自動修正")
    print("=" * 80)
    print()
    
    # 修正対象ファイル（deepCheckDuplicates.pyで検出されたもの）
    project_root = Path(__file__).parent.parent.parent
    target_files = [
        "dataEx/poseProfiles/goblin_restrained_insert_low.json",
        "dataEx/poseProfiles/スライム関係/slime_restrained_low.json",
        "dataEx/poseProfiles/堕落度high/触手堕落度high/挿入時絶頂描写.json",
        "dataEx/poseProfiles/堕落度low/ゴブリン関係/ゴブリンエロ攻撃セリフ_挿入（転移済み）.json"
    ]
    
    fixed_count = 0
    total_fixes = 0
    
    for file_rel in target_files:
        file_path = project_root / file_rel
        if not file_path.exists():
            print(f"⚠️  ファイルが見つかりません: {file_rel}")
            continue
        
        print(f"📄 処理中: {file_rel}")
        stats = fix_file(file_path)
        
        if stats:
            fixed_count += 1
            total_fixes += stats['fixed']
            
            for detail in stats['details']:
                print(f"  ✅ 修正: {detail['path']}")
                for removed in detail['removed']:
                    print(f"     削除: {removed}")
            
            print(f"  💾 バックアップ作成済み")
            print()
        else:
            print(f"  ℹ️  重複なし")
            print()
    
    print("=" * 80)
    if fixed_count > 0:
        print(f"✨ 完了: {fixed_count}ファイル、計{total_fixes}箇所を修正しました")
    else:
        print("✅ 修正が必要なファイルはありませんでした")
    print("=" * 80)
    print()

if __name__ == "__main__":
    main()

