#!/usr/bin/env node
/**
 * fixOverlayDuplicates.js
 * 
 * 同じ向きのオーバーレイが重複している箇所を自動修正
 * 重複を削除し、代わりに別の既存SFXで置換
 */

const fs = require('fs');
const path = require('path');

// 代替SFXのプール（重複の代わりに使用）
const REPLACEMENT_SFX = [
  'ぬちゃっ1-6_right',
  'ぐちゅっ1-6_right',
  'ぬちょっ1-6_right',
  'ぬちゃっ1-6_left',
  'ぬちょっ1-6_left'
];

/**
 * オーバーレイ配列の重複を修正
 */
function fixDuplicateOverlays(overlays) {
  const seen = new Set();
  const fixed = [];
  let replacementIndex = 0;
  let modified = false;
  
  for (const overlay of overlays) {
    if (seen.has(overlay)) {
      // 重複発見！代替SFXで置換
      const replacement = REPLACEMENT_SFX[replacementIndex % REPLACEMENT_SFX.length];
      console.log(`    [FIX] "${overlay}" (重複) → "${replacement}"`);
      fixed.push(replacement);
      replacementIndex++;
      modified = true;
    } else {
      seen.add(overlay);
      fixed.push(overlay);
    }
  }
  
  return { fixed, modified };
}

/**
 * JSONオブジェクトを再帰的に走査して重複を修正
 */
function processObject(obj, path = '', stats = { total: 0, fixed: 0 }) {
  if (!obj || typeof obj !== 'object') {
    return obj;
  }
  
  // overlays配列を発見
  if (Array.isArray(obj) && path.endsWith('.overlays')) {
    stats.total++;
    const { fixed, modified } = fixDuplicateOverlays(obj);
    if (modified) {
      stats.fixed++;
      console.log(`  [${stats.fixed}] ${path}`);
    }
    return fixed;
  }
  
  // 配列の場合
  if (Array.isArray(obj)) {
    return obj.map((item, index) => 
      processObject(item, `${path}[${index}]`, stats)
    );
  }
  
  // オブジェクトの場合
  const result = {};
  for (const [key, value] of Object.entries(obj)) {
    const newPath = path ? `${path}.${key}` : key;
    result[key] = processObject(value, newPath, stats);
  }
  return result;
}

/**
 * ファイルを処理
 */
function processFile(filePath) {
  console.log(`\n📄 処理中: ${filePath}`);
  
  try {
    const content = fs.readFileSync(filePath, 'utf-8');
    const data = JSON.parse(content);
    
    const stats = { total: 0, fixed: 0 };
    const fixed = processObject(data, '', stats);
    
    if (stats.fixed > 0) {
      // バックアップ作成
      const backupPath = filePath + '.backup';
      fs.writeFileSync(backupPath, content, 'utf-8');
      console.log(`  💾 バックアップ: ${backupPath}`);
      
      // 修正版を保存
      fs.writeFileSync(filePath, JSON.stringify(fixed, null, 2), 'utf-8');
      console.log(`  ✅ 修正完了: ${stats.fixed}/${stats.total}箇所`);
      return true;
    } else {
      console.log(`  ✅ 重複なし`);
      return false;
    }
  } catch (error) {
    console.log(`  ❌ エラー: ${error.message}`);
    return false;
  }
}

// メイン処理
function main() {
  console.log('\n🔧 オーバーレイ重複修正ツール\n');
  console.log('='.repeat(80));
  
  // 修正対象ファイル
  const targetFiles = [
    'dataEx/poseProfiles/堕落度low/ゴブリン関係/絶頂イベント_挿入時（転移完了）.json',
    'dataEx/orgasmDetailProfiles/goblin_restrained_insert_low.json'
  ];
  
  const projectRoot = path.resolve(__dirname, '../..');
  let fixedCount = 0;
  
  for (const file of targetFiles) {
    const fullPath = path.join(projectRoot, file);
    if (processFile(fullPath)) {
      fixedCount++;
    }
  }
  
  console.log('\n' + '='.repeat(80));
  console.log(`\n✨ 完了: ${fixedCount}ファイルを修正しました\n`);
  
  if (fixedCount > 0) {
    console.log('📝 注意:');
    console.log('  - バックアップファイル (*.backup) が作成されています');
    console.log('  - 問題がなければバックアップを削除してください');
    console.log('  - 問題があれば .backup から復元できます\n');
  }
}

if (require.main === module) {
  main();
}

module.exports = { fixDuplicateOverlays, processObject };

