#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
removeVisualsOverlays.py

visuals セクションから slime_breast 系オーバーレイと underOverlays を削除
- lines セクションは変更しない
- visuals セクションのみ対象
"""

import json
import sys
from pathlib import Path

sys.stdout.reconfigure(encoding='utf-8')

def clean_visuals_section(obj, path="", removed_count=0):
    """visuals セクション内の不要なオーバーレイを削除"""
    if isinstance(obj, dict):
        # visuals セクションを見つけたら処理
        if path.endswith('visuals') or '/visuals/' in path:
            # overlays から slime_breast を削除
            if 'overlays' in obj and isinstance(obj['overlays'], list):
                original = obj['overlays'].copy()
                obj['overlays'] = [o for o in obj['overlays'] if 'slime_breast' not in str(o)]
                if len(original) != len(obj['overlays']):
                    removed_items = [o for o in original if o not in obj['overlays']]
                    removed_count += len(removed_items)
                    print(f"  [削除] {path}/overlays: {removed_items}")
            
            # underOverlays を削除
            if 'underOverlays' in obj:
                removed_count += 1
                print(f"  [削除] {path}/underOverlays: {obj['underOverlays']}")
                del obj['underOverlays']
        
        # 再帰的に処理
        for key, value in obj.items():
            new_path = f"{path}/{key}" if path else key
            removed_count = clean_visuals_section(value, new_path, removed_count)
    
    elif isinstance(obj, list):
        for idx, item in enumerate(obj):
            new_path = f"{path}[{idx}]"
            removed_count = clean_visuals_section(item, new_path, removed_count)
    
    return removed_count

def process_file(file_path):
    """単一ファイルを処理"""
    print(f"\n📄 処理中: {file_path.name}")
    
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            data = json.load(f)
        
        removed_count = clean_visuals_section(data)
        
        if removed_count > 0:
            # バックアップ作成
            backup_path = file_path.with_suffix(file_path.suffix + '.backup3')
            with open(backup_path, 'w', encoding='utf-8') as f:
                json.dump(json.loads(open(file_path, 'r', encoding='utf-8').read()), 
                         f, ensure_ascii=False, indent=2)
            
            # 修正版を保存
            with open(file_path, 'w', encoding='utf-8') as f:
                json.dump(data, f, ensure_ascii=False, indent=2)
            
            print(f"  ✅ 完了: {removed_count}箇所を削除")
            print(f"  💾 バックアップ: {backup_path.name}")
            return True
        else:
            print(f"  ℹ️  削除不要（既にクリーン）")
            return False
            
    except Exception as e:
        print(f"  ❌ エラー: {e}")
        return False

def main():
    print("=" * 80)
    print("🧹 visuals セクションクリーンアップツール")
    print("=" * 80)
    print("📝 処理内容:")
    print("   - visuals セクションから slime_breast_low/high を削除")
    print("   - visuals セクションから underOverlays を削除")
    print("   - lines セクションは変更なし")
    print("=" * 80)
    
    project_root = Path(__file__).parent.parent.parent
    
    # 処理対象ファイル
    target_files = [
        project_root / 'dataEx/poseProfiles/slime_restrained_low.json',
        project_root / 'dataEx/poseProfiles/slime_restrained_high.json',
        project_root / 'dataEx/poseProfiles/slime_restrained_insert_low.json',
        project_root / 'dataEx/poseProfiles/slime_restrained_insert_high.json',
    ]
    
    fixed_count = 0
    
    for file_path in target_files:
        if not file_path.exists():
            print(f"\n⚠️  ファイルが見つかりません: {file_path.name}")
            continue
        
        if process_file(file_path):
            fixed_count += 1
    
    print("\n" + "=" * 80)
    if fixed_count > 0:
        print(f"✨ 完了: {fixed_count}ファイルをクリーンアップしました")
        print(f"💡 visuals セクションがシンプルになりました")
    else:
        print("✅ すべてのファイルが既にクリーンです")
    print("=" * 80)
    print()

if __name__ == "__main__":
    main()

