#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
removeVisualsUnderOverlays.py

visuals セクション内の全ての underOverlays を削除
"""

import json
import sys
from pathlib import Path

sys.stdout.reconfigure(encoding='utf-8')

def remove_under_overlays_from_visuals(data):
    """visuals セクション内の全ての underOverlays を削除"""
    removed_count = 0
    
    if 'visuals' not in data:
        return removed_count
    
    visuals = data['visuals']
    
    # visuals 内の全ての階層を再帰的に処理
    def process_dict(obj, path="visuals"):
        nonlocal removed_count
        
        if not isinstance(obj, dict):
            return
        
        # underOverlays があれば削除
        if 'underOverlays' in obj:
            print(f"  [削除] {path}/underOverlays: {obj['underOverlays']}")
            del obj['underOverlays']
            removed_count += 1
        
        # 再帰的に処理
        for key, value in list(obj.items()):
            if isinstance(value, dict):
                process_dict(value, f"{path}.{key}")
    
    process_dict(visuals)
    return removed_count

def process_file(file_path):
    """単一ファイルを処理"""
    print(f"\n📄 処理中: {file_path.name}")
    
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            data = json.load(f)
        
        removed_count = remove_under_overlays_from_visuals(data)
        
        if removed_count > 0:
            # バックアップ作成
            backup_path = file_path.with_suffix(file_path.suffix + '.backup4')
            with open(backup_path, 'w', encoding='utf-8') as f:
                json.dump(json.loads(open(file_path, 'r', encoding='utf-8').read()), 
                         f, ensure_ascii=False, indent=2)
            
            # 修正版を保存
            with open(file_path, 'w', encoding='utf-8') as f:
                json.dump(data, f, ensure_ascii=False, indent=2)
            
            print(f"  ✅ 完了: {removed_count}箇所の underOverlays を削除")
            print(f"  💾 バックアップ: {backup_path.name}")
            return removed_count
        else:
            print(f"  ℹ️  削除不要（underOverlays なし）")
            return 0
            
    except Exception as e:
        print(f"  ❌ エラー: {e}")
        import traceback
        traceback.print_exc()
        return 0

def main():
    print("=" * 80)
    print("🧹 visuals セクション underOverlays 削除ツール")
    print("=" * 80)
    print("📝 処理内容:")
    print("   - visuals セクション内の全ての underOverlays を削除")
    print("   - lines セクションは変更なし")
    print("=" * 80)
    
    project_root = Path(__file__).parent.parent.parent
    
    # 処理対象ファイル
    target_files = [
        project_root / 'dataEx/poseProfiles/slime_restrained_low.json',
        project_root / 'dataEx/poseProfiles/slime_restrained_high.json',
        project_root / 'dataEx/poseProfiles/slime_restrained_insert_low.json',
        project_root / 'dataEx/poseProfiles/slime_restrained_insert_high.json',
    ]
    
    total_removed = 0
    fixed_count = 0
    
    for file_path in target_files:
        if not file_path.exists():
            print(f"\n⚠️  ファイルが見つかりません: {file_path.name}")
            continue
        
        removed = process_file(file_path)
        if removed > 0:
            fixed_count += 1
            total_removed += removed
    
    print("\n" + "=" * 80)
    if fixed_count > 0:
        print(f"✨ 完了: {fixed_count}ファイルをクリーンアップしました")
        print(f"📊 合計: {total_removed}箇所の underOverlays を削除")
    else:
        print("✅ すべてのファイルが既にクリーンです")
    print("=" * 80)
    print()

if __name__ == "__main__":
    main()

