	/*---------------------------------------------------------------------------*
	 * 2025/10/23 改造版（xo ekus）
	 * 元: PictureColorChange.js by kido
	 *---------------------------------------------------------------------------*/

	/*:
	 * @plugindesc ピクチャの色相・色合い・グレー値（彩度）を変更するプラグイン
	 * @author kido + xo ekus
	 * @help
	 * ▼機能一覧
	 *  - RotateHue : 色相を回転
	 *  - SetTint   : 色合い（RGB補正）
	 *  - SetGray   : グレー値（彩度）を変更
	 * 
	 * -----------------------------------------------------
	 * プラグインコマンド
	 * -----------------------------------------------------
	 *  RotateHue ピクチャ番号 角度
	 *    例: RotateHue 1 180
	 * 
	 *  SetTint ピクチャ番号 色
	 *    例: SetTint 1 0xFF0000
	 *    または RGB指定：
	 *    例: SetTint 1 255 0 0
	 * 
	 *  SetGray ピクチャ番号 グレー値
	 *    グレー値: 0（通常）～255（完全に白黒）
	 *    例: SetGray 1 128
	 * -----------------------------------------------------
	 */

	(function()
	{
		var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
		Game_Interpreter.prototype.pluginCommand = function(command, args)
		{
			_Game_Interpreter_pluginCommand.call(this, command, args);

			if (command === 'RotateHue') {
			  if(args.length < 2) return console.error('ピクチャ番号と色相の値を入れてください');
			  var pic = $gameScreen.picture(parseInt(convertEscapeCharacter(args[0])));
			  if(!pic) return;
			  var hue = parseInt(convertEscapeCharacter(args[1]));
			  if(Number.isNaN(hue)) return console.error('色相の値が正しくありません');
			  pic.rotateHue(hue);

			} else if (command === 'SetTint') {
			  if(args.length != 2 && args.length != 4) return console.error('ピクチャ番号と色合いの値を入れてください');
			  var pic = $gameScreen.picture(parseInt(convertEscapeCharacter(args[0])));
			  if(!pic) return;

			  var tint;
			  if(args.length == 2){
				tint = parseInt(convertEscapeCharacter(args[1]));
			  }else{
				var r = parseInt(convertEscapeCharacter(args[1]));
				var g = parseInt(convertEscapeCharacter(args[2]));
				var b = parseInt(convertEscapeCharacter(args[3]));
				if(Number.isNaN(r) || Number.isNaN(g) || Number.isNaN(b)) return console.error('色合いの値が正しくありません');
				tint =  r << 16 | g << 8 | b;
			  }
			  if(Number.isNaN(tint)) return console.error('色合いの値が正しくありません');
			  pic.setTint(tint);

			}
			else if (command === 'SetGray')
			{
				if (args.length < 2)
				{ return console.error('ピクチャ番号とグレー値を入れてください'); } 

				  var pic = $gameScreen.picture(parseInt(convertEscapeCharacter(args[0])));

				if (!pic)
				{ return; }
				
				var gray = parseInt(convertEscapeCharacter(args[1]));

				if (Number.isNaN(gray))
				{ return console.error('グレー値が正しくありません'); }

				gray = Math.max(0, Math.min(255, gray)); // 範囲制限
				pic.setGray(gray);
			}
		};

		function convertEscapeCharacter(text){
			text = text.replace(/\\/g, '\x1b');
			text = text.replace(/\x1bV\[(\d+)\]/gi, function() {
				return $gameVariables.value(parseInt(arguments[1]));
			}.bind(this));
			return text;
		}

	  // ピクチャクラス拡張
	  Game_Picture.prototype.rotateHue = function(hue) { this.hue = hue; };
	  Game_Picture.prototype.setTint = function(tint) { this.tint = tint; };
	  Game_Picture.prototype.setGray = function(gray) { this.gray = gray; };
	  Game_Picture.prototype.resetHue = function() { this.hue = 0; };
	  Game_Picture.prototype.resetTint = function() { this.tint = 0xFFFFFF; };
	  Game_Picture.prototype.resetGray = function() { this.gray = 0; };

	  var _show = Game_Picture.prototype.show;
	  Game_Picture.prototype.show = function(name, origin, x, y, scaleX, scaleY, opacity, blendMode) {
		_show.call(this, name, origin, x, y, scaleX, scaleY, opacity, blendMode);
		this.resetHue();
		this.resetTint();
		this.resetGray();
	  };

	  var _updateBitmap = Sprite_Picture.prototype.updateBitmap;
	  Sprite_Picture.prototype.updateBitmap = function() {
		var picture = this.picture();
		var isPictureChanged = picture && picture.name() !== this._pictureName;
		_updateBitmap.call(this); 

		if(!picture){
		  this.__hue = 0; this.__tint = 0xFFFFFF; this.__gray = 0;
		  return;
		}

		var isHueChanged = picture.hue != this.__hue;
		var isTintChanged = picture.tint != this.__tint;
		var isGrayChanged = picture.gray != this.__gray;

		if(isHueChanged || isTintChanged || isGrayChanged || isPictureChanged){
		  this.__hue = picture.hue;
		  this.__tint = picture.tint;
		  this.__gray = picture.gray;
		  this.bitmap.addLoadListener(function(){
			if(isPictureChanged){
			  this.defaultBitmap = this.bitmap;
			}
			var bitmap = Object.create(Bitmap.prototype);
			bitmap.initialize(this.defaultBitmap.width, this.defaultBitmap.height);
			bitmap.smooth = true;
			bitmap.blt(this.defaultBitmap, 0, 0, this.defaultBitmap.width, this.defaultBitmap.height, 0, 0);
			this.bitmap = bitmap;
			if(isTintChanged || this.__tint != 0xFFFFFF) this.bitmap.setTint(this.__tint);
			if(isHueChanged || this.__hue != 0) this.bitmap.rotateHue(this.__hue);
			if(isGrayChanged || this.__gray != 0) this.bitmap.setGray(this.__gray);
		  }.bind(this));
		}
	  };

	  Bitmap.prototype.setTint = function(color){
		function hex2rgb(hex, out){
		  out = out || [];
		  out[0] = ((hex >> 16) & 0xFF) / 255;
		  out[1] = ((hex >> 8) & 0xFF) / 255;
		  out[2] = (hex & 0xFF) / 255;
		  return out;
		}
		const context = this._context;
		const rgbValues = hex2rgb(color);
		const r = rgbValues[0], g = rgbValues[1], b = rgbValues[2];
		const pixelData = context.getImageData(0, 0, this.width, this.height);
		const pixels = pixelData.data;
		for (var i = 0; i < pixels.length; i += 4){
		  pixels[i + 0] *= r;
		  pixels[i + 1] *= g;
		  pixels[i + 2] *= b;
		}
		context.putImageData(pixelData, 0, 0);
		this._setDirty();
	  };

	  // ★ グレー化処理（明度・彩度調整）
	  Bitmap.prototype.setGray = function(gray){
		const context = this._context;
		const pixelData = context.getImageData(0, 0, this.width, this.height);
		const pixels = pixelData.data;
		const grayRate = gray / 255;
		for (let i = 0; i < pixels.length; i += 4) {
		  const avg = (pixels[i] + pixels[i + 1] + pixels[i + 2]) / 3;
		  pixels[i]     = pixels[i] * (1 - grayRate) + avg * grayRate;
		  pixels[i + 1] = pixels[i + 1] * (1 - grayRate) + avg * grayRate;
		  pixels[i + 2] = pixels[i + 2] * (1 - grayRate) + avg * grayRate;
		}
		context.putImageData(pixelData, 0, 0);
		this._setDirty();
	  };

	})();
