/*:
 * @plugindesc v1.0 指定スイッチがONになるまでプレイヤー操作をロックする（移動/決定/キャンセル/メニュー等）
 * @author xo
 *
 * @param LockSwitchId
 * @text ロック解除スイッチID
 * @type switch
 * @default 1
 *
 * @param DisableTouch
 * @text タッチ操作も無効化
 * @type boolean
 * @default true
 *
 * @param AlsoDisableMouse
 * @text マウスクリック(左/右)も無効化
 * @type boolean
 * @default true
 *
 * @help
 * ■概要
 *  - 指定スイッチがOFFの間、プレイヤーの操作を受け付けなくします。
 *  - スイッチがONになると自動で解除されます。
 *
 * ■想定用途
 *  - 演出中/画像ロード待ち/チュートリアル中など、操作を一時停止したい時
 *  - 並列イベントで条件成立→スイッチON→操作解禁
 *
 * ■注意
 *  - イベント処理（並列/自動実行）は止まりません。
 *  - 「メニュー禁止」も含め、入力自体を遮断します。
 *
 * ■プラグインコマンド
 *  LockInput 12     # スイッチ12がONになるまでロック（以後そのIDが解除条件）
 *  UnlockInput      # 強制解除（スイッチに関係なく解除）
 *  LockInputOn      # 現在の解除条件スイッチを使ってロックON
 *  LockInputOff     # ロックOFF（UnlockInputと同等）
 *
 * ■スクリプトから
 *  SwitchInputLock.lock(12);
 *  SwitchInputLock.unlock();
 */

(function() {
    "use strict";

    var pluginName = "SwitchInputLock";
    var params = PluginManager.parameters(pluginName);

    var defaultSwitchId = Number(params["LockSwitchId"] || 1);
    var disableTouch = String(params["DisableTouch"] || "true") === "true";
    var alsoDisableMouse = String(params["AlsoDisableMouse"] || "true") === "true";

    // グローバルAPI（呼び出し用）
    window.SwitchInputLock = window.SwitchInputLock || {};
    var API = window.SwitchInputLock;

    // 内部状態
    var _lockEnabled = true;          // ロック機能自体がONか（初期はON：スイッチOFFならロック）
    var _unlockSwitchId = defaultSwitchId;

	function isLockedNow() {
		if (!_lockEnabled) return false;

		// ▼ ここが重要：データ未初期化(タイトル等)ではロックしない
		if (!$gameSwitches) return false;

		// スイッチIDが不正/未設定っぽい時も安全側
		var id = Number(_unlockSwitchId || defaultSwitchId || 1);
		if (id <= 0) return false;

		// スイッチがONなら解除、OFFならロック
		return !$gameSwitches.value(id);
	}

    API.lock = function(switchId) {
        _unlockSwitchId = Number(switchId || _unlockSwitchId || defaultSwitchId);
        _lockEnabled = true;
    };

    API.unlock = function() {
        _lockEnabled = false;
    };

    API.setSwitch = function(switchId) {
        _unlockSwitchId = Number(switchId || _unlockSwitchId || defaultSwitchId);
    };

    API.isLocked = function() {
        return isLockedNow();
    };

    // ------------------------------------------------------------
    // 入力遮断：Input / TouchInput を潰す
    // ------------------------------------------------------------

    // 通常入力（キーボード/ゲームパッド）
    var _Input_update = Input.update;
    Input.update = function() {
        _Input_update.call(this);

        if (SceneManager._scene && isLockedNow()) {
            // そのフレームの入力を全消し
            this._currentState = {};
            this._latestButton = null;
            this._pressedTime = 0;
        }
    };

    // タッチ/クリック
    if (disableTouch) {
        var _TouchInput_update = TouchInput.update;
        TouchInput.update = function() {
            _TouchInput_update.call(this);

            if (SceneManager._scene && isLockedNow()) {
                // タッチ状態を全消し
                this._triggered = false;
                this._cancelled = false;
                this._moved = false;
                this._released = false;
                this._wheelX = 0;
                this._wheelY = 0;

                // 押下中も解除
                this._pressed = false;
            }
        };
    }

    // 念のためマウス左右クリックも止めたい場合（TouchInputに入るが、環境差対策）
    if (alsoDisableMouse) {
        var _TouchInput_onLeftButtonDown = TouchInput._onLeftButtonDown;
        TouchInput._onLeftButtonDown = function(event) {
            if (SceneManager._scene && isLockedNow()) return;
            _TouchInput_onLeftButtonDown.call(this, event);
        };

        var _TouchInput_onRightButtonDown = TouchInput._onRightButtonDown;
        TouchInput._onRightButtonDown = function(event) {
            if (SceneManager._scene && isLockedNow()) return;
            _TouchInput_onRightButtonDown.call(this, event);
        };
    }

    // ------------------------------------------------------------
    // 追加で「プレイヤー自体の移動処理」も抑止（入力以外からの移動は許可）
    // ------------------------------------------------------------
    var _Game_Player_canMove = Game_Player.prototype.canMove;
    Game_Player.prototype.canMove = function() {
        if (SceneManager._scene && isLockedNow()) return false;
        return _Game_Player_canMove.call(this);
    };

    // ------------------------------------------------------------
    // プラグインコマンド（MV）
    // ------------------------------------------------------------
    var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
    Game_Interpreter.prototype.pluginCommand = function(command, args) {
        _Game_Interpreter_pluginCommand.call(this, command, args);

        if (!command) return;
        switch (command) {
            case "LockInput":
                API.lock(Number(args[0] || _unlockSwitchId || defaultSwitchId));
                break;
            case "UnlockInput":
            case "LockInputOff":
                API.unlock();
                break;
            case "LockInputOn":
                _lockEnabled = true;
                break;
        }
    };

})();
