/*:
 * @plugindesc PNG内iTXtからcCharaを読み込む（XoChara） v1.0.0
 * @author ChatGPT
 *
 * @help
 * ■概要
 * PNGの iTXt チャンク(keyword指定)に埋め込まれた JSON を読み取り、
 * cChara 1体分を復元します。
 *
 * ■プラグインコマンド
 * 1) パス指定（NW.js向け）
 *   LOAD_CHARA_PNG C:\path\to\capture.png
 *   LOAD_CHARA_PNG ./captures/capture.png
 *
 * 2) パス省略（NW.js/ブラウザ両対応）
 *   LOAD_CHARA_PNG
 *   → ファイル選択ダイアログが出ます（pngを選択）
 *
 * ■復元先
 * - デフォルト: window.Target にセット
 * - SetToChr が true の場合: Chr[index] にもセット
 *   indexは IndexVar(変数ID) の値を使用
 *
 * ■注意
 * - 保存側が「iTXtの非圧縮（compression flag=0）」で入れている前提。
 *   （圧縮(iTXt flag=1)は現状未対応）
 *
 * @param ChunkKeyword
 * @text iTXtのkeyword
 * @type string
 * @default XoChara
 *
 * @param Magic
 * @text magic文字列
 * @type string
 * @default XOCHARA
 *
 * @param SetTarget
 * @text window.Targetにセットする
 * @type boolean
 * @default true
 *
 * @param SetToChr
 * @text Chr[index]にもセットする
 * @type boolean
 * @default false
 *
 * @param IndexVar
 * @text Chrのindexに使う変数ID
 * @type variable
 * @default 0
 *
 * @param OkSwitch
 * @text 読み込み成功でONにするスイッチID
 * @type switch
 * @default 0
 * 
 *
 * @param NgSwitch
 * @text 読み込み失敗でONにするスイッチID
 * @type switch
 * @default 0
 */

(function(){
	'use strict';

	// =========================================================
	// Params
	// =========================================================
	var PLUGIN_NAME = 'XoPngLoadChara'; // ★ファイル名（拡張子なし）に合わせてね
	var params = PluginManager.parameters(PLUGIN_NAME);

	var CHUNK_KEYWORD = String(params.ChunkKeyword || 'XoChara');
	var MAGIC         = String(params.Magic || 'XOCHARA');

	var SET_TARGET = String(params.SetTarget || 'true') === 'true';
	var SET_TO_CHR = String(params.SetToChr || 'false') === 'true';

	var INDEX_VAR  = Number(params.IndexVar || 0) | 0;
	var OK_SWITCH  = Number(params.OkSwitch || 0) | 0;
	var NG_SWITCH  = Number(params.NgSwitch || 0) | 0;



	// =========================================================
	// UTF-8 decode
	// =========================================================
	function utf8Decode(u8){
		if (window.TextDecoder) return new TextDecoder('utf-8').decode(u8);

		// fallback
		var out = '', i = 0;
		while (i < u8.length) {
			var c = u8[i++];
			if (c < 0x80) out += String.fromCharCode(c);
			else if ((c & 0xE0) === 0xC0) {
				var c2 = u8[i++] & 0x3F;
				out += String.fromCharCode(((c & 0x1F) << 6) | c2);
			} else if ((c & 0xF0) === 0xE0) {
				var c2b = u8[i++] & 0x3F, c3 = u8[i++] & 0x3F;
				out += String.fromCharCode(((c & 0x0F) << 12) | (c2b << 6) | c3);
			} else {
				// 4byte → surrogate pair
				var c2c = u8[i++] & 0x3F, c3b = u8[i++] & 0x3F, c4 = u8[i++] & 0x3F;
				var cp = ((c & 0x07) << 18) | (c2c << 12) | (c3b << 6) | c4;
				cp -= 0x10000;
				out += String.fromCharCode(0xD800 + (cp >> 10));
				out += String.fromCharCode(0xDC00 + (cp & 0x3FF));
			}
		}
		return out;
	}

	// =========================================================
	// PNG iTXt parse
	// =========================================================
	function readU32be(u8, off){
		return ((u8[off] << 24) | (u8[off+1] << 16) | (u8[off+2] << 8) | (u8[off+3])) >>> 0;
	}

	function isPng(u8){
		if (!u8 || u8.length < 8) return false;
		var sig = [137,80,78,71,13,10,26,10];
		for (var i=0;i<8;i++) if (u8[i] !== sig[i]) return false;
		return true;
	}

	function parseITXtData(dataU8){
		// iTXt:
		// keyword\0
		// compression flag(1)
		// compression method(1)
		// language tag\0
		// translated keyword\0
		// text (UTF-8, compressed if flag=1)
		var p = 0;

		// keyword (latin-1) until null
		var kStart = 0;
		while (p < dataU8.length && dataU8[p] !== 0) p++;
		var keyword = String.fromCharCode.apply(null, Array.prototype.slice.call(dataU8.slice(kStart, p)));
		p++; // null

		if (p + 2 > dataU8.length) return null;
		var compFlag = dataU8[p++];     // 0/1
		var compMeth = dataU8[p++];     // 0

		// language tag
		while (p < dataU8.length && dataU8[p] !== 0) p++;
		p++; // null

		// translated keyword
		while (p < dataU8.length && dataU8[p] !== 0) p++;
		p++; // null

		var textBytes = dataU8.slice(p);

		if (compFlag !== 0) {
			// 今回は保存側が compFlag=0 の前提。必要なら zlib inflate を追加できる
			throw new Error('iTXtが圧縮されています（compFlag=1）。現バージョンは未対応です。');
		}

		var text = utf8Decode(textBytes);
		return { keyword: keyword, text: text };
	}

	function extractEmbeddedJsonFromPng(u8, keywordWanted){
		if (!isPng(u8)) throw new Error('PNGではありません');

		var off = 8; // after signature
		while (off + 12 <= u8.length) {
			var len = readU32be(u8, off);
			var type = String.fromCharCode(u8[off+4], u8[off+5], u8[off+6], u8[off+7]);
			var dataOff = off + 8;
			var dataEnd = dataOff + len;
			if (dataEnd + 4 > u8.length) break;

			if (type === 'iTXt') {
				var data = u8.slice(dataOff, dataEnd);
				var parsed = parseITXtData(data);
				if (parsed && parsed.keyword === keywordWanted) {
					return parsed.text;
				}
			}
			if (type === 'IEND') break;
			off += 12 + len;
		}
		return null;
	}

	// =========================================================
	// Restore cChara
	// =========================================================
	function deepClone(obj){
		// JSON由来を想定（循環なし）
		if (obj === null || typeof obj !== 'object') return obj;
		if (Array.isArray(obj)) return obj.map(deepClone);
		var out = {};
		Object.keys(obj).forEach(function(k){ out[k] = deepClone(obj[k]); });
		return out;
	}

	function restoreCharaFromPayload(payload){
		if (!payload || payload.magic !== MAGIC) throw new Error('magic不一致');
		if (!payload.chara || typeof payload.chara !== 'object') throw new Error('charaがありません');

		var base = deepClone(payload.chara);

		// cChara が存在するならインスタンス化して詰める
		if (typeof window.cChara === 'function') {
			var inst = new window.cChara();
			Object.keys(base).forEach(function(k){ inst[k] = base[k]; });
			return inst;
		}

		// ない場合はplain objectで返す
		return base;
	}

	function applyLoadedChara(ch){
		if (SET_TARGET) window.Target = ch;

		if (SET_TO_CHR && Array.isArray(window.Chr) && INDEX_VAR > 0) {
			var idx = $gameVariables.value(INDEX_VAR) | 0;
			window.Chr[idx] = ch;
		}
	}

	// =========================================================
	// Load (NW.js path / file picker)
	// =========================================================
	function loadFromPathNWjs(pathStr){
		var fs = require('fs');
		var buf = fs.readFileSync(pathStr);
		var u8 = new Uint8Array(buf);
		return u8;
	}

	function pickFileAsUint8(callback){
		var input = document.createElement('input');
		input.type = 'file';
		input.accept = 'image/png';
		input.style.display = 'none';
		document.body.appendChild(input);

		input.addEventListener('change', function(){
			var file = input.files && input.files[0];
			document.body.removeChild(input);
			if (!file) return;

			var reader = new FileReader();
			reader.onload = function(){
				var arr = reader.result; // ArrayBuffer
				callback(new Uint8Array(arr));
			};
			reader.readAsArrayBuffer(file);
		});

		input.click();
	}

	// =========================================================
	// Main command
	// =========================================================
	function doLoadPng(pathStr)
	{
		// 先にスイッチを落とす運用が欲しければここでやる（今回は成功/失敗だけ対応
		$gameSwitches.setValue(72, false);
		$gameSwitches.setValue(73, false);

		var onSuccess = function(u8)
		{
			try
			{
			console.log(CHUNK_KEYWORD);
				var jsonText = extractEmbeddedJsonFromPng(u8, CHUNK_KEYWORD);
				if (!jsonText) throw new Error('埋め込みデータが見つかりません（iTXt keyword不一致 or なし）');

				var payload = JSON.parse(jsonText);
				var chara = restoreCharaFromPayload(payload);

				applyLoadedChara(chara);

				// 成功スイッチ
				$gameSwitches.setValue(72, true);
			}
			catch (e)
			{
				console.error('[LOAD_CHARA_PNG] Failed:', e);
				// 失敗スイッチ
				$gameSwitches.setValue(73, true);
			}
		};

		// パス指定あり：NW.jsで読む
		if (pathStr && pathStr.length > 0) {
			try 
			{
				if (!Utils.isNwjs()) throw new Error('パス指定読み込みはNW.js(PC版)前提です');
				var u8 = loadFromPathNWjs(pathStr);
				onSuccess(u8);
			}
			catch (e) 
			{
				console.error('[LOAD_CHARA_PNG] Failed:', e);
				// 失敗スイッチ
				$gameSwitches.setValue(73, true);
			}
			return;
		}

		// パス省略：ファイル選択
		pickFileAsUint8(onSuccess);
	}

	// =========================================================
	// Plugin command hook
	// =========================================================
	var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
	Game_Interpreter.prototype.pluginCommand = function(command, args) {
		_Game_Interpreter_pluginCommand.call(this, command, args);

		if (String(command).toUpperCase() === 'LOAD_CHARA_PNG') {
			// argsはスペース区切りなので、パスにスペースがあると分割される
			// → なので join で戻す（"C:\My Folder\xxx.png" 対応）
			var pathStr = (args && args.length) ? args.join(' ') : '';
			doLoadPng(pathStr);
		}
	};

})();
