/*:
 * @plugindesc 画面キャプチャしてPNG保存（RenderTexture経由/ゲーム解像度固定）+Target埋め込み v1.6.3 (crop 50px each side)
 * @author ChatGPT
 *
 * @help
 * ■使い方（プラグインコマンド）
 *   CAPTURE_PNG
 *   CAPTURE_PNG <保存先パス>
 *
 * ■埋め込み
 * - PNGのiTXtチャンク(keyword="XoChara")にwindow.TargetをJSONで埋め込みます。
 * - 読み込み側は keyword="XoChara" / magic="XOCHARA" を前提にしてください。
 *
 * ■追加機能（v1.6.3）
 * - 保存する画像を「上下左右 50px」ずつ削って保存します（中央部分だけ保存）。
 */

(function() {
	'use strict';

	// =========================================================
	//  決め打ち設定
	// =========================================================
	var FIXED_DIR_NAME  = 'captures';
	var FIXED_FILE_NAME = 'capture.png';

	// ★追加：クロップ量（上下左右）
	var CROP_MARGIN = 50;

	// 埋め込み設定（読み込み側と合わせる）
	var ITXT_KEYWORD = 'XoChara';
	var EMBED_MAGIC  = 'XOCHARA';
	var EMBED_VER    = 1;

	// =========================================================
	//  DataURL -> Buffer
	// =========================================================
	function dataUrlToBuffer(dataUrl) {
		var base64 = dataUrl.replace(/^data:image\/png;base64,/, '');
		return Buffer.from(base64, 'base64');
	}

	// =========================================================
	//  NW.js デフォルト出力パス
	// =========================================================
	function getFixedOutputPathNWjs() {
		var path = require('path');
		var fs = require('fs');

		var gameDir = process.cwd();
		var outDir = path.join(gameDir, FIXED_DIR_NAME);

		if (!fs.existsSync(outDir)) {
			fs.mkdirSync(outDir, { recursive: true });
		}
		return path.join(outDir, FIXED_FILE_NAME);
	}

	// =========================================================
	//  ユーザー指定パスの解決（NW.js）
	// =========================================================
	function resolveOutputPathNWjs(userPath) {
		var path = require('path');
		var fs = require('fs');

		if (!userPath || !userPath.trim()) {
			return getFixedOutputPathNWjs();
		}

		var p = userPath.trim();

		// クォートで囲われててもOKにする
		if ((p[0] === '"' && p[p.length - 1] === '"') || (p[0] === "'" && p[p.length - 1] === "'")) {
			p = p.slice(1, -1);
		}

		// 相対 -> 絶対
		var abs = path.isAbsolute(p) ? p : path.resolve(process.cwd(), p);

		// ディレクトリ指定（末尾が区切り）
		if (/[\\\/]$/.test(abs)) {
			abs = path.join(abs, FIXED_FILE_NAME);
		} else {
			// 拡張子がないなら .png
			if (!path.extname(abs)) {
				abs += '.png';
			}
		}

		// フォルダ作成
		var dir = path.dirname(abs);
		if (!fs.existsSync(dir)) {
			fs.mkdirSync(dir, { recursive: true });
		}

		return abs;
	}

	// =========================================================
	//  ブラウザ用：指定パスからファイル名だけ抽出
	// =========================================================
	function resolveDownloadName(userPath) {
		if (!userPath || !userPath.trim()) return FIXED_FILE_NAME;

		var p = userPath.trim();
		if ((p[0] === '"' && p[p.length - 1] === '"') || (p[0] === "'" && p[p.length - 1] === "'")) {
			p = p.slice(1, -1);
		}

		// 末尾が区切りならデフォ名
		if (/[\\\/]$/.test(p)) return FIXED_FILE_NAME;

		// ファイル名部分だけ
		var name = p.split(/[\\\/]/).pop();
		if (!name) return FIXED_FILE_NAME;

		// 拡張子がなければ .png
		if (!/\.[a-zA-Z0-9]+$/.test(name)) name += '.png';

		return name;
	}

	// =========================================================
	//  ゲーム解像度
	// =========================================================
	function getGameSize() {
		if (window.$dataSystem && $dataSystem.screenWidth && $dataSystem.screenHeight) {
			return { w: $dataSystem.screenWidth | 0, h: $dataSystem.screenHeight | 0 };
		}
		return {
			w: (Graphics.boxWidth || Graphics.width || 816) | 0,
			h: (Graphics.boxHeight || Graphics.height || 624) | 0
		};
	}

	// =========================================================
	//  RenderTexture生成
	// =========================================================
	function createRenderTexture(w, h) {
		if (PIXI.RenderTexture && PIXI.RenderTexture.create) {
			return PIXI.RenderTexture.create(w, h);
		}
		return new PIXI.RenderTexture(Graphics._renderer, w, h);
	}

	// =========================================================
	//  ★追加：canvasを上下左右marginずつクロップして返す
	// =========================================================
	function cropCanvas(srcCanvas, t, b, l, r) {
		if (!srcCanvas) return srcCanvas;

		// 未指定(undefined)の場合は0にする
		t = t | 0;
		b = b | 0;
		l = l | 0;
		r = r | 0;

		var sw = srcCanvas.width | 0;
		var sh = srcCanvas.height | 0;

		// 新しい幅と高さを計算（元のサイズ - 左右 / 元のサイズ - 上下）
		var cw = sw - l - r;
		var ch = sh - t - b;

		// 削りすぎてサイズが0以下になる場合は処理しない
		if (cw <= 0 || ch <= 0) return srcCanvas;

		var c = document.createElement('canvas');
		c.width = cw;
		c.height = ch;

		var ctx = c.getContext('2d');
    
		// 元画像の (l, t) から (cw, ch) の範囲を切り出す
		ctx.drawImage(srcCanvas, l, t, cw, ch, 0, 0, cw, ch);
    
		return c;
	}

	// =========================================================
	//  Scene -> RenderTexture -> PNG DataURL
	// =========================================================
	function captureSceneViaRenderTextureToDataUrl() {
		var renderer = Graphics._renderer;
		var scene = SceneManager._scene;

		if (!renderer) throw new Error('Graphics._renderer が見つかりません');
		if (!scene) throw new Error('SceneManager._scene が見つかりません');

		var extractor = renderer.plugins && renderer.plugins.extract;
		if (!extractor || !extractor.canvas) {
			throw new Error('PIXI Extract が利用できません (renderer.plugins.extract.canvas が見つかりません)');
		}

		var size = getGameSize();
		var outW = size.w, outH = size.h;

		var rt = createRenderTexture(outW, outH);

		// 表示上のscale/positionの影響を避けるため一時退避
		var oldX = scene.x, oldY = scene.y;
		var oldSX = scene.scale.x, oldSY = scene.scale.y;
		var oldRot = scene.rotation;

		try {
			scene.x = 0; scene.y = 0;
			scene.scale.x = 1; scene.scale.y = 1;
			scene.rotation = 0;

			if (renderer.render.length >= 5) {
				renderer.render(scene, rt, true, null, false);
			} else {
				renderer.render(scene, rt, true);
			}
		} finally {
			scene.x = oldX; scene.y = oldY;
			scene.scale.x = oldSX; scene.scale.y = oldSY;
			scene.rotation = oldRot;
		}

		var canvas = extractor.canvas(rt);

		// ★追加：上下左右を一定サイズ削る
		canvas = cropCanvas(canvas, 50, 50, 30, 30);

		return canvas.toDataURL('image/png');
	}

	// =========================================================
	//  window.Target -> JSON（関数/undefinedは除外）
	// =========================================================
	function charaToSafeObject(ch) {
		if (!ch || typeof ch !== 'object') return null;
		var out = {};
		Object.keys(ch).forEach(function(k) {
			var v = ch[k];
			if (typeof v === 'function') return;
			if (typeof v === 'undefined') return;
			out[k] = v;
		});
		return out;
	}

	function buildEmbedJsonString() {
		var payload = {
			magic: EMBED_MAGIC,
			ver: EMBED_VER,
			t: Date.now(),
			chara: charaToSafeObject(window.Target)
		};
		return JSON.stringify(payload);
	}

	// =========================================================
	//  PNG iTXt 追加（非圧縮）
	// =========================================================
	function readU32be(u8, off) {
		return ((u8[off] << 24) | (u8[off+1] << 16) | (u8[off+2] << 8) | (u8[off+3])) >>> 0;
	}
	function u32beBytes(n) {
		return new Uint8Array([(n>>>24)&255, (n>>>16)&255, (n>>>8)&255, n&255]);
	}
	function asciiBytes(s) {
		var a = new Uint8Array(s.length);
		for (var i=0;i<s.length;i++) a[i] = s.charCodeAt(i) & 0x7F;
		return a;
	}
	function latin1Bytes(s) {
		var a = new Uint8Array(s.length);
		for (var i=0;i<s.length;i++) a[i] = s.charCodeAt(i) & 0xFF;
		return a;
	}
	function utf8Bytes(str) {
		if (window.TextEncoder) return new TextEncoder().encode(str);
		// 最低限フォールバック
		var out = [];
		for (var i=0; i<str.length; i++) {
			var c = str.charCodeAt(i);
			if (c < 0x80) out.push(c);
			else if (c < 0x800) out.push(0xC0 | (c >> 6), 0x80 | (c & 0x3F));
			else out.push(0xE0 | (c >> 12), 0x80 | ((c >> 6) & 0x3F), 0x80 | (c & 0x3F));
		}
		return new Uint8Array(out);
	}

	var CRC_TABLE = null;
	function makeCrcTable() {
		var table = new Uint32Array(256);
		for (var n=0; n<256; n++) {
			var c = n;
			for (var k=0; k<8; k++) c = (c & 1) ? (0xEDB88320 ^ (c >>> 1)) : (c >>> 1);
			table[n] = c >>> 0;
		}
		return table;
	}
	function crc32(u8) {
		if (!CRC_TABLE) CRC_TABLE = makeCrcTable();
		var c = 0xFFFFFFFF;
		for (var i=0;i<u8.length;i++) c = CRC_TABLE[(c ^ u8[i]) & 0xFF] ^ (c >>> 8);
		return (c ^ 0xFFFFFFFF) >>> 0;
	}

	function findIendOffset(u8) {
		var off = 8;
		while (off + 12 <= u8.length) {
			var len = readU32be(u8, off);
			var type = String.fromCharCode(u8[off+4], u8[off+5], u8[off+6], u8[off+7]);
			if (type === 'IEND') return off;
			off += 12 + len;
		}
		return -1;
	}

	function addITXtChunk(pngBuf, keyword, utf8Text) {
		var u8 = (pngBuf instanceof Uint8Array) ? pngBuf : new Uint8Array(pngBuf);

		// PNG signature確認
		var sig = [137,80,78,71,13,10,26,10];
		for (var i=0;i<8;i++) if (u8[i] !== sig[i]) throw new Error('PNGではありません');

		var keyBytes = latin1Bytes(keyword);
		var textBytes = utf8Bytes(utf8Text);

		// keyword\0 + flag + method + lang\0 + trans\0 + text
		var data = new Uint8Array(keyBytes.length + 1 + 1 + 1 + 1 + 1 + textBytes.length);
		var p = 0;
		data.set(keyBytes, p); p += keyBytes.length;
		data[p++] = 0;
		data[p++] = 0; // comp flag: 0(非圧縮)
		data[p++] = 0; // comp method
		data[p++] = 0; // language tag empty
		data[p++] = 0; // translated keyword empty
		data.set(textBytes, p); p += textBytes.length;

		var type = asciiBytes('iTXt');
		var lengthBytes = u32beBytes(data.length);

		var crcInput = new Uint8Array(4 + data.length);
		crcInput.set(type, 0);
		crcInput.set(data, 4);
		var crcBytes = u32beBytes(crc32(crcInput));

		var chunk = new Uint8Array(4 + 4 + data.length + 4);
		chunk.set(lengthBytes, 0);
		chunk.set(type, 4);
		chunk.set(data, 8);
		chunk.set(crcBytes, 8 + data.length);

		var insertPos = findIendOffset(u8);
		if (insertPos < 0) throw new Error('IENDが見つかりません');

		var out = new Uint8Array(u8.length + chunk.length);
		out.set(u8.subarray(0, insertPos), 0);
		out.set(chunk, insertPos);
		out.set(u8.subarray(insertPos), insertPos + chunk.length);
		return out;
	}

	// =========================================================
	//  保存（NW.js）※iTXt埋め込みあり
	// =========================================================
	function savePngNWjs(dataUrl, outPath) {
		var fs = require('fs');
		var buf = dataUrlToBuffer(dataUrl);

		var json = buildEmbedJsonString();
		var outU8 = addITXtChunk(buf, ITXT_KEYWORD, json);

		fs.writeFileSync(outPath, Buffer.from(outU8));
		return outPath;
	}

	// =========================================================
	//  保存（ブラウザ：ダウンロード）※iTXt埋め込みあり
	// =========================================================
	function downloadPngBrowser(dataUrl, fileName) {
		var baseBuf = dataUrlToBuffer(dataUrl);

		var json = buildEmbedJsonString();
		var outU8 = addITXtChunk(baseBuf, ITXT_KEYWORD, json);

		var blob = new Blob([outU8], { type: 'image/png' });
		var url = URL.createObjectURL(blob);

		var a = document.createElement('a');
		a.href = url;
		a.download = fileName || FIXED_FILE_NAME;
		document.body.appendChild(a);
		a.click();
		document.body.removeChild(a);

		setTimeout(function(){ URL.revokeObjectURL(url); }, 1000);
	}

	// =========================================================
	//  実行（1フレーム待ってから撮る：黒回避の保険）
	// =========================================================
	function doCaptureAndSave(userPath) {
		// 既存仕様：開始でOFF / 完了でON（メッセージ表示なし）
		$gameSwitches.setValue(71, false);

		requestAnimationFrame(function() {
			try {
				var dataUrl = captureSceneViaRenderTextureToDataUrl();

				if (Utils.isNwjs()) {
					var outPath = resolveOutputPathNWjs(userPath);
					savePngNWjs(dataUrl, outPath);
				} else {
					var name = resolveDownloadName(userPath);
					downloadPngBrowser(dataUrl, name);
				}

				$gameSwitches.setValue(71, true);

			} catch (e) {
				console.error('[CAPTURE_PNG] Failed:', e);
			}
		});
	}

	// =========================================================
	//  プラグインコマンド
	// =========================================================
	var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
	Game_Interpreter.prototype.pluginCommand = function(command, args) {
		_Game_Interpreter_pluginCommand.call(this, command, args);

		if (String(command).toUpperCase() === 'CAPTURE_PNG') {
			var pathStr = (args && args.length) ? args.join(' ') : '';
			doCaptureAndSave(pathStr);
		}
	};

})();
