/*:
 * @target MZ
 * @plugindesc 画像カーソルを選択中項目の左に表示（敵ターゲット選択にも対応） v1.1
 * @author ChatGPT
 *
 * @help
 * v1.1 変更点：敵/味方のターゲット選択ウィンドウ（Window_BattleEnemy / Window_BattleActor）
 * にも画像カーソルを表示できるようにしました。
 *
 * 標準の半透明カーソル矩形は非表示にし、指定した画像を
 * 選択中項目の「左側」に表示します。
 *
 * 【設置】
 *  1) 本プラグインを有効化
 *  2) 画像を img/system または img/pictures に入れる
 *  3) パラメータで画像名などを設定
 *
 * @param CursorImage
 * @text カーソル画像名
 * @type file
 * @dir img/system
 * @require 1
 * @desc 使用するカーソル画像（img/system）。img/picturesを使いたい場合は下のフォルダ設定を変更。
 * @default
 *
 * @param ImageFolder
 * @text 画像フォルダ
 * @type select
 * @option img/system @value system
 * @option img/pictures @value picture
 * @default system
 *
 * @param OffsetX
 * @text Xオフセット
 * @type number
 * @min -999
 * @max 999
 * @default -8
 *
 * @param OffsetY
 * @text Yオフセット
 * @type number
 * @min -999
 * @max 999
 * @default 0
 *
 * @param Scale
 * @text スケール(%) 
 * @type number
 * @min 10
 * @max 400
 * @default 100
 *
 * @param ApplyScope
 * @text 適用範囲
 * @type select
 * @option 戦闘コマンドのみ（アクター/パーティ） @value battleCommands
 * @option 戦闘コマンド＋ターゲット選択（敵/味方） @value battleAll
 * @option すべての選択ウィンドウ @value all
 * @default battleAll
 */
(() => {
  "use strict";

  const paramsId = document.currentScript.src.match(/([^/\\]+)\.js$/)[1];
  const P = PluginManager.parameters(paramsId);
  const cursorName = String(P["CursorImage"] || "");
  const folder = String(P["ImageFolder"] || "system");
  const offsetX = Number(P["OffsetX"] || -8);
  const offsetY = Number(P["OffsetY"] || 0);
  const scalePct = Math.max(10, Number(P["Scale"] || 100));
  const scope = String(P["ApplyScope"] || "battleAll");

  function loadCursorBitmap(){
    if (!cursorName) return null;
    return folder === "picture" ? ImageManager.loadPicture(cursorName) : ImageManager.loadSystem(cursorName);
  }

  function isTargetWindow(win){
    if (scope === "all") return win instanceof Window_Selectable;
    if (scope === "battleAll") {
      return (win instanceof Window_ActorCommand) ||
             (win instanceof Window_PartyCommand) ||
             (win instanceof Window_BattleEnemy) ||
             (win instanceof Window_BattleActor);
    }
    // battleCommands
    return (win instanceof Window_ActorCommand) || (win instanceof Window_PartyCommand);
  }

  // 拡張：Window_Selectable に画像カーソル
  const _WS_initialize = Window_Selectable.prototype.initialize;
  Window_Selectable.prototype.initialize = function(rect){
    _WS_initialize.call(this, rect);
    this._useImageCursor = isTargetWindow(this);
    if (this._useImageCursor){
      this._imageCursorSprite = new Sprite();
      const bmp = loadCursorBitmap();
      if (bmp) this._imageCursorSprite.bitmap = bmp;
      this._imageCursorSprite.visible = false;
      this.addChild(this._imageCursorSprite);
      this.cursorOpacity = 0; // 既定の矩形カーソルは消す
    }
  };

  const _WS_update = Window_Selectable.prototype.update;
  Window_Selectable.prototype.update = function(){
    _WS_update.call(this);
    if (this._useImageCursor) this.updateImageCursor();
  };

  Window_Selectable.prototype.updateImageCursor = function(){
    const spr = this._imageCursorSprite;
    if (!spr){ return; }
    const active = this.active && this.openness > 0;
    if (!active){ spr.visible = false; return; }

    const index = this.index();
    if (index < 0){ spr.visible = false; return; }

    const r = this.itemRect(index);
    const bmp = spr.bitmap;
    const w = bmp ? bmp.width  * (scalePct/100) : 16 * (scalePct/100);
    const h = bmp ? bmp.height * (scalePct/100) : 16 * (scalePct/100);
    spr.scale.x = spr.scale.y = scalePct/100;
    spr.x = r.x + offsetX;
    spr.y = r.y + offsetY + Math.floor((r.height - h) / 2);
    spr.visible = true;
  };

  // 表示/非表示時の同期
  const _WS_hide = Window_Selectable.prototype.hide;
  Window_Selectable.prototype.hide = function(){
    _WS_hide.call(this);
    if (this._imageCursorSprite) this._imageCursorSprite.visible = false;
  };
  const _WS_deactivate = Window_Selectable.prototype.deactivate;
  Window_Selectable.prototype.deactivate = function(){
    _WS_deactivate.call(this);
    if (this._imageCursorSprite) this._imageCursorSprite.visible = false;
  };

})();