/*:
 * @target MZ
 * @plugindesc v1.3.7 KT_ElementDamageToast: 攻撃力/防御倍率/属性(基本ID)＋難易度補正でHPダメージ（最低HP保証）＋中央付近ランダムトースト（通常ウィンドウ見た目）＋イベント用うめき声メッセージ
 * @author ChatGPT
 *
 * @help
 * ■「3回目ほどからダメージ0」について
 * 既定設定では HP最低値(minHp)=1 のため、HPが1に到達するとそれ以上は減りません。
 * 以前の版は「実際に減った量」を表示していたため、HPが1の時は 0 と表示されました。
 * v1.3.2 では、表示用 {damage} を「計算結果のダメージ」に変更し、0表示になりにくくしています。
 * 実際に減った量を表示したい場合は {appliedDamage} を使ってください。
 *
 * ■概要
 * イベント等から「攻撃力(数値)」「アクター防御力(倍率適用)」「属性(エレメント)倍率」を適用して
 * 指定アクターにHPダメージを与えます。メッセージはトーストで表示します。
 *
 * さらに、難易度ID（既定：変数35番）に応じて以下を自動補正できます。
 * - ダメージ倍率
 * - 防御倍率（アクターDEFに掛かる倍率）
 *
 * ■イベント用 追加機能
 * - マップイベントからプラグインコマンドを実行した時、ダメージトースト表示の後に
 *   顔画像付きメッセージ（うめき声）を自動表示できます。
 * - うめき声はプラグインパラメータで複数指定し、ランダムで1つ選ばれます。
 *
 * ■見た目
 * - トーストウィンドウを「通常のウィンドウ」と同じ見た目（枠＋背景）で表示します。
 *
 * ■計算式（2段）
 * 1) 前段式 formulaPre : x, atk, tech を使って a を作る
 * 2) 防御適用           : y = max(0, a - defEff)
 *    defEff = def * (defRate/100) * (difficultyDefRate/100)
 * 3) 属性倍率           : y2 = y * rate
 * 4) 後段式 formulaPost : a, def, defEff, y, rate, atk, tech を使って B を作る（既定: floor(y*rate)）
 * 5) 難易度ダメージ倍率 : damage = floor(max(0,B) * (difficultyDamageRate/100))
 *
 * ■メッセージの置換（トースト）
 * {target} {damage} {appliedDamage} {atk} {atkBase} {def} {defEff} {defRate} {rate} {x} {a} {element} {elementId} {tech}
 * {diffId} {diffAtkRate} {diffDmgRate} {diffDefRate}
 *
 * - {damage}        : 計算結果のダメージ（v1.3.2からこれ）
 * - {appliedDamage} : 実際にHPが減った量（最低HP保証で減らない時は0）

 * - 最低ダメージ(補正) を 1 などにすると、y>0 かつ rate>0 でも丸めで 0 になる場合に最低値を適用できます。
 *
 *
 * ■プラグインパラメータ詳細（長文はここに記載）
 * 1) 基本属性ID
 *    - applyDamage の elementId が 0 のときに採用する属性IDです。
 *    - 0 の場合は属性倍率を 1(等倍)として扱います。
 *
 * 2) 難易度変数ID / 難易度設定リスト
 *    - 難易度IDは、指定変数(difficultyVarId)の値を使います（0なら常に等倍）。
 *    - difficultySettings の id が一致する要素を探し、見つからない場合は atkRate=100 / damageRate=100 / defRate=100 です。
 *    - atkRate は「攻撃力(atk)」に掛けます（前段式の入力に反映）。
 *    - damageRate は「計算後のダメージ」に掛けます。
 *    - defRate は「有効防御力(defEff)」に掛けます（値が大きいほどダメージが減ります）。
 *
 * 3) トースト表示
 *    - duration/fadeIn/fadeOut はフレーム単位です（60フレーム=約1秒）。
 *    - maxWidth=0 の場合、文章に合わせて自動調整します。
 *
 * 4) イベント後メッセージ（マップのみ）
 *    - イベントからプラグインコマンドを実行したとき、ダメージトーストの後に
 *      「文章の表示」でうめき声を自動表示できます（delay はフレーム）。
 *    - 戦闘中は対象外です（Scene_Map のみ）。
 *
 * ■プラグインコマンド applyDamage（計算の流れ）
 * 0) x = 乱数(min～max)
 * 1) atk = attackerAtk * (難易度atkRate/100)
 * 2) a = 前段式(formulaPre) を評価（既定: floor((x+atk)*(tech/100))）
 * 3) defEff = DEF * (defRate/100) * (難易度defRate/100)
 *    y = max(0, a - defEff)
 * 3) rate = actor.elementRate(elementId)（属性ID<=0なら 1）
 * 4) Braw = 後段式(formulaPost) を評価（既定: floor(y*rate)）
 * 5) calcDamage = floor(max(0,Braw) * (難易度damageRate/100))
 * 6) afterHp = max(minHp, beforeHp - calcDamage)（beforeHp<=0 は変更しません）
 *
 * ■メッセージ置換
 * {target} {damage}(計算結果) {appliedDamage}(実際に減った量) {atk} {def} {defEff}
 * {defRate} {rate} {x} {a} {element} {elementId} {tech} {diffId} {diffDmgRate} {diffDefRate}

 * @param baseElementId
 * @text 基本属性ID
 * @desc 既定で使う属性ID（applyDamageのelementId=0時）
 * @type number
 * @min 1
 * @default 10
 *
 * @param difficultyVarId
 * @text 難易度ID変数番号
 * @desc 難易度IDが入っているゲーム変数ID（0で無効）
 * @type variable
 * @default 35
 *
 * @param difficultySettings
 * @text 難易度設定
 * @desc 難易度IDごとの倍率補正リスト（詳細はヘルプ）
 * @type struct<DifficultySetting>[]
 * @default []
 *
 * @param eventAfterMessageEnabled
 * @text イベント後メッセージ有効
 * @desc マップで実行時、うめき声メッセージも表示する
 * @type boolean
 * @default true
 *
 * @param eventAfterMessageDelay
 * @text メッセージ表示遅延(フレーム)
 * @desc うめき声表示の遅延(フレーム)
 * @type number
 * @min 0
 * @default 12
 *
 * @param eventFaceName
 * @text 顔画像ファイル
 * @desc うめき声メッセージの顔グラ名（img/faces）
 * @type file
 * @dir img/faces
 * @default
 *
 * @param eventFaceIndex
 * @text 顔画像インデックス
 * @desc 顔グラ番号(0-7)
 * @type number
 * @min 0
 * @max 7
 * @default 0
 *
 * @param eventGroans
 * @text うめき声リスト
 * @desc うめき声候補リスト（ランダム）
 * @type string[]
 * @default ["うぐぅ！","あうっ！？","ううう・・・。","うぎぃぃ・・・！！！"]
 *
 * @param toastMaxWidth
 * @text トースト最大幅
 * @desc トースト最大幅(px)。0で自動
 * @type number
 * @min 200
 * @default 520
 *
 * @param toastPadding
 * @text トースト余白
 * @desc トースト余白(px)
 * @type number
 * @min 0
 * @default 12
 *
 * @param toastFontSize
 * @text トースト文字サイズ
 * @desc トースト文字サイズ
 * @type number
 * @min 10
 * @default 22
 *
 * @param toastDuration
 * @text トースト表示時間(フレーム)
 * @desc 表示時間(フレーム)
 * @type number
 * @min 30
 * @default 150
 *
 * @param toastFadeIn
 * @text フェードイン(フレーム)
 * @desc フェードイン時間(フレーム)
 * @type number
 * @min 0
 * @default 10
 *
 * @param toastFadeOut
 * @text フェードアウト(フレーム)
 * @desc フェードアウト時間(フレーム)
 * @type number
 * @min 0
 * @default 16
 *
 * @param defaultMessage
 * @text 既定メッセージ
 * @desc message未指定時の既定文（置換可）
 * @type string
 * @default {target}は{damage}のダメージを受けた！（{element}）

 *
 * @param minDamageIfPositive
 * @text 最低ダメージ(補正)
 * @desc y>0/rate>0でも0になる時の最低値。0で無効
 * @type number
 * @min 0
 * @default 0
 *
 * @command applyDamage
 * @text ダメージ適用（トースト）
 * @desc HPダメージ＋トースト表示。0/空欄引数はプラグイン設定の既定値
 *
 * @arg targetActorId
 * @text 対象アクターID
 * @desc ダメージ対象のアクターID
 * @type number
 * @min 1
 * @default 1
 *
 * @arg attackerAtk
 * @text 攻撃力(atk)
 * @desc 攻撃力(数値)（式の atk に入る）
 * @type number
 * @min 0
 * @default 20
 *
 * @arg techPower
 * @text 技倍率(tech %)
 * @desc 技威力(%)（式の tech に入る）
 * @type number
 * @min 0
 * @default 100
 *
 * @arg elementId
 * @text 属性ID（0=基本属性）
 * @desc 属性ID。0=既定（プラグイン設定: 基本属性ID）
 * @type number
 * @min 0
 * @default 0
 *
 * @arg defRate
 * @text 防御倍率(defRate %)
 * @desc 対象DEFへ掛ける倍率(%)（難易度補正は別途）
 * @type number
 * @min 0
 * @default 100
 *
 * @arg min
 * @text 乱数min
 * @desc 基礎乱数の最小値（x）
 * @type number
 * @default 1
 *
 * @arg max
 * @text 乱数max
 * @desc 基礎乱数の最大値（x）
 * @type number
 * @default 10
 *
 * @arg formulaPre
 * @text 前段式(formulaPre)
 * @desc 前段式（x/atk/tech→a）。空欄で既定
 * @type string
 * @default Math.floor((x + atk) * (tech / 100))
 *
 * @arg formulaPost
 * @text 後段式(formulaPost)
 * @desc 後段式（y/rate→Braw）。空欄で既定
 * @type string
 * @default Math.floor(y * rate)
 *
 * @arg minHp
 * @text HP最低値
 * @desc 最低HP（この値以下には減らさない）
 * @type number
 * @min 0
 * @default 1
 *
 * @arg message
 * @text メッセージ（トースト）
 * @desc トースト文。空欄=既定（プラグイン設定: 既定メッセージ）
 * @type string
 * @default
 *
 * @arg overrideDuration
 * @text 表示時間上書き
 * @desc 表示時間(フレーム)。0=既定（プラグイン設定: トースト表示時間）
 * @type number
 * @min 0
 * @default 0
 */

/*~struct~DifficultySetting:
 * @param id
 * @text 難易度ID
 * @desc 適用する難易度ID（difficultyVarIdの値と一致）
 * @type number
 * @min 0
 * @default 0
 *
 * @param atkRate
 * @text 攻撃力倍率(%)
 * @desc 難易度攻撃力倍率(%)：atk（前段式入力）に掛ける
 * @type number
 * @min 0
 * @default 100
 *
 * @param damageRate
 * @text ダメージ倍率(%)
 * @desc 難易度ダメージ倍率(%)：最終ダメージに掛ける
 * @type number
 * @min 0
 * @default 100
 *
 * @param defRate
 * @text 防御倍率(%)
 * @desc 難易度防御倍率(%)：有効防御力(defEff)に掛ける
 * @type number
 * @min 0
 * @default 100
 */

(() => {
  "use strict";

  const PLUGIN_NAME = (() => {
    const cs = document.currentScript;
    if (cs && cs.src) {
      const m = cs.src.match(/([^\/\\]+)\.js$/i);
      if (m) return m[1];
    }
    return "KT_ElementDamageToast";
  })();

  const params = PluginManager.parameters(PLUGIN_NAME);

  const toBool = v => String(v).toLowerCase() === "true";

  function parseStructArray(paramValue) {
    if (!paramValue) return [];
    try {
      const arr = JSON.parse(paramValue);
      if (!Array.isArray(arr)) return [];
      return arr.map(s => {
        try { return JSON.parse(s); } catch { return null; }
      }).filter(Boolean);
    } catch {
      return [];
    }
  }

  function parseStringArray(paramValue, fallbackArr) {
    if (!paramValue) return fallbackArr;
    try {
      const a = JSON.parse(paramValue);
      if (!Array.isArray(a)) return fallbackArr;
      return a.map(x => String(x));
    } catch {
      return fallbackArr;
    }
  }

  function pickRandom(arr, fallback) {
    const a = Array.isArray(arr) ? arr.filter(s => String(s).length > 0) : [];
    if (!a.length) return fallback;
    return a[Math.floor(Math.random() * a.length)];
  }

  const difficultySettingsRaw = parseStructArray(params.difficultySettings);

  const P = {
    baseElementId: Math.max(1, Number(params.baseElementId || 10)),
    difficultyVarId: Number(params.difficultyVarId || 35),
    difficultySettings: difficultySettingsRaw.map(d => ({
      id: Number(d.id || 0),
      atkRate: Math.max(0, Number((d.atkRate ?? 100))),
      damageRate: Math.max(0, Number(d.damageRate || 100)),
      defRate: Math.max(0, Number(d.defRate || 100)),
    })),

    eventAfterMessageEnabled: toBool(params.eventAfterMessageEnabled ?? "true"),
    eventAfterMessageDelay: Math.max(0, Number(params.eventAfterMessageDelay || 12)),
    eventFaceName: String(params.eventFaceName || ""),
    eventFaceIndex: Math.max(0, Math.min(7, Number(params.eventFaceIndex || 0))),
    eventGroans: parseStringArray(
      params.eventGroans,
      ["うぐぅ！", "あうっ！？", "ううう・・・。", "うぎぃぃ・・・！！！"]
    ),

    maxWidth: Number(params.toastMaxWidth || 520),
    padding: Number(params.toastPadding || 12),
    fontSize: Number(params.toastFontSize || 22),
    duration: Number(params.toastDuration || 150),
    fadeIn: Number(params.toastFadeIn || 10),
    fadeOut: Number(params.toastFadeOut || 16),
    defaultMessage: String(params.defaultMessage || "{target}は{damage}のダメージを受けた！（{element}）"),
    minDamageIfPositive: Math.max(0, Number(params.minDamageIfPositive || 0)),
  };

  function getDifficultyRates() {
    const diffId = Number($gameVariables.value(P.difficultyVarId) || 0);
    const found = P.difficultySettings.find(d => d.id === diffId);
    const atk = found ? found.atkRate : 100;
    const dmg = found ? found.damageRate : 100;
    const def = found ? found.defRate : 100;
    return {
      diffId,
      diffAtkRate: Math.max(0, Number(atk)),
      diffDmgRate: Math.max(0, Number(dmg)),
      diffDefRate: Math.max(0, Number(def)),
    };
  }

  // ----------------------------
  // Toast Manager (Queue)
  // ----------------------------
  const KTToast = {
    _queue: [],
    _window: null,
    push(text, durationFrames) {
      if (!text) return;
      this._queue.push({ text: String(text), duration: Number(durationFrames) || 0 });
      if (this._window) this._window._ktTryDequeue();
    },
    attachToScene(scene) {
      if (!scene || !scene.addWindow) return;

      if (scene._ktToastWindow) {
        this._window = scene._ktToastWindow;
        this._window._ktTryDequeue();
        return;
      }

      const rect = this._makeRect();
      const w = new Window_KTToast(rect);
      scene._ktToastWindow = w;
      scene.addWindow(w);
      this._window = w;
      w._ktTryDequeue();
    },
    detachIfOwnedByScene(scene) {
      if (scene && scene._ktToastWindow && this._window === scene._ktToastWindow) {
        this._window = null;
      }
    },
    _makeRect() {
      const width = Math.min(P.maxWidth, Graphics.boxWidth);
      return new Rectangle(0, 0, width, 1);
    },
  };

  // ----------------------------
  // Window_KTToast
  // ----------------------------
  class Window_KTToast extends Window_Base {
    initialize(rect) {
      super.initialize(rect);
      this.openness = 255;
      this._phase = "idle";
      this._t = 0;
      this._duration = P.duration;
      this._lines = [];
      this._textRaw = "";
      this._baseW = rect.width;

      this._baseOpacity = 255;
      this._baseBackOpacity = 192;
      this.opacity = this._baseOpacity;
      this.backOpacity = this._baseBackOpacity;

      this.contentsOpacity = 0;
      this.visible = false;
    }

    standardPadding() { return P.padding; }

    resetFontSettings() {
      super.resetFontSettings();
      this.contents.fontSize = P.fontSize;
    }

    lineHeight() {
      return Math.max(36, (this.contents ? this.contents.fontSize : P.fontSize) + 8);
    }

    _wrapTextToLines(text, innerWidth) {
      const lines = [];
      let line = "";
      for (const ch of text) {
        if (ch === "\n") {
          lines.push(line);
          line = "";
          continue;
        }
        const test = line + ch;
        if (this.textWidth(test) > innerWidth && line.length > 0) {
          lines.push(line);
          line = ch;
        } else {
          line = test;
        }
      }
      if (line.length > 0) lines.push(line);
      return lines.length ? lines : [""];
    }

    _randomPosForSize(w, h) {
      const W = Graphics.boxWidth;
      const H = Graphics.boxHeight;

      const xMin = Math.floor(W * 0.15);
      const xMax = Math.floor(W * 0.85 - w);
      const yMin = Math.floor(H * 0.25);
      const yMax = Math.floor(H * 0.55 - h);

      const clamp = (v, lo, hi) => Math.max(lo, Math.min(hi, v));
      const randInt = (lo, hi) => lo + Math.floor(Math.random() * (hi - lo + 1));

      let x = (xMax >= xMin) ? randInt(xMin, xMax) : Math.floor((W - w) / 2);
      let y = (yMax >= yMin) ? randInt(yMin, yMax) : Math.floor((H - h) / 2);

      x = clamp(x, 0, Math.max(0, W - w));
      y = clamp(y, 0, Math.max(0, H - h));
      return { x, y };
    }

    _applyLayoutAndRandomPos() {
      this.resetFontSettings();

      const pad = this.padding * 2;
      const innerW = this._baseW - pad;
      this._lines = this._wrapTextToLines(this._textRaw, innerW);

      const lineH = this.lineHeight();
      const h = pad + lineH * this._lines.length;
      const w = this._baseW;

      const pos = this._randomPosForSize(w, h);
      this.move(pos.x, pos.y, w, h);
      this.createContents();
    }

    _setAlphaAll(a01) {
      const a = Math.max(0, Math.min(1, a01));
      this.contentsOpacity = Math.floor(255 * a);
      this.opacity = Math.floor(this._baseOpacity * a);
      this.backOpacity = Math.floor(this._baseBackOpacity * a);
    }

    showToast(text, durationFrames) {
      this._textRaw = String(text ?? "");
      this._duration = Math.max(1, Number(durationFrames) || P.duration);

      this._applyLayoutAndRandomPos();
      this.refresh();

      this._phase = (P.fadeIn > 0) ? "fadeIn" : "hold";
      this._t = 0;

      if (P.fadeIn > 0) this._setAlphaAll(0);
      else this._setAlphaAll(1);

      this.visible = true;
    }

    refresh() {
      this.contents.clear();
      this.resetFontSettings();
      for (let i = 0; i < this._lines.length; i++) {
        this.drawTextEx(this._lines[i], 0, i * this.lineHeight(), this.innerWidth);
      }
    }

    _ktTryDequeue() {
      if (this._phase !== "idle") return;
      if (!KTToast._queue.length) return;
      const item = KTToast._queue.shift();
      const dur = item.duration > 0 ? item.duration : P.duration;
      this.showToast(item.text, dur);
    }

    update() {
      super.update();
      if (this._phase === "idle") {
        this._ktTryDequeue();
        return;
      }

      this._t++;

      if (this._phase === "fadeIn") {
        const denom = Math.max(1, P.fadeIn);
        const a = Math.min(1, this._t / denom);
        this._setAlphaAll(a);
        if (this._t >= P.fadeIn) {
          this._phase = "hold";
          this._t = 0;
          this._setAlphaAll(1);
        }
        return;
      }

      if (this._phase === "hold") {
        if (this._t >= this._duration) {
          this._phase = (P.fadeOut > 0) ? "fadeOut" : "idle";
          this._t = 0;
          if (P.fadeOut === 0) {
            this._setAlphaAll(0);
            this.visible = false;
            this._ktTryDequeue();
          }
        }
        return;
      }

      if (this._phase === "fadeOut") {
        const denom = Math.max(1, P.fadeOut);
        const a = Math.max(0, 1 - (this._t / denom));
        this._setAlphaAll(a);
        if (this._t >= P.fadeOut) {
          this._phase = "idle";
          this._t = 0;
          this._setAlphaAll(0);
          this.visible = false;
          this._ktTryDequeue();
        }
      }
    }
  }

  // Attach / Rebind toast window on Map & Battle scenes
  const _Scene_Map_createAllWindows = Scene_Map.prototype.createAllWindows;
  Scene_Map.prototype.createAllWindows = function() {
    _Scene_Map_createAllWindows.call(this);
    KTToast.attachToScene(this);
  };
  const _Scene_Map_start = Scene_Map.prototype.start;
  Scene_Map.prototype.start = function() {
    _Scene_Map_start.call(this);
    KTToast.attachToScene(this);
  };

  const _Scene_Battle_createAllWindows = Scene_Battle.prototype.createAllWindows;
  Scene_Battle.prototype.createAllWindows = function() {
    _Scene_Battle_createAllWindows.call(this);
    KTToast.attachToScene(this);
  };
  const _Scene_Battle_start = Scene_Battle.prototype.start;
  Scene_Battle.prototype.start = function() {
    _Scene_Battle_start.call(this);
    KTToast.attachToScene(this);
  };

  const _Scene_Base_terminate = Scene_Base.prototype.terminate;
  Scene_Base.prototype.terminate = function() {
    KTToast.detachIfOwnedByScene(this);
    _Scene_Base_terminate.call(this);
  };

  // ----------------------------
  // Event wait mode for after-message
  // ----------------------------
  const WAIT_MODE = "ktEDT_afterMsg";
  const _Game_Interpreter_updateWaitMode = Game_Interpreter.prototype.updateWaitMode;
  Game_Interpreter.prototype.updateWaitMode = function() {
    if (this._waitMode === WAIT_MODE) {
      const st = this._ktEDT_afterMsg;
      if (!st) {
        this.setWaitMode("");
        return false;
      }
      if (st.delay > 0) {
        st.delay--;
        return true;
      }
      if (!st.shown) {
        $gameMessage.setFaceImage(st.faceName, st.faceIndex);
        $gameMessage.add(st.text);
        st.shown = true;
        return true;
      }
      if ($gameMessage.isBusy()) return true;
      this._ktEDT_afterMsg = null;
      this.setWaitMode("");
      return false;
    }
    return _Game_Interpreter_updateWaitMode.call(this);
  };

  // ----------------------------
  // Helpers
  // ----------------------------
  function randInt(min, max) {
    const a = Math.min(min, max);
    const b = Math.max(min, max);
    return a + Math.floor(Math.random() * (b - a + 1));
  }

  function safeEvalNumber(expr, vars, fallbackValue) {
    try {
      const keys = Object.keys(vars);
      const vals = keys.map(k => vars[k]);
      // eslint-disable-next-line no-new-func
      const fn = new Function(...keys, `"use strict"; return (${expr});`);
      const v = fn(...vals);
      const n = Number(v);
      return Number.isFinite(n) ? n : fallbackValue;
    } catch {
      return fallbackValue;
    }
  }

  function elementNameById(elementId) {
    if (!elementId) return "無属性";
    const arr = $dataSystem && $dataSystem.elements;
    const name = arr && arr[elementId];
    return name ? String(name) : `属性${elementId}`;
  }

  function formatMessage(template, dict) {
    // {key} と ｛key｝（全角波括弧）の両方に対応
    return String(template).replace(/[\{\uff5b](\w+)[\}\uff5d]/g, (_, key) => {
      if (Object.prototype.hasOwnProperty.call(dict, key)) return String(dict[key]);
      return `{${key}}`;
    });
  }

  // ----------------------------
  // Plugin Command
  // ----------------------------
  PluginManager.registerCommand(PLUGIN_NAME, "applyDamage", function(args) {
    const targetActorId = Number(args.targetActorId || 1);
    const attackerAtk = Math.max(0, Number(args.attackerAtk || 0));
    const techPower = Math.max(0, Number(args.techPower || 100));

    const elementIdArg = Number(args.elementId || 0);
    const elementId = elementIdArg > 0 ? elementIdArg : P.baseElementId;

    const defRate = Math.max(0, Number(args.defRate || 100));
    const min = Number(args.min ?? 1);
    const max = Number(args.max ?? 10);
    const formulaPre = String(args.formulaPre || "Math.floor((x + atk) * (tech / 100))");
    const formulaPost = String(args.formulaPost || "Math.floor(y * rate)");
    const minHp = Math.max(0, Number(args.minHp || 1));
    const message = String(args.message || "").trim();
    const overrideDuration = Number(args.overrideDuration || 0);

    const actor = $gameActors.actor(targetActorId);
    if (!actor) return;

    const { diffId, diffAtkRate, diffDmgRate, diffDefRate } = getDifficultyRates();

    const x = randInt(min, max);
    const atkBase = attackerAtk;
    const atk = (atkBase * (diffAtkRate / 100));
    const tech = techPower;

    // 1) 前段
    const a = safeEvalNumber(
      formulaPre,
      { x, atk, atkBase, tech, Math },
      Math.floor((x + atk) * (tech / 100))
    );

    // 2) 防御（倍率 + 難易度防御倍率）
    const def = Number(actor.def || 0);
    const defEff = def * (defRate / 100) * (diffDefRate / 100);
    const y = Math.max(0, a - defEff);

    // 3) 属性倍率
    const rate = elementId > 0 ? Number(actor.elementRate(elementId) || 1) : 1;

    // 4) 後段
    const Braw = safeEvalNumber(
      formulaPost,
      { a, def, defEff, y, rate, atk, tech, Math },
      Math.floor(y * rate)
    );

    // 5) 難易度ダメージ倍率（計算結果）
    const damageRaw = Math.max(0, Number(Braw));
    let calcDamage = Math.floor(damageRaw * (diffDmgRate / 100));

    // y>0 & rate>0 なのに丸めで 0 になるケースを救済（例：属性倍率が極小）
    if (calcDamage === 0 && P.minDamageIfPositive > 0 && diffDmgRate > 0 && y > 0 && rate > 0) {
      calcDamage = P.minDamageIfPositive;
    }

    // HP適用（最低HP保証）
    const beforeHp = actor.hp;
    const afterHp = (beforeHp <= 0)
      ? beforeHp
      : Math.max(minHp, beforeHp - calcDamage);

    const appliedDamage = (beforeHp > 0) ? (beforeHp - afterHp) : 0;
    if (beforeHp > 0) actor.setHp(afterHp);

    // トースト用メッセージ
    const tpl = message.length ? message : P.defaultMessage;
    const eName = elementNameById(elementId);

    // v1.3.2: {damage} は calcDamage（計算結果）にする
    const text = formatMessage(tpl, {
      target: actor.name(),
      damage: calcDamage,
      appliedDamage: appliedDamage,
      atk,
      atkBase,
      def,
      defEff: defEff.toFixed(2),
      defRate,
      rate: rate.toFixed(2),
      x,
      a: Math.floor(a),
      element: eName,
      elementId,
      tech,
      diffId,
      diffAtkRate,
      diffDmgRate,
      diffDefRate
    });

    KTToast.attachToScene(SceneManager._scene);
    const dur = overrideDuration > 0 ? overrideDuration : P.duration;
    KTToast.push(text, dur);

    // マップイベント時：うめき声メッセージを遅延表示（イベント停止）
    const interpreter = this;
    const isMapScene = SceneManager._scene instanceof Scene_Map;
    if (P.eventAfterMessageEnabled && isMapScene && interpreter instanceof Game_Interpreter) {
      const groan = pickRandom(P.eventGroans, "うぐぅ！");
      interpreter._ktEDT_afterMsg = {
        delay: P.eventAfterMessageDelay,
        shown: false,
        faceName: P.eventFaceName,
        faceIndex: P.eventFaceIndex,
        text: groan
      };
      interpreter.setWaitMode(WAIT_MODE);
    }
  });
})();
