/*:
 * @target MZ
 * @plugindesc BigDigitsCapMZ 用アドオン：全画面の数値をカンマ区切りに（HP/MP/TP/所持金以外にも適用）
 * @author ChatGPT
 * @base BigDigitsCapMZ
 * @orderAfter BigDigitsCapMZ
 * 
 * @help BigDigitsCapMZ_CommaAll.js
 * 
 * 本プラグインは BigDigitsCapMZ の「カンマ表示」を全UIに拡張するアドオンです。
 * Bitmap.drawText / measureTextWidth をフックし、表示される文字列中の数字を
 * 自動で 3 桁区切りに整形します。内部の数値は変更しません。
 * 
 * ■主な特徴
 *  - BigDigitsCapMZ の UseThousandsSeparator が true の時のみ動作（連携）。
 *    ※このパラメータを OFF にすると、本アドオンも自動で無効になります。
 *  - 4桁以上の整数、および小数（整数部）に対応。負数もOK。
 *  - 右寄せの幅計算もカンマ後のテキストで行うため、桁区切りで位置がズレません。
 * 
 * @param minDigits
 * @text カンマ付与の最小桁数
 * @type number
 * @min 1
 * @max 15
 * @default 4
 * @desc ここで指定した桁数以上の整数にカンマを付けます。
 * 
 * @param applyInMessages
 * @text メッセージウィンドウにも適用
 * @type boolean
 * @default true
 * @desc 「文章の表示」などメッセージウィンドウに対してもカンマを付けます。
 * 
 */
(() => {
  "use strict";

  const pluginName = document.currentScript && document.currentScript.src
    ? (document.currentScript.src.match(/([^\/]+)\.js$/) || [])[1]
    : "BigDigitsCapMZ_CommaAll";

  const P = PluginManager.parameters(pluginName);
  const MIN_DIGITS = Math.max(1, Number(P.minDigits || 4));
  const APPLY_MESSAGES = String(P.applyInMessages || "true") === "true";

  // BigDigitsCapMZ のパラメータを参照（UseThousandsSeparator が true なら有効化）
  const getBDCParams = () => {
    // BigDigitsCapMZ のプラグイン名は動的なので、有効な全プラグインから探索
    const list = $plugins || [];
    for (const p of list) {
      if (!p || !p.status) continue;
      const name = p.name || "";
      // 名前に BigDigitsCapMZ を含むものを対象（v19/v18等も考慮）
      if (/BigDigitsCapMZ/.test(name)) {
        // param を直接参照できるケース
        if (p.parameters && Object.prototype.hasOwnProperty.call(p.parameters,"UseThousandsSeparator")) {
          return { enabled: String(p.parameters["UseThousandsSeparator"]||"true")==="true" };
        }
      }
    }
    // 見つからない・参照不可のときは「有効」とみなす（単独でも使えるように）
    return { enabled: true };
  };

  const bdc = getBDCParams();
  const ENABLED = bdc.enabled;

  // 数値パターン（4桁以上の整数、または小数を含む場合は整数部が4桁以上）
  const numberPattern = new RegExp(`-?\\d{${MIN_DIGITS},}(?:\\.\\d+)?`, "g");

  const addCommasInText = (text) => {
    if (text == null) return text;
    const s = String(text);
    return s.replace(numberPattern, (whole) => {
      let str = whole.replace(/,/g, "");
      const neg = str[0] === "-" ? "-" : "";
      if (neg) str = str.slice(1);
      const parts = str.split(".");
      const intPart = parts[0];
      const decPart = parts.length > 1 ? parts[1] : null;
      const grouped = intPart.replace(/\B(?=(\d{3})+(?!\d))/g, ",");
      return neg + grouped + (decPart ? "." + decPart : "");
    });
  };

  if (ENABLED) {
    const _drawText = Bitmap.prototype.drawText;
    Bitmap.prototype.drawText = function(text, x, y, maxWidth, lineHeight, align) {
      if (!APPLY_MESSAGES && SceneManager._scene instanceof Scene_Message) {
        return _drawText.call(this, text, x, y, maxWidth, lineHeight, align);
      }
      const formatted = addCommasInText(text);
      return _drawText.call(this, formatted, x, y, maxWidth, lineHeight, align);
    };

    const _measure = Bitmap.prototype.measureTextWidth;
    Bitmap.prototype.measureTextWidth = function(text) {
      if (!APPLY_MESSAGES && SceneManager._scene instanceof Scene_Message) {
        return _measure.call(this, text);
      }
      const formatted = addCommasInText(text);
      return _measure.call(this, formatted);
    };
  }
})();