/*:
 * @target MZ
 * @plugindesc v1.9 数値上限拡張＋NaN防止＋成長曲線フォールバック/外挿（俳優/敵/所持金/TP/レベル）＆カンマ表示
 * @author ChatGPT
 * 
 * @help BigDigitsCapMZ.js
 * 
 * ■変更点（v1.9）
 *  - 成長曲線が途切れたレベルでは「外挿」して能力値を増加させる機能を追加。
 *    ・外挿なし（最後の値を保持） / 線形外挿（直近の伸びを継続）を選択可能。
 *    ・直近 N レベルの平均伸び（移動平均）で安定化。
 * 
 * ■注意
 *  - 外挿は便宜機能です。最終的にはデータベース側で曲線を定義するのが最も正確です。
 *  - 本プラグインは出来るだけ下（後）に配置してください。
 * 
 * @param LevelMax
 * @text レベル上限
 * @type number
 * @min 1
 * @max 999999999
 * @default 99
 * 
 * @param TpMax
 * @text TP上限
 * @type number
 * @min 1
 * @max 999999
 * @default 100
 * 
 * @param HpDigits
 * @text HP桁数
 * @type number
 * @min 1
 * @max 15
 * @default 6
 * 
 * @param MpDigits
 * @text MP桁数
 * @type number
 * @min 1
 * @max 15
 * @default 4
 * 
 * @param AtkDigits
 * @text 攻撃力(ATK)桁数
 * @type number
 * @min 1
 * @max 15
 * @default 12
 * 
 * @param DefDigits
 * @text 防御力(DEF)桁数
 * @type number
 * @min 1
 * @max 15
 * @default 4
 * 
 * @param MatDigits
 * @text 魔法力(MAT)桁数
 * @type number
 * @min 1
 * @max 15
 * @default 4
 * 
 * @param MdfDigits
 * @text 魔法防御(MDF)桁数
 * @type number
 * @min 1
 * @max 15
 * @default 4
 * 
 * @param AgiDigits
 * @text 敏捷性(AGI)桁数
 * @type number
 * @min 1
 * @max 15
 * @default 4
 * 
 * @param LukDigits
 * @text 運(LUK)桁数
 * @type number
 * @min 1
 * @max 15
 * @default 4
 * 
 * @param GoldDigits
 * @text 所持金(ゴールド)桁数
 * @type number
 * @min 1
 * @max 15
 * @default 8
 * 
 * @param UseThousandsSeparator
 * @text カンマ区切りを有効化（HP/MP/TP/所持金）
 * @type boolean
 * @default true
 * 
 * @param ExtrapolateBeyondCurve
 * @text 成長曲線外を外挿で伸ばす
 * @type boolean
 * @default true
 * 
 * @param ExtrapolateMode
 * @text 外挿モード
 * @type select
 * @option 線形（直近の伸び）
 * @value linear
 * @option なし（最後の値のまま）
 * @value flat
 * @default linear
 * 
 * @param ExtrapolateWindow
 * @text 外挿の移動平均幅（レベル数）
 * @type number
 * @min 1
 * @max 50
 * @default 5
 * 
 * @param ExtrapolateMaxStep
 * @text 外挿1レベルあたり最大増分
 * @type number
 * @min 1
 * @max 999999999
 * @default 999999
 */
(() => {
  "use strict";

  const pluginName = (() => {
    const src = document.currentScript && document.currentScript.src;
    const m = src ? src.match(/([^\/]+)\.js$/) : null;
    return m ? m[1] : "BigDigitsCapMZ";
  })();

  const P = PluginManager.parameters(pluginName);
  const num = (k, d) => Number(P[k] ?? d);
  const bool = (k, d) => String(P[k] ?? d) === "true";
  const str = (k, d) => String(P[k] ?? d);

  // 基本設定
  const LEVEL_MAX = num("LevelMax", 99);
  const TP_MAX = num("TpMax", 100);
  const HP_DIG = num("HpDigits", 6);
  const MP_DIG = num("MpDigits", 4);
  const ATK_DIG = num("AtkDigits", 4);
  const DEF_DIG = num("DefDigits", 4);
  const MAT_DIG = num("MatDigits", 4);
  const MDF_DIG = num("MdfDigits", 4);
  const AGI_DIG = num("AgiDigits", 4);
  const LUK_DIG = num("LukDigits", 4);
  const GOLD_DIG = num("GoldDigits", 8);
  const USE_COMMA = bool("UseThousandsSeparator", true);

  // 外挿設定
  const EXT_ON = bool("ExtrapolateBeyondCurve", true);
  const EXT_MODE = str("ExtrapolateMode", "linear"); // linear or flat
  const EXT_WIN = Math.max(1, num("ExtrapolateWindow", 5));
  const EXT_STEP_MAX = Math.max(1, num("ExtrapolateMaxStep", 999999));

  const SAFE_MAX = Number.MAX_SAFE_INTEGER;
  const pow10 = (d) => Math.pow(10, Math.max(1, d));
  const digitsToMax = (d) => Math.min(Math.floor(pow10(d) - 1), SAFE_MAX);
  const safeNum = (v, fallback = 0, min = -SAFE_MAX, max = SAFE_MAX) => {
    if (typeof v !== "number") v = Number(v);
    if (!Number.isFinite(v) || Number.isNaN(v)) return fallback;
    return Math.min(Math.max(v, min), max);
  };

  const CAP = {
    [0]: Math.max(1, digitsToMax(HP_DIG)),
    [1]: digitsToMax(MP_DIG),
    [2]: digitsToMax(ATK_DIG),
    [3]: digitsToMax(DEF_DIG),
    [4]: digitsToMax(MAT_DIG),
    [5]: digitsToMax(MDF_DIG),
    [6]: digitsToMax(AGI_DIG),
    [7]: digitsToMax(LUK_DIG),
    gold: digitsToMax(GOLD_DIG),
    tp: safeNum(TP_MAX, 100, 1, SAFE_MAX)
  };

  // 1) レベル上限
  Game_Actor.prototype.maxLevel = function() {
    return safeNum(LEVEL_MAX, 99, 1, SAFE_MAX);
  };

  // 2) パラメータ上限
  Game_BattlerBase.prototype.paramMax = function(paramId) {
    const cap = CAP[paramId];
    return safeNum(cap ?? SAFE_MAX, SAFE_MAX, paramId === 0 ? 1 : 0, SAFE_MAX);
  };

  // 3) 成長曲線の外挿/フォールバック
  const _paramBase = Game_Actor.prototype.paramBase;
  Game_Actor.prototype.paramBase = function(paramId) {
    const v = _paramBase.call(this, paramId);
    if (Number.isFinite(v)) return paramId === 0 ? Math.max(1, v) : v;

    const arr = this.currentClass().params?.[paramId];
    if (!Array.isArray(arr) || arr.length === 0) {
      return paramId === 0 ? 1 : 0;
    }

    // 最終有効レベルを取得（未使用やundefinedをスキップ）
    let lastLv = 0;
    for (let i = arr.length - 1; i >= 0; i--) {
      if (Number.isFinite(arr[i])) { lastLv = i; break; }
    }
    const lastVal = safeNum(arr[lastLv], paramId === 0 ? 1 : 0, 0, this.paramMax(paramId));
    const lvl = this._level;

    if (!EXT_ON || EXT_MODE === "flat" || lvl <= lastLv) {
      // flat か、定義範囲内ならそのまま/最後の値
      const inside = safeNum(arr[lvl], lastVal, 0, this.paramMax(paramId));
      return paramId === 0 ? Math.max(1, inside) : inside;
    }

    // 線形外挿：直近EXT_WINレベルの平均増分を継続
    let sumDelta = 0;
    let cnt = 0;
    let prev = lastVal;
    for (let i = lastLv; i > 0 && cnt < EXT_WIN; i--) {
      const cur = safeNum(arr[i], prev, 0, this.paramMax(paramId));
      const pre = safeNum(arr[i - 1], cur, 0, this.paramMax(paramId));
      sumDelta += (cur - pre);
      cnt++;
      prev = pre;
    }
    const avgDelta = safeNum(sumDelta / Math.max(1, cnt), 0, -EXT_STEP_MAX, EXT_STEP_MAX);
    const addLv = Math.max(0, lvl - lastLv);
    let ext = lastVal + avgDelta * addLv;
    ext = safeNum(ext, lastVal, paramId === 0 ? 1 : 0, this.paramMax(paramId));
    return ext;
  };

  // 4) 現在値の安全化
  const _param = Game_BattlerBase.prototype.param;
  Game_BattlerBase.prototype.param = function(paramId) {
    const v = _param.call(this, paramId);
    const min = paramId === 0 ? 1 : 0;
    return safeNum(v, min, min, this.paramMax(paramId));
  };

  Game_BattlerBase.prototype.maxTp = function() {
    return CAP.tp;
  };
  Game_Party.prototype.maxGold = function() {
    return CAP.gold;
  };

  const _refresh = Game_BattlerBase.prototype.refresh;
  Game_BattlerBase.prototype.refresh = function() {
    _refresh.call(this);
    this._hp = safeNum(this._hp, 1, 0, this.mhp);
    this._mp = safeNum(this._mp, 0, 0, this.mmp);
    this._tp = safeNum(this._tp, 0, 0, this.maxTp());
  };

  const _levelUp = Game_Actor.prototype.levelUp;
  Game_Actor.prototype.levelUp = function() {
    _levelUp.call(this);
    this._hp = Math.max(1, Math.min(this._hp, this.mhp));
    this._mp = Math.min(this._mp, this.mmp);
    this._tp = Math.min(this._tp, this.maxTp());
  };

  // 5) 表示のカンマ整形
  const formatNum = (v) => {
    const n = safeNum(v, 0, -SAFE_MAX, SAFE_MAX);
    try { return n.toLocaleString(); } catch { return String(n); }
  };
  if (USE_COMMA) {
    const _drawCurrentAndMax = Window_Base.prototype.drawCurrentAndMax;
    Window_Base.prototype.drawCurrentAndMax = function(current, max, x, y, width, color1, color2) {
      const c = typeof current === "number" ? formatNum(current) : current;
      const m = typeof max === "number" ? formatNum(max) : max;
      _drawCurrentAndMax.call(this, c, m, x, y, width, color1, color2);
    };
    const _drawCurrencyValue = Window_Gold.prototype.drawCurrencyValue;
    Window_Gold.prototype.drawCurrencyValue = function(value, unit, x, y, width) {
      const v = typeof value === "number" ? formatNum(value) : value;
      _drawCurrencyValue.call(this, v, unit, x, y, width);
    };
  }
})();