/*:
 * @target MZ
 * @plugindesc v1.1 カスタム状態異常：パラメータ倍率/行動時HP減少/歩数ダメージ/属性クリティカル/移動速度倍率（メモタグ） 
 * @author ChatGPT
 *
 * @help CustomAilmentsMZ.js
 *
 * ■概要
 * ステート（状態異常）のメモ欄にタグを書くことで、以下の拡張効果を付与できます。
 *  - 指定パラメータを%倍率で補正（複数可、重複は乗算）
 *  - 行動直後にHPを%減少（反動）
 *  - N歩ごとにHPを%減少（毒や呪い風）
 *  - 指定属性の攻撃を受けると確定クリティカル
 *  - 【NEW】マップ移動速度を倍率で変更（プレイヤー/隊列メンバー）
 *
 * ■メモ欄タグ
 *   <ParamMul: atk=150, def=120, agi=110>
 *     … 攻撃/防御/敏捷などを%で乗算（100=等倍、150=1.5倍）
 *     … 対象キー：mhp,mmp,atk,def,mat,mdf,agi,luk
 *
 *   <ActionHpLoss: 5%>
 *     … 行動直後に最大HPの5%を減らす（複数ステートは合算）
 *
 *   <StepHpLoss: 1%/50>
 *     … 50歩ごとに最大HPの1%を減らす（複数ステートは歩数別に蓄積され合算）
 *
 *   <CritOnElement: 2>
 *     … 属性ID=2（例：物理）を受けた時は被クリティカル確定
 *
 *   <MoveSpeedMul: 150%>
 *   <MoveSpeedMul: 1.5x>
 *   <MoveSpeedMul: 1.5>
 *     … マップ上での移動速度（距離/フレーム）を乗算します。
 *         例）150% = 1.5倍, 1.5x = 1.5倍, 1.5 = 1.5倍
 *     … 複数ステートが重なると「乗算」されます（例：1.5倍と0.8倍で合計1.2倍）
 *     … 対象：プレイヤー（先頭アクターのステート）、フォロワー（各自のアクターのステート）
 *     … 乗り物中は未適用（元の速度のまま）
 *
 * ■注意
 *  - 重複ステート：各種倍率は乗算、HP減少は合算されます
 *  - ダメージで戦闘不能になり得ます。必要なら別途セーフティを実装してください
 */

(() => {
  'use strict';

  // 文字→数値（失敗時は0）
  const toNum = s => isNaN(Number(s)) ? 0 : Number(s);

  // ステートのメモ解析（キャッシュ）
  function parseStateMeta(state){
    if (!state) return {};
    if (state.__ca_meta) return state.__ca_meta;
    const meta = {};
    const note = state.note || "";

    // ParamMul
    const m1 = note.match(/<\s*ParamMul\s*:\s*([^>]+)>/i);
    if (m1){
      const obj = {}; 
      m1[1].split(",").forEach(pair=>{
        const kv = pair.split("=");
        if (kv.length===2){
          obj[kv[0].trim().toLowerCase()] = toNum(kv[1].trim()) || 100;
        }
      });
      meta.paramMul = obj;
    }

    // ActionHpLoss
    const m2 = note.match(/<\s*ActionHpLoss\s*:\s*([\d.]+)%\s*>/i);
    if (m2) meta.actionHpLoss = Math.max(0, toNum(m2[1]));

    // StepHpLoss
    const m3 = note.match(/<\s*StepHpLoss\s*:\s*([\d.]+)%\s*\/\s*(\d+)\s*>/i);
    if (m3) meta.stepHpLoss = {
      pct: Math.max(0, toNum(m3[1])),
      step: Math.max(1, Number(m3[2]))
    };

    // CritOnElement
    const m4 = note.match(/<\s*CritOnElement\s*:\s*(\d+)\s*>/i);
    if (m4) meta.critOnElement = Number(m4[1]);

    // MoveSpeedMul（150% / 1.5x / 1.5）
    const m5 = note.match(/<\s*MoveSpeedMul\s*:\s*([\d.]+)\s*(%|x)?\s*>/i);
    if (m5) {
      const val = toNum(m5[1]);
      const unit = (m5[2]||"").toLowerCase();
      let factor = 1.0;
      if (unit === "%")      factor = Math.max(0, val/100);
      else /* x or empty */ factor = Math.max(0, val);
      meta.moveSpeedMul = factor || 1.0;
    }

    state.__ca_meta = meta;
    return meta;
  }

  // ---- パラメータ倍率（乗算） ----
  const PARAM_KEYS = ["mhp","mmp","atk","def","mat","mdf","agi","luk"];
  const _paramRate = Game_BattlerBase.prototype.paramRate;
  Game_BattlerBase.prototype.paramRate = function(paramId) {
    let rate = _paramRate.call(this, paramId);
    const key = PARAM_KEYS[paramId] || "";
    if (key){
      const mul = this.states().reduce((acc, st)=>{
        const m = parseStateMeta(st);
        if (m.paramMul && m.paramMul[key] != null){
          return acc * Math.max(1, m.paramMul[key]) / 100;
        }
        return acc;
      }, 1);
      rate *= mul;
    }
    return rate;
  };

  // ---- 行動時HP減少（反動） ----
  const _useItem = Game_Battler.prototype.useItem;
  Game_Battler.prototype.useItem = function(item) {
    _useItem.call(this, item);
    const ratio = this.states().reduce((acc, st)=>{
      const m = parseStateMeta(st);
      return acc + (m.actionHpLoss || 0);
    }, 0);
    if (ratio > 0){
      const dmg = Math.max(1, Math.floor(this.mhp * (ratio/100)));
      this.gainHp(-dmg);
      if (this.isActor()){
        $gameMessage.add(`${this.name()}は反動で ${dmg} ダメージ！`);
      }
    }
  };

  // ---- 歩数ダメージ ----
  const _increaseSteps = Game_Player.prototype.increaseSteps;
  Game_Player.prototype.increaseSteps = function() {
    _increaseSteps.call(this);
    $gameParty.members().forEach(actor=>{
      let totalPct = 0;
      actor.states().forEach(st=>{
        const m = parseStateMeta(st);
        if (m.stepHpLoss){
          st.__ca_stepCounter = (st.__ca_stepCounter||0) + 1;
          if (st.__ca_stepCounter >= m.stepHpLoss.step){
            st.__ca_stepCounter = 0;
            totalPct += m.stepHpLoss.pct;
          }
        }
      });
      if (totalPct > 0){
        const dmg = Math.max(1, Math.floor(actor.mhp * (totalPct/100)));
        actor.gainHp(-dmg);
      }
    });
  };

  // ---- 指定属性で被クリ確定 ----
  const _itemCri = Game_Action.prototype.itemCri;
  Game_Action.prototype.itemCri = function(target){
    let v = _itemCri.call(this, target);
    const item = this.item();
    const eid = item && item.damage ? item.damage.elementId : 0;
    if (eid > 0){
      const hit = target.states().some(st=>{
        const m = parseStateMeta(st);
        return m.critOnElement === eid;
      });
      if (hit) v = 1.0; // クリ確定
    }
    return v;
  };

  // ===== NEW: マップ移動速度の倍率適用 =====
  function actorMoveSpeedMul(actor){
    if (!actor) return 1.0;
    return actor.states().reduce((acc, st)=>{
      const m = parseStateMeta(st);
      if (m.moveSpeedMul != null){
        return acc * Math.max(0, m.moveSpeedMul);
      }
      return acc;
    }, 1.0);
  }

  // 距離/フレームの最終値に乗算するのが安全（MZ内部は 2^realMoveSpeed/256 を使用）
  const _playerDistancePerFrame = Game_Player.prototype.distancePerFrame;
  Game_Player.prototype.distancePerFrame = function(){
    let d = _playerDistancePerFrame.call(this);
    // 乗り物中は未適用
    if (this.isInVehicle && this.isInVehicle()) return d;
    const leader = $gameParty.leader();
    if (leader) d *= actorMoveSpeedMul(leader);
    return d;
  };

  const _followerDistancePerFrame = Game_Follower.prototype.distancePerFrame;
  Game_Follower.prototype.distancePerFrame = function(){
    let d = _followerDistancePerFrame.call(this);
    const actor = this.actor && this.actor();
    if (actor) d *= actorMoveSpeedMul(actor);
    return d;
  };

})();
